package com.flybits.concierge.guidedoptin.presentation.viewmodel

import android.arch.lifecycle.LiveData
import android.arch.lifecycle.MutableLiveData
import android.arch.lifecycle.ViewModel
import android.content.res.Resources
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.concierge.BuildConfig
import com.flybits.concierge.FlybitsConcierge
import com.flybits.concierge.guidedoptin.data.repository.GuidedOptInRepositoryImpl
import com.flybits.concierge.guidedoptin.domain.usecase.GetGuidedOptInUseCase
import com.flybits.concierge.guidedoptin.presentation.mapper.DataToPresentationMapper
import com.flybits.concierge.guidedoptin.presentation.model.*
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.io.InputStream

internal class GuidedOptInViewModel(private val flybitsConcierge: FlybitsConcierge) :
    ViewModel() {
    private val _guidedOptIn = MutableLiveData<GuidedOptIn>()
    val guidedOptIn: LiveData<GuidedOptIn>
        get() = _guidedOptIn

    private val _navigateToLocationOptInFragment = MutableLiveData<Boolean>()
    val navigateToLocationOptInFragment: LiveData<Boolean>
        get() = _navigateToLocationOptInFragment

    private val _navigateToNotificationsOptInFragment = MutableLiveData<Boolean>()
    val navigateToNotificationsOptInFragment: LiveData<Boolean>
        get() = _navigateToNotificationsOptInFragment

    private val _finishGuidedOptInFlow = MutableLiveData<Boolean>()
    val finishGuidedOptInFlow: LiveData<Boolean>
        get() = _finishGuidedOptInFlow

    // Region data fetching
    fun fetchGuidedOptIn(resources: Resources, res: Int, packageName: String) {
        CoroutineScope(Dispatchers.Main).launch {
            val stream = getStream(resources, res)
            if (stream == null) {
                _guidedOptIn.value = GuidedOptIn(
                    BenefitsOptIn(),
                    LocationOptIn(),
                    NotificationsOptIn(),
                    ConciergeOptOut()
                )
            } else {
                val guidedOptIn = GetGuidedOptInUseCase(
                    GuidedOptInRepositoryImpl,
                    stream,
                    BuildConfig.DEBUG
                ).execute()
                _guidedOptIn.value =
                    guidedOptIn?.let { DataToPresentationMapper().map(resources, packageName, it) }
            }
        }
    }

    private suspend fun getStream(resources: Resources, res: Int): InputStream? {
        return withContext(Dispatchers.IO) {
            try {
                resources.openRawResource(res)
            } catch (e: Resources.NotFoundException) {
                null
            }
        }
    }

    /**
     * Returns [BenefitsOptIn] presentation model.
     */
    fun getBenefitsOptIn(): BenefitsOptIn? = guidedOptIn.value?.benefitsOptIn

    /**
     * Returns [LocationOptIn] presentation model.
     */
    fun getLocationOptIn(): LocationOptIn? = guidedOptIn.value?.locationOptIn

    /**
     * Returns [NotificationsOptIn] presentation model.
     */
    fun getNotificationOptIn(): NotificationsOptIn? = guidedOptIn.value?.notificationsOptIn

    /**
     * Returns [ConciergeOptOut] presentation model.
     */
    fun getConciergeOptOut(): ConciergeOptOut? = guidedOptIn.value?.conciergeOptOut

    // Endregion data fetching

    // Region fragments navigation
    /**
     * Handles business logic for continue/next button click event in BenefitsOptInFragment.
     */
    fun onBenefitsContinueClicked() {
        if (guidedOptIn.value?.locationOptIn?.exclude!!) {
            if (guidedOptIn.value?.notificationsOptIn?.exclude!!) {
                finishGuidedOptInFlow()
            } else {
                navigateToNotificationsOptInFragment()
            }
        } else {
            navigateToLocationOptInFragment()
        }
    }

    /**
     * Handles business logic for continue/next button click event in LocationOptInFragment.
     */
    fun onLocationContinueClicked(locationGranted: Boolean) {
        if (locationGranted) {
            moveFromLocationOptInFragment()
        } else {
            flybitsConcierge.permissionsCallback?.requestLocationPermission()
                ?: moveFromLocationOptInFragment()

            //location permission is automatically granted on API 22 and lower, so move to next screen
            if (android.os.Build.VERSION.SDK_INT < android.os.Build.VERSION_CODES.M) {
                moveFromLocationOptInFragment()
            }
        }
    }

    /**
     * Handles business logic for skip button click event in LocationOptInFragment.
     */
    fun onLocationSkipClicked() {
        moveFromLocationOptInFragment()
    }

    private fun moveFromLocationOptInFragment() {
        if (guidedOptIn.value?.notificationsOptIn?.exclude!!) {
            finishGuidedOptInFlow()
        } else {
            navigateToNotificationsOptInFragment()
        }
    }

    /**
     * Handles business logic for continue/next button click event in NotificationsOptInFragment.
     */
    fun onNotificationsContinueClicked() {
        finishGuidedOptInFlow()
    }

    fun optOut() : LiveData<OptOutResult> {
        val liveData = MutableLiveData<OptOutResult>()
        flybitsConcierge.optOut(object : BasicResultCallback {
            override fun onSuccess() {
                liveData.postValue(OptOutResult(true))
            }

            override fun onException(e: FlybitsException) {
                liveData.postValue(OptOutResult(false, e))
            }
        })
        return liveData
    }

    fun onOptOutCancelClicked() {
        finishGuidedOptInFlow()
    }

    private fun navigateToLocationOptInFragment() {
        _navigateToLocationOptInFragment.value = true
    }

    /**
     * Function to be called when navigation to the LocationOptInFragment completes.
     */
    fun navigateToLocationOptInFragmentCompleted() {
        _navigateToLocationOptInFragment.value = false
    }

    private fun navigateToNotificationsOptInFragment() {
        _navigateToNotificationsOptInFragment.value = true
    }

    /**
     * Function to be called when navigation to the NotificationsOptInFragment completes.
     */
    fun navigateToNotificationsOptInFragmentCompleted() {
        _navigateToNotificationsOptInFragment.value = false
    }

    private fun finishGuidedOptInFlow() {
        _finishGuidedOptInFlow.value = true
    }

    /**
     * Function to be called when Guided Opt-In navigation completes.
     */
    fun finishGuidedOptInFlowCompleted() {
        _finishGuidedOptInFlow.value = false
    }
    // Endregion fragments navigation
}