package com.flybits.concierge.analytics

import com.flybits.android.kernel.ContentAnalytics
import com.flybits.commons.library.logging.Logger
import com.flybits.concierge.analytics.ContentViewer.VisibilityEvent
import com.flybits.concierge.models.BaseTemplate
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.launch
import java.util.*
import java.util.concurrent.BlockingQueue
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.LinkedBlockingQueue
import kotlin.concurrent.schedule

const val VIEWING_THRESHOLD = 250L

/**
 * Handles the processing of [VisibilityEvent], and knowing how and when to track viewing analytics
 * for content displayed in the Concierge.
 *
 * @param contentAnalytics [ContentAnalytics] used for tracking analytics.
 */
internal class ContentViewer(private val contentAnalytics: ContentAnalytics) {

    /**
     * Holds the items visible in the view to the user.
     *
     * Wrapper was used instead of passing the list directly since [BaseTemplate] may
     * cease to exist down the line, and the structure of this class may grow in complexity.
     *
     * @param visibleItems All [BaseTemplate] items current visible.
     */
    data class VisibilityEvent(val visibleItems: List<BaseTemplate>)

    private val currentTasks: ConcurrentHashMap<String, TimerTask> = ConcurrentHashMap()
    private val visibilityEventQueue: BlockingQueue<VisibilityEvent> = LinkedBlockingQueue()
    private var running: Job? = null

    /**
     * Retrieve currently queued viewing timer tasks.
     *
     * @return queued [TimerTask] map.
     */
    fun getTasks(): Map<String, TimerTask> = currentTasks

    /**
     * Add to viewing timer task queue.
     *
     * @param task id and [TimerTask] pair.
     */
    fun addTask(task: Pair<String, TimerTask>) {
        currentTasks[task.first] = task.second
    }

    /**
     * Start the processing of queued [VisibilityEvent]s.
     *
     * @param coroutineScope The [CoroutineScope] that [VisibilityEvent]s will be processed on.
     *
     * @return True if processing was started. False if it is already in progress.
     */
    fun start(coroutineScope: CoroutineScope): Boolean {
        return synchronized(this) {
            if (running != null) false  //Already running
            else {
                Logger.setTag("ContentViewer").d("starting!")
                //Launch job that runs through queue
                running = coroutineScope.launch {
                    while (true) {
                        handleVisibilityEvent(visibilityEventQueue.take())
                    }
                }
                true
            }
        }

    }

    /**
     * Stop the processing of queued [VisibilityEvent]s and cancel and all current tasks.
     */
    fun stop() {
        Logger.setTag("ContentViewer").d("stop()")
        running?.cancel()
        running = null
        currentTasks.forEach {
            it.value.cancel()
        }
        currentTasks.clear()
    }

    /**
     * @return Whether the [VisibilityEvent] queue is currently being processed.
     */
    fun isRunning(): Boolean = running != null

    /**
     * Queue [VisibilityEvent] for processing.
     *
     * @param [VisibilityEvent] to be queued.
     */
    fun queueVisibilityEvent(visibilityEvent: VisibilityEvent) {
        visibilityEventQueue.add(visibilityEvent)
    }

    /**
     * Process [VisibilityEvent]. Calling this function multiple times in parallel will lead to bugs
     * and possible errors/exceptions.
     *
     * @param [VisibilityEvent] to be processed.
     */
    fun handleVisibilityEvent(visibilityEvent: VisibilityEvent) {
        val visibleContentList = visibilityEvent.visibleItems.asSequence()
                .map { it.content }
                .filter { it != null }
                .toList()

        //Cancel then remove tasks which disappeared from the screen after scrolling
        currentTasks.keys.asSequence()
                .filter {
                    //Filter out all the content that can be found in the visible list
                    visibleContentList.find { item -> item.id == it } == null
                }.forEach {
                    //Cancel the timer for the view that disappeared(no longer present in visible list)
                    currentTasks[it]?.cancel()
                    currentTasks.remove(it)
                }

        //Schedule the view actions for views which just appeared on the screen after scrolling
        visibleContentList.forEach {
            //This timer will be canceled if this view disappears before the 250ms timer is done
            if (currentTasks[it.id] == null) {
                currentTasks[it.id] = Timer().schedule(VIEWING_THRESHOLD){
                    Logger.setTag("ContentViewer").d("Marking content id ${it.id} as viewed!")
                    contentAnalytics.trackViewed(it, System.currentTimeMillis())
                    //Do not remove the task here because it may still be visible and this will cause a duplicate trackViewed()
                }
            }
        }
    }
}