package com.flybits.concierge.viewmodels

import android.arch.lifecycle.LiveData
import android.arch.lifecycle.MutableLiveData
import android.arch.lifecycle.ViewModel
import android.arch.paging.DataSource
import android.arch.paging.LivePagedListBuilder
import android.arch.paging.PagedList
import com.flybits.android.kernel.models.Content
import com.flybits.android.kernel.models.results.ContentResult
import com.flybits.android.kernel.utilities.ContentParameters
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback
import com.flybits.commons.library.exceptions.APIUsageExceededException
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.exceptions.NotConnectedException
import com.flybits.commons.library.exceptions.UserOptedOutException
import com.flybits.commons.library.logging.Logger
import com.flybits.commons.library.models.internal.Pagination
import com.flybits.commons.library.utils.jbool_expressions.And
import com.flybits.commons.library.utils.jbool_expressions.Variable
import com.flybits.concierge.FlybitsConcierge
import com.flybits.concierge.FlybitsViewProvider
import com.flybits.concierge.R
import com.flybits.concierge.ResourceProvider
import com.flybits.concierge.analytics.ContentViewer
import com.flybits.concierge.models.BaseTemplate
import com.flybits.concierge.models.Category
import com.flybits.concierge.repository.ModelConverter
import com.flybits.concierge.repository.content.BaseTemplateDataSource
import com.flybits.concierge.repository.content.ContentRepository
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import java.util.*

internal class CategoryViewModel(private val category: Category
                        , private val flybitsConcierge: FlybitsConcierge
                        , private val contentRepository: ContentRepository
                        , private val contentViewer: ContentViewer
                        , private val resourceProvider: ResourceProvider
                        , private val modelConverter: ModelConverter) : ViewModel(), PagedResultCallback<Content> {

    interface FeedErrorDisplayer {
        fun onError(err: String)
    }

    companion object {
        const val PAGE_SIZE = 20
        const val PRE_FETCH_DISTANCE = 10
        const val INITIAL_LOAD_SIZE = PAGE_SIZE
    }

    private var contentResult: ContentResult? = null
    private var onRefreshFinishedCallback: (() -> Unit)? = null
    private var feedErrorDisplayer: FeedErrorDisplayer? = null
    private var loadedAll: Boolean = false
    private var contentDataSource: DataSource<Int, BaseTemplate>? = null
    private var gotLocal = false
    private var loadedContent = false
    var isNoData : LiveData<Boolean> = MutableLiveData()
        private set
    private var lastVisibility = false

    val feedContent: LiveData<PagedList<BaseTemplate?>>
        get() {
            loadedAll = false
            val config = PagedList.Config.Builder()
                    .setInitialLoadSizeHint(INITIAL_LOAD_SIZE)
                    .setPageSize(PAGE_SIZE)
                    .setEnablePlaceholders(true)
                    .setPrefetchDistance(PRE_FETCH_DISTANCE)
                    .build()

            val typeList = ArrayList<String>()
            for (viewProvider in flybitsConcierge.flybitsViewProviders ?: emptyList()) {
                typeList.add(viewProvider.getContentType())
            }
            Logger.d("Category: $category, returning LivePagedList")
            //Use name if no labels exist
            val dataSourceFactory = BaseTemplateDataSource.getFactory(
                    category.name, contentRepository, modelConverter) {
                contentDataSource = it
            }

            val livePagedList =  LivePagedListBuilder(dataSourceFactory, config)
                    .setBoundaryCallback(object : PagedList.BoundaryCallback<BaseTemplate?>() {

                override fun onZeroItemsLoaded() {
                    Logger.d("onZeroItemsLoaded()")
                    requestContent()
                }

                override fun onItemAtEndLoaded(itemAtEnd: BaseTemplate) {
                    Logger.d("Category: $category, onItemAtEndLoaded() ")
                    if (!loadedAll) {
                        contentResult?.getMore(this@CategoryViewModel)
                    }
                }
            }).build()

            /* We need to invalidate the datasource here in case content has been loaded remotely but
            *  the data source was not invalidated in the onSuccess() method call because it was waiting
            *  for the local data to be retrieved first. If we do not do this then remote data won't
            *  come back to the UI
            *
            *  The if statement should only execute once, when the feed is first loaded*/
            livePagedList.observeForever {
                Logger.d("Category: $category, got local data, loadedContent = $loadedAll gotLocal = $gotLocal")
                if (loadedContent && !gotLocal){
                    contentDataSource?.invalidate()
                }
                gotLocal = true
            }

            return livePagedList
        }

    init {
        loadedAll = false
        (isNoData as MutableLiveData).postValue(false)
    }

    fun refresh(refreshResultListener: () -> Unit = {}) {
        loadedAll = false
        contentResult = null
        requestContent()
        this.onRefreshFinishedCallback = refreshResultListener
    }

    fun clean() {
        contentViewer.stop()
    }

    private fun requestContent() {
        if (contentResult == null) {
            val paramsBuilder = ContentParameters.Builder()
                    .setPaging(PAGE_SIZE.toLong(), 0)
                    .setCaching(category.name, INITIAL_LOAD_SIZE)

            //Ignore labels for home tab, show all
            if (!category.name.equals(resourceProvider.getString(R.string.flybits_con_default_tab_name), ignoreCase = true)){
                val variables = category.labels.map { Variable.of(it) }
                //Todo: Remove this if else statement once kernel one variable expression starts formatting correctly
                if (variables.size > 1){
                    paramsBuilder.setLabels(And.of(variables))
                } else if (variables.size == 1) {
                    paramsBuilder.setLabels(category.labels.first())
                }
            }

            val params = paramsBuilder.build()

            contentResult = contentRepository.getContent(params,
                flybitsConcierge.viewProviderSupportedContentTypes,this)
        }
    }

    fun setFeedErrorDisplayer(feedErrorDisplayer: FeedErrorDisplayer) {
        this.feedErrorDisplayer = feedErrorDisplayer
    }

    override fun onSuccess(items: ArrayList<Content>, pagination: Pagination) {
        Logger.d("Category: $category, onSuccess() items count: " + items.size)
        onRefreshFinishedCallback?.let{ it() }
        onRefreshFinishedCallback = null

        /*invalidate the database only if the local data has already been returned from the data
        * source otherwise you may interrupt the local data flow which will cause long loading*/
        loadedContent = true
        if (gotLocal){
            Logger.d("Category: $category, gotLocal = true, invalidating content data source")
            contentDataSource?.invalidate()
        }

        (isNoData as MutableLiveData).postValue(items.size == 0)
    }

    override fun onException(exception: FlybitsException) {
        Logger.exception(javaClass.simpleName, exception)

        if (exception is NotConnectedException || exception is APIUsageExceededException) {
            flybitsConcierge.unauthenticateWithoutLogout(exception)
        }

        onRefreshFinishedCallback?.let{ it() }
        onRefreshFinishedCallback = null

        if (exception !is UserOptedOutException) {
            feedErrorDisplayer?.onError(resourceProvider.getString(R.string.flybits_con_generic_error))
        }

        (isNoData as MutableLiveData).postValue(false)
    }

    override fun onLoadedAllItems() {
        loadedAll = true
    }

    //these scroll events need to be queued to prevent a concurrent modification
    fun visibilityChange(visible: Boolean, visibleItems: List<BaseTemplate>? = null) {
        Logger.setTag("Concierge-DEBUG").d("CategoryViewModel line 201: VisibilityChange $visible , $lastVisibility")
        if (visible != lastVisibility) {
            lastVisibility = visible
            if (visible && visibleItems != null) {
                Logger.setTag("Concierge-DEBUG").d("CategoryViewModel line 205: visible && visibleItems")
                contentViewer.start(CoroutineScope(Dispatchers.IO))
                contentViewer.queueVisibilityEvent(ContentViewer.VisibilityEvent(visibleItems))
            } else {
                Logger.setTag("Concierge-DEBUG").d("CategoryViewModel line 209: else")
                contentViewer.stop()
            }
        }
    }

    fun onScroll(visibleItems: List<BaseTemplate>) {
        Logger.setTag("Concierge-DEBUG").d("CategoryViewModel line 216: onScroll ${visibleItems.size}")

        if (contentViewer.isRunning()) {
            contentViewer.queueVisibilityEvent(ContentViewer.VisibilityEvent(visibleItems))
        }
    }
}
