package com.flybits.concierge

import android.content.res.XmlResourceParser
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.concierge.ConciergeConfiguration.Companion.DEFAULT_TIME_TO_UPLOAD_CONTEXT
import com.flybits.concierge.ConciergeConfiguration.Companion.KEY_TIME_TO_UPLOAD_CONTEXT
import org.xmlpull.v1.XmlPullParserException
import java.io.IOException

/**
 *  Parser that extracts configuration fields from the xml file and builds a conciergeConfiguration object with
 *  the fields set
 */
internal object ConciergeConfigurationXMLParser {

    fun generateConfigurationFromXML(xmlResourceParser: XmlResourceParser) : ConciergeConfiguration {

        var projectID: String
        var tncUrl: String
        var privacyPolicyUrl: String
        var timeToUploadContext: Long
        var gatewayUrl: String

        val conciergeTag = "concierge"

        lateinit var conciergeConfiguration : ConciergeConfiguration

        try {

            var eventType = xmlResourceParser.eventType

            while (eventType != XmlResourceParser.END_DOCUMENT) {

                when(eventType) {
                    XmlResourceParser.START_TAG -> {
                        when (xmlResourceParser.name) {

                            conciergeTag -> {
                                // extract projectID, tncUrl, privacyPolicyUrl, and timeToUploadContext
                                projectID = xmlResourceParser.getAttributeValue(null, "projectID")
                                tncUrl = xmlResourceParser.getAttributeValue(null, "tncUrl")
                                privacyPolicyUrl = xmlResourceParser.getAttributeValue(null, "privacyPolicyUrl")
                                timeToUploadContext = xmlResourceParser.getAttributeIntValue(null, KEY_TIME_TO_UPLOAD_CONTEXT, DEFAULT_TIME_TO_UPLOAD_CONTEXT).toLong()
                                gatewayUrl =  xmlResourceParser.getAttributeValue(null, "gatewayUrl")

                                conciergeConfiguration = ConciergeConfiguration.Builder(projectID).setPrivacyPolicyUrl(privacyPolicyUrl)
                                    .setTermsAndServicesRequired(tncUrl).setTimeToUploadContext(timeToUploadContext).setGatewayUrl(gatewayUrl).build()
                            }
                        }

                    }

                    XmlResourceParser.END_TAG -> {
                        when (xmlResourceParser.name) {
                            conciergeTag -> {
                                return conciergeConfiguration
                            }
                        }
                    }

                }

                eventType = xmlResourceParser.next()
            }

            throw FlybitsException("Incorrect Concierge Configuration XML Format : No Concierge tag found")

        } catch (xmlPullParserException : XmlPullParserException) {
            throw IllegalStateException("Concierge: Cfg syntax error.")
        } catch (ioException : IOException) {
            throw IOException("Concierge: Error reading XML resource.")
        }
    }

}