package com.flybits.internal.models.preferences;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;

import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.ObjectResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.exceptions.InvalidJsonFormatException;
import com.flybits.commons.library.exceptions.NotFoundException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.JsonParser;
import com.flybits.commons.library.models.internal.Result;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.util.ArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import okhttp3.MediaType;
import okhttp3.RequestBody;
import okio.Buffer;

import static com.flybits.internal.models.preferences.FlybitsPreferences.PreferencesType.IS_DOUBLE;
import static com.flybits.internal.models.preferences.FlybitsPreferences.PreferencesType.IS_LONG;
import static com.flybits.internal.models.preferences.FlybitsPreferences.PreferencesType.IS_OBJECT;
import static com.flybits.internal.models.preferences.FlybitsPreferences.PreferencesType.IS_OBJECT_LIST;

/**
 * The {@code FlybitsPreference} class is responsible for storing key-pair values on the Flybits
 * server to act as preferences for the user. These preferences are saved on a per user basis
 * regardless of the device.
 */
public class FlybitsPreferences{

    static final String API_USER_PREF = "/kernel/userpref";
    private static final String TAG = "FlyPrefs";
    static MediaType MEDIA_TYPE_PLAIN = MediaType.parse("text/plain; charset=utf-8");
    private Context context;
    private Handler handler;

    /**
     * Default constructor used to construct the {@code FlybitsPreferences} class.
     *
     * @param context The context of the activity initializing the {@code FlybitsPreference}.
     */
    public FlybitsPreferences(Context context){
        this.context    = context;
        this.handler    = new Handler(Looper.getMainLooper());
    }

    /**
     * Constructor used to construct the {@code FlybitsPreferences} class with a custom handler.
     *
     * @param context The context of the activity initializing the {@code FlybitsPreference}.
     * @param handler A custom handler to be used for communicating to the Main Thread.
     */
    public FlybitsPreferences(Context context, Handler handler){
        this.context    = context;
        this.handler    = handler;
    }

    /**
     * Delete the entire list of preferences.
     *
     * @param callback The callback that indicates whether or not all the preferences were
     *                 successfully clear.
     * @return The {@link BasicResult} which contains additional information about the request.
     */
    public BasicResult clear(BasicResultCallback callback) {
        return delete(context, null, callback);
    }

    /**
     * Get the boolean preference associated to the {@code key} parameter. If a value cannot be
     * found then it will default to the {@code defaultValue}.
     *
     * @param key The attribute name that the preference is saved under.
     * @param defaultValue The default value that should be returned in case the {@code key} cannot
     *                     be found.
     * @param callback The callback that is used to indicate whether or not the request was
     *                 successful. If successful, the {@code ObjectResultCallback#onSuccess(Boolean)}
     *                 will return an {@code Boolean}.
     * @return The {@link ObjectResult} which contains additional information about the request.
     */
    public ObjectResult<Boolean> getBoolean(String key, boolean defaultValue, ObjectResultCallback<Boolean> callback){
        return get(context, key, defaultValue, Boolean.class, callback);
    }

    /**
     * Get a preference attribute that is comprised of an {@code ArrayList<Boolean>}. The GET
     * equivalent of {@link #putBooleanList(String, ArrayList, BasicResultCallback)}.
     *
     * @param key The attribute name that the preference is saved under.
     * @param callback The callback that is used to indicate whether or not the request was
     *                 successful. If successful, the {@link ObjectResultCallback#onSuccess(Object)}
     *                 will return an {@code ArrayList<Boolean>}.
     * @return The {@link ObjectResult} which contains additional information about the request.
     */
    public ObjectResult<ArrayList<Boolean>> getBooleanList(String key, ObjectResultCallback<ArrayList<Boolean>> callback) {
        return getList(context, key, callback);
    }

    /**
     * Get the context set for the {@code FlybitsPreferences} class through the constructor.
     *
     * @return The context of the activity that is calling the {@code FlybitsPreferences} object.
     */
    public Context getContext(){
        return context;
    }

    /**
     * Get the double preference associated to the {@code key} parameter. If a value cannot be
     * found then it will default to the {@code defaultValue}.
     *
     * @param key The attribute name that the preference is saved under.
     * @param defaultValue The default value that should be returned in case the {@code key} cannot
     *                     be found.
     * @param callback The callback that is used to indicate whether or not the request was
     *                 successful. If successful, the {@code ObjectResultCallback#onSuccess(Double)}
     *                 will return an {@code Boolean}.
     * @return The {@link ObjectResult} which contains additional information about the request.
     */
    public ObjectResult<Double> getDouble(String key, double defaultValue, ObjectResultCallback<Double> callback){
        return get(context, key, defaultValue, Double.class, callback);
    }

    /**
     * Get a preference attribute that is comprised of an {@code ArrayList<Double>}. The GET
     * equivalent of {@link #putDoubleList(String, ArrayList, BasicResultCallback)}.
     *
     * @param key The attribute name that the preference is saved under.
     * @param callback The callback that is used to indicate whether or not the request was
     *                 successful. If successful, the {@link ObjectResultCallback#onSuccess(Object)}
     *                 will return an {@code ArrayList<Double>}.
     * @return The {@link ObjectResult} which contains additional information about the request.
     */
    public ObjectResult<ArrayList<Double>> getDoubleList(String key, ObjectResultCallback<ArrayList<Double>> callback) {
        return getList(context, key, callback);
    }

    /**
     * Get the {@code Handler} associated to the UI thread.
     *
     * @return The {@code Handler} associated to the UI thread.
     */
    protected Handler getHandler(){
        return handler;
    }

    /**
     * Get the integer preference associated to the {@code key} parameter. If a value cannot be
     * found then it will default to the {@code defaultValue}.
     *
     * @param key The attribute name that the preference is saved under.
     * @param defaultValue The default value that should be returned in case the {@code key} cannot
     *                     be found.
     * @param callback The callback that is used to indicate whether or not the request was
     *                 successful. If successful, the {@code ObjectResultCallback#onSuccess(Integer)}
     *                 will return an {@code Integer}.
     * @return The {@link ObjectResult} which contains additional information about the request.
     */
    public ObjectResult<Integer> getInt(String key, int defaultValue, ObjectResultCallback<Integer> callback){
        return get(context, key, defaultValue, Integer.class, callback);
    }

    /**
     * Get a preference attribute that is comprised of an {@code ArrayList<Integer>}. The GET
     * equivalent of {@link #putIntList(String, ArrayList, BasicResultCallback)}.
     *
     * @param key The attribute name that the preference is saved under.
     * @param callback The callback that is used to indicate whether or not the request was
     *                 successful. If successful, the {@link ObjectResultCallback#onSuccess(Object)}
     *                 will return an {@code ArrayList<Integer>}.
     * @return The {@link ObjectResult} which contains additional information about the request.
     */
    public ObjectResult<ArrayList<Integer>> getIntList(String key, ObjectResultCallback<ArrayList<Integer>> callback) {
        return getList(context, key, callback);
    }

    /**
     * Get the long preference associated to the {@code key} parameter. If a value cannot be
     * found then it will default to the {@code defaultValue}.
     *
     * @param key The attribute name that the preference is saved under.
     * @param defaultValue The default value that should be returned in case the {@code key} cannot
     *                     be found.
     * @param callback The callback that is used to indicate whether or not the request was
     *                 successful. If successful, the {@code ObjectResultCallback#onSuccess(Long)}
     *                 will return an {@code Long}.
     * @return The {@link ObjectResult} which contains additional information about the request.
     */
    public ObjectResult<Long> getLong(String key, long defaultValue, ObjectResultCallback<Long> callback){
        return get(context, key, defaultValue, Long.class, callback);
    }

    /**
     * Get a preference attribute that is comprised of an {@code ArrayList<Long>}. The GET
     * equivalent of {@link #putLongList(String, ArrayList, BasicResultCallback)}.
     *
     * @param key The attribute name that the preference is saved under.
     * @param callback The callback that is used to indicate whether or not the request was
     *                 successful. If successful, the {@link ObjectResultCallback#onSuccess(Object)}
     *                 will return an {@code ArrayList<Long>}.
     * @return The {@link ObjectResult} which contains additional information about the request.
     */
    public ObjectResult<ArrayList<Long>> getLongList(String key, ObjectResultCallback<ArrayList<Long>> callback) {
        return getList(context, key, callback);
    }

    /**
     * Get an Object that was stored within the preferences.
     *
     * @param key The attribute name that the preference is saved under.
     * @param clazz The class that implements the {@code JsonParser} interface, which will be used
     *              for parsing the preference.
     * @param callback The callback that is used to indicate whether or not the request was
     *                 successful. If successful, the {@link ObjectResultCallback#onSuccess(Object)}
     *                 will return an {@code Object} that implements {@code JsonParser}.
     * @param <T> The generic class that must implement {@code JsonParser}.
     * @return The {@link ObjectResult} which contains additional information about the request.
     */
    public <T extends JsonParser> ObjectResult<T> getObject(String key, Class<T> clazz, ObjectResultCallback<T> callback) {
        return getJson(context, key, clazz, callback);
    }

    /**
     * Get a list of Objects that were stored within the preferences.
     *
     * @param key The attribute name that the preference is saved under.
     * @param clazz The class that implements the {@code JsonParser} interface, which will be used
     *              for parsing the preference.
     * @param callback The callback that is used to indicate whether or not the request was
     *                 successful. If successful, the
     *                 {@code ObjectResultCallback#onSuccess(ArrayList<Object>)} will return an
     *                 ArrayList of {@code Object} that implements {@code JsonParser}.
     * @param <T> The generic class that must implement {@code JsonParser}.
     * @return The {@link ObjectResult} which contains additional information about the request.
     */
    public <T extends JsonParser> ObjectResult<ArrayList<T>> getObjectList(String key, Class<T> clazz, ObjectResultCallback<ArrayList<T>> callback) {
        return getJsonList(context, key, clazz, callback);
    }

    /**
     * Get the String preference associated to the {@code key} parameter. If a value cannot be found
     * then it will default to the {@code defaultValue}.
     *
     * @param key The attribute name that the preference is saved under.
     * @param defaultValue The default value that should be returned in case the {@code key} cannot
     *                     be found.
     * @param callback The callback that is used to indicate whether or not the request was
     *                 successful. If successful, the {@code ObjectResultCallback#onSuccess(String)}
     *                 will return an {@code String}.
     * @return The {@link ObjectResult} which contains additional information about the request.
     */
    public ObjectResult<String> getString(String key, String defaultValue, ObjectResultCallback<String> callback){
        return get(context, key, defaultValue, String.class, callback);
    }

    /**
     * Get a preference attribute that is comprised of an {@code ArrayList<String>}. The GET
     * equivalent of {@link #putStringList(String, ArrayList, BasicResultCallback)}.
     *
     * @param key The attribute name that the preference is saved under.
     * @param callback The callback that is used to indicate whether or not the request was
     *                 successful. If successful, the {@link ObjectResultCallback#onSuccess(Object)}
     *                 will return an {@code ArrayList<String>}.
     * @return The {@link ObjectResult} which contains additional information about the request.
     */
    public ObjectResult<ArrayList<String>> getStringList(String key, ObjectResultCallback<ArrayList<String>> callback) {
        return getList(context, key, callback);
    }

    /**
     * Store a Boolean preference in your preferences list.
     *
     * @param key The attribute name that the preference is saved under.
     * @param value The preference to be saved.
     * @param callback The callback that indicates whether or not saving the preference was
     *                 successful.
     * @return The {@link BasicResult} which contains additional information about the request.
     */
    public BasicResult putBoolean(String key, boolean value, BasicResultCallback callback) {
        return update(context, key, RequestBody.create(MEDIA_TYPE_PLAIN, Boolean.toString(value)), callback, PreferencesType.IS_BOOLEAN);
    }

    /**
     * Store a list of Booleans in your preferences list.
     *
     * @param key The attribute name that the preference is saved under.
     * @param values The preference to be saved.
     * @param callback The callback that indicates whether or not saving the preference was
     *                 successful.
     * @return The {@link BasicResult} which contains additional information about the request.
     */
    public BasicResult putBooleanList(String key, ArrayList<Boolean> values, BasicResultCallback callback) {
        return putInternalArrayList(key, values, callback);
    }

    /**
     * Store a Double preference in your preferences list.
     *
     * @param key The attribute name that the preference is saved under.
     * @param value The preference to be saved.
     * @param callback The callback that indicates whether or not saving the preference was
     *                 successful.
     * @return The {@link BasicResult} which contains additional information about the request.
     */
    public BasicResult putDouble(String key, double value, BasicResultCallback callback) {
        return update(context, key, RequestBody.create(MEDIA_TYPE_PLAIN, Double.toString(value)), callback,IS_DOUBLE);
    }

    /**
     * Store a list of Doubles in your preferences list.
     *
     * @param key The attribute name that the preference is saved under.
     * @param values The preference to be saved.
     * @param callback The callback that indicates whether or not saving the preference was
     *                 successful.
     * @return The {@link BasicResult} which contains additional information about the request.
     */
    public BasicResult putDoubleList(String key, ArrayList<Double> values, BasicResultCallback callback) {
        return putInternalArrayList(key, values, callback);
    }

    /**
     * Store a Integer preference in your preferences list.
     *
     * @param key The attribute name that the preference is saved under.
     * @param value The preference to be saved.
     * @param callback The callback that indicates whether or not saving the preference was
     *                 successful.
     * @return The {@link BasicResult} which contains additional information about the request.
     */
    public BasicResult putInt(String key, int value, BasicResultCallback callback) {
        return update(context, key, RequestBody.create(MEDIA_TYPE_PLAIN, Integer.toString(value)), callback,PreferencesType.IS_INTEGER);
    }

    /**
     * Store a list of Integers in your preferences list.
     *
     * @param key The attribute name that the preference is saved under.
     * @param values The preference to be saved.
     * @param callback The callback that indicates whether or not saving the preference was
     *                 successful.
     * @return The {@link BasicResult} which contains additional information about the request.
     */
    public BasicResult putIntList(String key, ArrayList<Integer> values, BasicResultCallback callback) {
        return putInternalArrayList(key, values, callback);
    }

    /**
     * Store a Long preference in your preferences list.
     *
     * @param key      The attribute name that the preference is saved under.
     * @param value    The preference to be saved.
     * @param callback The callback that indicates whether or not saving the preference was
     *                 successful.
     * @return The {@link BasicResult} which contains additional information about the request.
     * @deprecated putLong has been replaced by putLong(String key, long value, BasicResultCallback callback) as it is takes long value as preference.
     * deprecated in version 2.3.2, will be removed in version 3.0.0
     */
    @Deprecated
    public BasicResult putLong(String key, int value, BasicResultCallback callback) {
        return update(context, key, RequestBody.create(MEDIA_TYPE_PLAIN, Integer.toString(value)), callback, IS_LONG);
    }

    /**
     * Store a Long preference in your preferences list.
     *
     * @param key      The attribute name that the preference is saved under.
     * @param value    The preference to be saved in long type.
     * @param callback The callback that indicates whether or not saving the preference was
     *                 successful.
     * @return The {@link BasicResult} which contains additional information about the request.
     */
    public BasicResult putLong(String key, long value, BasicResultCallback callback) {
        return update(context, key, RequestBody.create(MEDIA_TYPE_PLAIN, Long.toString(value)), callback, IS_LONG);
    }

    /**
     * Store a list of Longs in your preferences list.
     *
     * @param key The attribute name that the preference is saved under.
     * @param values The preference to be saved.
     * @param callback The callback that indicates whether or not saving the preference was
     *                 successful.
     * @return The {@link BasicResult} which contains additional information about the request.
     */
    public BasicResult putLongList(String key, ArrayList<Long> values, BasicResultCallback callback) {
        return putInternalArrayList(key, values, callback);
    }

    /**
     * Stores an object that implements the {@link JsonParser} into a preferences.
     *
     * @param key The attribute name that the preference is saved under.
     * @param value The preference to be saved, the object must implement the {@link JsonParser}
     *              interface.
     * @param callback The callback that indicates whether or not saving the preference was
     *                 successful.
     * @return The {@link BasicResult} which contains additional information about the request.
     */
    public BasicResult putObject(String key, JsonParser value, BasicResultCallback callback) {
        RequestBody json = RequestBody.create(MediaType.parse("application/json; charset=utf-8"), value.toJson().toString());
        return update(context, key, json, callback, IS_OBJECT);
    }

    /**
     * Store a list of objects that implement the {@link JsonParser} interface in your preferences
     * list.
     *
     * @param key The attribute name that the preference is saved under.
     * @param values The preference to be saved.
     * @param callback The callback that indicates whether or not saving the preference was
     *                 successful.
     * @return The {@link BasicResult} which contains additional information about the request.
     */
    public BasicResult putObjectList(String key, ArrayList<? extends JsonParser> values, BasicResultCallback callback) {
        JSONArray array = new JSONArray();
        for (JsonParser value : values) {
            array.put(value.toJson());
        }
        RequestBody json = RequestBody.create(MediaType.parse("application/json; charset=utf-8"), array.toString());
        return update(context, key, json, callback, IS_OBJECT_LIST);
    }

    /**
     * Store a String preference in your preferences list.
     * If value is JSONObject then no need for filtering the string.
     *
     * @param key The attribute name that the preference is saved under.
     * @param value The preference to be saved.
     * @param callback The callback that indicates whether or not saving the preference was
     *                 successful.
     * @return The {@link BasicResult} which contains additional information about the request.
     */
    public BasicResult putString(String key, String value, BasicResultCallback callback) {
        //Check if the value is JsonObject
        String valueFilter = value;
        try {
            new JSONObject(value);
        } catch (JSONException e) {
            try {
                new JSONArray(value);
            } catch (JSONException ignored) {
                valueFilter = value.replace("\n", "").replace("\t", "").trim();
            }
        }
        return update(context, key, RequestBody.create(MEDIA_TYPE_PLAIN, valueFilter), callback, PreferencesType.IS_STRING);
    }

    /**
     * Store a list of Strings in your preferences list.
     *
     * @param key The attribute name that the preference is saved under.
     * @param values The preference to be saved.
     * @param callback The callback that indicates whether or not saving the preference was
     *                 successful.
     * @return The {@link BasicResult} which contains additional information about the request.
     */
    public BasicResult putStringList(String key, ArrayList<String> values, BasicResultCallback callback) {
        return putInternalArrayList(key, values, callback);
    }

    /**
     * Removes a specific attribute from the list of preferences that is associated to the
     * {@code key} parameter.
     *
     * @param key The attribute name of the preference to remove.
     * @param callback The callback that indicates whether or not saving the preference was
     *                 successful.
     * @return The {@link BasicResult} which contains additional information about the request.
     */
    public BasicResult remove(String key, BasicResultCallback callback) {
        return delete(context, key, callback);
    }

    protected BasicResult delete(@NonNull final Context context, final String key, BasicResultCallback callback){
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult request = new BasicResult(callback, handler, executorService);
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                String methodIdentifier             = "UserPreference.clear";
                try {
                    String url = API_USER_PREF;
                    if (key != null){
                        url += ("/" + key);
                    }
                    final Result deleteUserPref = FlyAway.delete(context, url, methodIdentifier, null);
                    if (deleteUserPref.getStatus() == RequestStatus.COMPLETED || deleteUserPref.getStatus() == RequestStatus.NOT_FOUND){
                        request.setSuccess();
                    }else {
                        request.setResult(deleteUserPref);
                    }
                }catch (final FlybitsException e){
                    request.setFailed(e);
                }
            }
        });
        return request;
    }

    protected <T> ObjectResult<T> get(@NonNull final Context context, final String key, final T defaultValue, final Class<T> clazz, ObjectResultCallback<T> callback){
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<T> request = new ObjectResult<>(callback, handler, executorService);
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    String url                  = API_USER_PREF + "/" + key;
                    String methodIdentifier     = "UserPreference.get";
                    final Result<T> getUserPrefs = FlyAway.get(context, url, null, methodIdentifier, null);

                    //Check to make sure that there is properties that should be returned
                    if (getUserPrefs.getStatus() == RequestStatus.COMPLETED){

                        //Parse the returned object based on the fromJSON(...) method
                        String response = getUserPrefs.getResultAsString();
                        if (clazz.isAssignableFrom(String.class)) {
                            request.setSuccess((T) response);
                        } else if (clazz.isAssignableFrom(Integer.class)) {
                            request.setSuccess((T) Integer.valueOf(response));
                        } else if (clazz.isAssignableFrom(Boolean.class)) {
                            request.setSuccess((T) Boolean.valueOf(response));
                        } else if (clazz.isAssignableFrom(Double.class)) {
                            request.setSuccess((T) Double.valueOf(response));
                        } else if (clazz.isAssignableFrom(Long.class)) {
                            request.setSuccess((T) Long.valueOf(response));
                        } else {
                            Logger.e("FlybitsPreferences : InvalidJsonFormatException :" + key);
                            request.setFailed(new InvalidJsonFormatException());
                        }

                    }else if(getUserPrefs.getStatus() == RequestStatus.NOT_FOUND){
                        request.setSuccess(defaultValue);
                    }else {
                        request.setResult(getUserPrefs);
                    }

                } catch (final FlybitsException e) {
                    request.setFailed(e);
                } catch (final Exception exception) {
                    request.setFailed(new FlybitsException(exception.getMessage()));
                }
            }
        });
        return request;
    }

    protected <T extends JsonParser> ObjectResult<T> getJson(@NonNull final Context context, final String key, final Class<T> clazz, ObjectResultCallback<T> callback){
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<T> request = new ObjectResult<>(callback, handler, executorService);
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    String url                  = API_USER_PREF + "/" + key;
                    String methodIdentifier     = "UserPreference.get";
                    final Result<T> getUserPrefs = FlyAway.get(context, url, null, methodIdentifier, null);

                    //Check to make sure that there is properties that should be returned
                    if (getUserPrefs.getStatus() == RequestStatus.COMPLETED){

                        //Parse the returned object based on the fromJSON(...) method
                        String response = getUserPrefs.getResultAsString();
                        try{
                            final T newInstance    = clazz.newInstance();
                            newInstance.fromJson(response);
                            request.setSuccess(newInstance);
                        }catch (ClassCastException | InstantiationException | IllegalAccessException ex){
                            //If the response object cannot be
                            request.setFailed(new InvalidJsonFormatException());
                        }
                    }else if(getUserPrefs.getStatus() == RequestStatus.NOT_FOUND){
                        request.setFailed(new NotFoundException());
                    }else {
                        request.setResult(getUserPrefs);
                    }
                }catch (final FlybitsException e){
                    request.setFailed(e);
                }
            }
        });
        return request;
    }

    protected <T extends JsonParser> ObjectResult<ArrayList<T>> getJsonList(@NonNull final Context context, final String key, final Class<T> clazz, ObjectResultCallback<ArrayList<T>> callback){
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<ArrayList<T>> request = new ObjectResult<>(callback, handler, executorService);
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    String url                  = API_USER_PREF + "/" + key;
                    String methodIdentifier     = "UserPreference.get";
                    final Result<ArrayList<T>> getUserPrefs = FlyAway.get(context, url, null, methodIdentifier, null);

                    //Check to make sure that there is properties that should be returned
                    if (getUserPrefs.getStatus() == RequestStatus.COMPLETED){

                        //Parse the returned object based on the fromJSON(...) method
                        String response = getUserPrefs.getResultAsString();
                        try{
                            final ArrayList<T> listOfObjects  = new ArrayList<>();
                            JSONArray array = new JSONArray(response);
                            for (int i = 0; i < array.length(); i++){
                                final T newInstance    = clazz.newInstance();
                                newInstance.fromJson(array.getJSONObject(i).toString());
                                listOfObjects.add(newInstance);
                            }
                            request.setSuccess(listOfObjects);
                        }catch (JSONException | ClassCastException | InstantiationException | IllegalAccessException ex){
                            //If the response object cannot be
                            request.setFailed(new InvalidJsonFormatException());
                        }
                    }else if(getUserPrefs.getStatus() == RequestStatus.NOT_FOUND){
                        request.setSuccess(new ArrayList<T>());
                    }else {
                        request.setResult(getUserPrefs);
                    }
                }catch (final FlybitsException e){
                    request.setFailed(e);
                }
            }
        });
        return request;
    }

    protected <T> ObjectResult<ArrayList<T>> getList(@NonNull final Context context, final String key, ObjectResultCallback<ArrayList<T>> callback){
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<ArrayList<T>> request = new ObjectResult<>(callback, handler, executorService);
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    String url                  = API_USER_PREF + "/" + key;
                    String methodIdentifier     = "UserPreference.get";
                    final Result<ArrayList<T>> getUserPrefs = FlyAway.get(context, url, null, methodIdentifier, null);

                    //Check to make sure that there is properties that should be returned
                    if (getUserPrefs.getStatus() == RequestStatus.COMPLETED){

                        //Parse the returned object based on the fromJSON(...) method
                        String response = getUserPrefs.getResultAsString();
                        try{
                            JSONArray jsonArray = new JSONArray(response);
                            final ArrayList<T> listOfItems    = new ArrayList<>();
                            for (int i=0;i<jsonArray.length();i++){
                                listOfItems.add((T) jsonArray.get(i));
                            }
                            request.setSuccess(listOfItems);
                        }catch (JSONException | ClassCastException ex){
                            //If the response object cannot be
                            request.setFailed(new InvalidJsonFormatException());
                        }
                    }else if(getUserPrefs.getStatus() == RequestStatus.NOT_FOUND){
                        request.setSuccess(new ArrayList<T>());
                    }else {
                        request.setResult(getUserPrefs);
                    }
                }catch (final FlybitsException e){
                    request.setFailed(e);
                }
            }
        });
        return request;
    }

    protected <T> BasicResult putInternalArrayList(String key, ArrayList<T> values, BasicResultCallback callback){
        JSONArray jsonArray = new JSONArray();
        for (T value : values) {
            jsonArray.put(value);
        }
        return update(context, key, jsonArray, callback);
    }

    protected BasicResult update(@NonNull final Context context, final String key, final RequestBody value, BasicResultCallback callback, PreferencesType preferencesType) {
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult request = new BasicResult(callback, handler, executorService);
        executorService.execute(() -> {
            final String methodIdentifier = "UserPreference.update";
            final String url = API_USER_PREF + "/" + key;
            try {
                Result putUserPrefs;
                final Buffer buffer = new Buffer();
                try {
                    value.writeTo(buffer);
                } catch (IOException e) {
                }
                String newString = buffer.readUtf8();
                if (preferencesType == PreferencesType.IS_STRING) {
                    try {
                        new JSONObject(newString);
                        putUserPrefs = FlyAway.put(context, url, newString, null, methodIdentifier, null);
                    } catch (JSONException e) {
                        try {
                            new JSONArray(newString);
                            putUserPrefs = FlyAway.put(context, url, newString, null, methodIdentifier, null);
                        } catch (JSONException ignored) {
                            putUserPrefs = FlyAway.put(context, url, "\"" + newString + "\"", null, methodIdentifier, null);
                        }
                    }
                } else {
                    putUserPrefs = FlyAway.put(context, url, value, null, methodIdentifier, null);
                }
                if (putUserPrefs.getStatus() == RequestStatus.NOT_FOUND) {
                    Result createUserPrefs;
                    if (preferencesType == PreferencesType.IS_STRING) {
                        try {
                            new JSONObject(newString);
                            createUserPrefs = FlyAway.post(context, url, newString, null, methodIdentifier, null);
                        } catch (JSONException e) {
                            try {
                                new JSONArray(newString);
                                createUserPrefs = FlyAway.post(context, url, newString, null, methodIdentifier, null);
                            } catch (JSONException ignored) {
                                createUserPrefs = FlyAway.post(context, url, "\"" + newString + "\"", null, methodIdentifier, null);
                            }
                        }
                    } else {
                        createUserPrefs = FlyAway.post(context, url, value, null, methodIdentifier, null);
                    }
                    if (createUserPrefs.getStatus() == RequestStatus.NOT_FOUND) {
                        final JSONObject rootObject = new JSONObject();
                        final JSONObject object = new JSONObject();
                        try {
                            switch (preferencesType) {
                                case IS_INTEGER: {
                                    object.put(key, Integer.parseInt(newString));
                                    break;
                                }
                                case IS_BOOLEAN: {
                                    object.put(key, Boolean.parseBoolean(newString));
                                    break;
                                }
                                case IS_DOUBLE: {
                                    object.put(key, Double.parseDouble(newString));
                                    break;
                                }
                                case IS_LONG: {
                                    object.put(key, Long.parseLong(newString));
                                    break;
                                }
                                case IS_OBJECT_LIST: {
                                    object.put(key, new JSONArray(newString));
                                    break;
                                }
                                case IS_OBJECT:
                                case IS_STRING: {
                                    try {
                                        object.put(key, new JSONObject(newString));
                                    } catch (JSONException e) {
                                        try {
                                            object.put(key, new JSONArray(newString));
                                        } catch (JSONException ex) {
                                            object.put(key, newString);
                                        }
                                    }
                                    break;
                                }
                            }
                            rootObject.put("data", object);
                        } catch (JSONException e) {
                            Logger.appendTag(TAG).e("JsonException thrown");
                        }
                        final String body = rootObject.toString();
                        final Result createRootUserPrefs = FlyAway.post(context, API_USER_PREF, body, null, methodIdentifier, null);
                        request.setResult(createRootUserPrefs);
                    } else {
                        request.setResult(createUserPrefs);
                    }
                } else {
                    request.setResult(putUserPrefs);
                }
            } catch (final FlybitsException e) {
                request.setFailed(e);
            }
        });
        return request;
    }

    protected BasicResult update(@NonNull final Context context, final String key, final JSONArray value, BasicResultCallback callback){
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult request = new BasicResult(callback, handler, executorService);
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                request.setResult(update(key, value));
            }
        });
        return request;
    }


    protected Result update(final String key, final JSONArray value){

        String methodIdentifier             = "UserPreference.update";
        String url              = API_USER_PREF + "/" + key;
        try {
            final Result putUserPrefs = FlyAway.put(context, url, value.toString(),null, methodIdentifier, null);
            if (putUserPrefs.getStatus() == RequestStatus.NOT_FOUND){

                final Result createUserPrefs = FlyAway.post(context, url, value.toString(), null, methodIdentifier, null);
                if (createUserPrefs.getStatus() == RequestStatus.NOT_FOUND){

                    JSONObject rootObject   = new JSONObject();
                    try{
                        JSONObject object   = new JSONObject();
                        object.put(key, value);
                        rootObject.put("data", object);
                    }catch (JSONException ex){}

                    String body             = rootObject.toString();
                    final Result createRootUserPrefs = FlyAway.post(context, API_USER_PREF, body, null, methodIdentifier, null);
                    return createRootUserPrefs;
                }else {
                    return createUserPrefs;
                }
            }else {
                return putUserPrefs;
            }
        }catch (final FlybitsException e){
           return new Result(e, "");
        }
    }

    /**
     * PreferencesType will define types of objects to be stored.
     */
    protected enum PreferencesType {
        IS_INTEGER(),
        IS_STRING(),
        IS_BOOLEAN(),
        IS_OBJECT(),
        IS_OBJECT_LIST(),
        IS_LONG(),
        IS_DOUBLE()
    }
}
