package com.flybits.commons.library.models

import android.content.Context
import android.os.Build
import com.flybits.commons.library.SharedElements
import com.flybits.commons.library.SharedElementsFactory
import com.flybits.commons.library.api.FlyAway
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback
import com.flybits.commons.library.exceptions.NoDeviceIDSetException
import com.flybits.commons.library.http.RequestStatus
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.launch
import org.json.JSONObject

/**
 * The [Device] model is responsible for all operation related to the physical device. Every time a
 * user calls the connect(...) function in [FlybitsManager], a device is associated to that session.
 * The reason this is needed is due to the fact that a user can have multiple active devices and the
 * Flybits server needs to understand which device it needs to target for Rule Evaluation and
 * sending push notifications.
 *
 * The following information can be associated to a device:
 *     (i) Push Token from FCM - Allows Flybits Server to send push notification to a specific
 *         device.
 *    (ii) Device-scoped Context Plugins - Context Plugins like Location are assoviated to a
 *         specific device not a user as a user can have multiple devices.
 *
 * @param make The manufacturer of the device ("Samsung", "LG", etc.). Default value is
 *             "undisclosed".
 * @param model The model of the physical device ("Note 5", "Pixel 3", etc.). Default value is
 *              "undisclosed.
 * @param osVersion The os version for the physical device ("Android 10", "Android 9", etc.) Default
 *                  value is "undisclosed".
 * @param name The name of the device. Default value is [make]-[model].
 */
data class Device(
    val make: String = Build.MANUFACTURER,
    val model: String = Build.MODEL,
    val osVersion: String = Build.VERSION.SDK_INT.toString(),
    val name: String = "$make-$model"
) {

    companion object {
        internal const val API_DEVICES = "/sso/devices"

        internal const val UNDISCLOSED = "undisclosed"
        internal const val ENDPOINT_ASSOCIATED = "/associations"
    }

    /**
     * Anonymizes the device information based by removing them from the server.
     *
     * @param context The context of the application.
     * @param callback The callback that is used to find out whether the API request was successful.
     *                 Null is allowed at which point the request is fired and result is ignored.
     *
     * @return The [Job] of the request being made to Flybits.
     */
    fun anonymize(context: Context, callback: BasicResultCallback? = null): Job =
        update(context, UNDISCLOSED, UNDISCLOSED, UNDISCLOSED, UNDISCLOSED, callback)

    /**
     * Update the device information based on the attributes entered in the constructor.
     *
     * @param context The context of the application.
     * @param callback The callback that is used to find out whether the API request was successful.
     *                 Null is allowed at which point the request is fired and result is ignored.
     *
     * @return The [Job] of the request being made to Flybits.
     */
    fun update(context: Context, callback: BasicResultCallback? = null): Job? =
        update(context, make, model, osVersion, name, callback)

    /**
     * Update the device information to add a new identifier to the device. This identifier can be
     * any string that the application believes it needs to be associated to the device.
     *
     * @param context The context of the application.
     * @param associationId The identifier to be associated to the device.
     * @param callback The callback that is used to find out whether the API request was successful.
     *                 Null is allowed at which point the request is fired and result is ignored.
     * @param sharedElements The [SharedElements] class that stores all user-specific information
     * needed by the SDK to function correctly. Default the internal built in [SharedElementsFactory].
     *
     * @return The [Job] of the request being made to Flybits.
     */
    fun addAssociation(
        context: Context,
        associationId: String,
        callback: BasicResultCallback? = null
    ): Job = addAssociation(context, associationId, callback, SharedElementsFactory.get(context))

    /**
     * Update the device information to remove a identifier from the device. This identifier can be
     * any string that the application believes it needs to be associated to the device.
     *
     * @param context The context of the application.
     * @param associationId The identifier to be de-associated from the device.
     * @param callback The callback that is used to find out whether the API request was successful.
     *                 Null is allowed at which point the request is fired and result is ignored.
     * @param sharedElements The [SharedElements] class that stores all user-specific information
     * needed by the SDK to function correctly. Default the internal built in [SharedElementsFactory].
     *
     * @return The [Job] of the request being made to Flybits.
     */
    fun deleteAssociation(
        context: Context,
        associationId: String,
        callback: BasicResultCallback? = null
    ): Job = deleteAssociation(context, associationId, callback, SharedElementsFactory.get(context))

    internal fun addAssociation(
        context: Context,
        associationId: String,
        callback: BasicResultCallback? = null,
        sharedElements: SharedElements
    ): Job {

        return CoroutineScope((Dispatchers.Default)).launch {

            val deviceID = sharedElements.getDeviceID()
            if (deviceID.isEmpty()) {
                CoroutineScope(Dispatchers.Main).launch {
                    callback?.onException(NoDeviceIDSetException())
                }
            } else {
                val json = JSONObject()
                json.put("deviceIdentifier", associationId)

                val addAssociationResult = FlyAway.post<Any>(
                    context, "$API_DEVICES/$deviceID$ENDPOINT_ASSOCIATED",
                    json.toString(), null, "Device.addAssociation", null
                )

                CoroutineScope(Dispatchers.Main).launch {
                    when (addAssociationResult.status) {
                        RequestStatus.NOT_FOUND -> callback?.onException(NoDeviceIDSetException())
                        RequestStatus.COMPLETED -> callback?.onSuccess()
                        else -> callback?.onException(addAssociationResult.exception)
                    }
                }
            }
        }
    }

    internal fun deleteAssociation(
        context: Context, associationId: String, callback: BasicResultCallback? = null,
        sharedElements: SharedElements = SharedElementsFactory.get(context)
    ): Job {

        return CoroutineScope((Dispatchers.Default)).launch {

            val deviceID = sharedElements.getDeviceID()
            if (deviceID.isEmpty()) {
                CoroutineScope(Dispatchers.Main).launch {
                    callback?.onException(NoDeviceIDSetException())
                }
            } else {

                val removeAssociationResult = FlyAway.delete(
                    context,
                    "$API_DEVICES/$deviceID$ENDPOINT_ASSOCIATED",
                    "Device.deleteAssociation",
                    associationId
                )

                CoroutineScope(Dispatchers.Main).launch {
                    when (removeAssociationResult.status) {
                        //If it can't be found it means it's not there so no need to remove
                        RequestStatus.NOT_FOUND, RequestStatus.COMPLETED -> callback?.onSuccess()
                        else -> callback?.onException(removeAssociationResult.exception)
                    }
                }
            }
        }
    }

    private fun update(
        context: Context, make: String, model: String, osVersion: String,
        name: String?, callback: BasicResultCallback?
    ): Job {

        val json = JSONObject()
        json.put("make", make)
        json.put("model", model)
        json.put("osVersion", osVersion)

        if (name != null) {
            json.put("name", name)
        }

        return CoroutineScope((Dispatchers.Default)).launch {

            val updateDeviceResult = FlyAway.put<Any>(
                context, API_DEVICES, json.toString(),
                null, "Device.update", null, null
            )

            CoroutineScope(Dispatchers.Main).launch {
                if (updateDeviceResult.status == RequestStatus.COMPLETED) {
                    callback?.onSuccess()
                } else callback?.onException(updateDeviceResult.exception)
            }
        }
    }
}