package com.flybits.commons.library.api.results;


import android.content.Context;
import android.os.Handler;
import android.support.annotation.NonNull;

import com.flybits.commons.library.api.results.callbacks.ListResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.models.internal.Result;

import java.util.ArrayList;
import java.util.concurrent.ExecutorService;

/**
 * The {@code ListResult} class is used to query the server when the response is excepted to an
 * list of object that are not paged. If the network request yields an error then null if provided
 * in the {@link #getResult()} method.
 *
 * @param <T> The object that is expected to be returned.
 */
public class ListResult<T> extends FlybitsResult {

    private ArrayList<T> object;
    private ListResultCallback<T> callback;
    private Handler handler;

    /**
     * Constructor used to initialize the {@code ListResult} object.
     *  @param context The context of the application.
     * @param callback The {@link ObjectResultCallback} that indicates whether or not the network
     */
    public ListResult(@NonNull Context context, ListResultCallback<T> callback){
        super(context);
        this.callback   = callback;
    }

    /**
     * Constructor used to initialize the {@code BasicResult} object with a Handler object which
     * allows you run on the UI thread.
     *
     * @param context The context of the application.
     * @param callback The {@link ObjectResultCallback} that indicates whether or not the network
     *                 request was successful or not.
     * @param service The {@link ObjectResultCallback} that is used to run the network request
     *                thread.
     * @param handler The handler that allows methods to be run in the UI.
     */
    public ListResult(@NonNull Context context, ListResultCallback<T> callback, @NonNull ExecutorService service, @NonNull Handler handler){
        super(context, service);
        this.callback       = callback;
        this.handler        = handler;
    }

    /**
     * Get the object that is returned as a result of the network request.
     *
     * @return T The object that is generated from the HTTP response.
     */
    public ArrayList<T> getResult(){
        if (getStatus() == QueryStatus.SUCCESS) {
            return object;
        }else{
            return null;
        }
    }
    /**
     * Sets the {@link FlybitsException} that was thrown when request more items to be loaded.
     *
     * @param e The {@link FlybitsException} that caused the network to fail.
     * @return true if the error has been set, false otherwise
     */
    @Override
    public boolean setFailed(final FlybitsException e) {
        if (super.setFailed(e)) {
            if (callback != null){
                if (handler != null){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            callback.onException(e);
                        }
                    });
                }else {
                    callback.onException(e);
                }
            }
            return true;
        }
        return false;
    }

    /**
     * Sets the result of the network request as successful and returns the object associated
     * to the request.
     *
     * @param result The result that was returned when a successful network request was made.
     */
    public void setSuccess(final @NonNull ArrayList<T> result){
        if (setSuccess()) {
            this.object = result;
            if (callback != null){
                if (handler != null){
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            callback.onSuccess(result);
                        }
                    });
                }else {
                    callback.onSuccess(result);
                }
            }
        }
    }

    /**
     * Set the {@link Result} of the network request that contains the object returned from
     * the network request.
     *
     * @param result The {@link Result} of the network request with the result as an object
     *               stored in {@link Result#getResult()}.
     */
    public void setResult(@NonNull Result<ArrayList<T>> result){
        if (result.getStatus() == RequestStatus.COMPLETED){
            setSuccess(result.getResult());
        }else{
            setFailed(result.getException());
        }
    }
}
