package com.flybits.commons.library.api.results;

import android.content.Context;

import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.models.internal.Pagination;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

/**
 * The {@code BasicResult} class is used to query the server when no object is expected to be
 * returned, but rather a simple onSuccess or onException with an exception as the reason for the
 * failure.
 */
abstract class FlybitsResult {

    private ScheduledExecutorService sched;
    private ScheduledFuture<?> scheduledFuture;
    private Context context;
    private ExecutorService service;
    private QueryStatus status;

    /**
     * Default constructor that sets all the default variables.
     *
     * @param context The context of the application
     */
    public FlybitsResult(Context context){
        this.context    = context;
        status          = QueryStatus.INITIALIZED;
        sched           = Executors.newScheduledThreadPool(1);
    }

    /**
     * Default constructor that sets all the default variables.
     *
     * @param context The context of the application
     * @param service The {@link ExecutorService} that is used to run the network request thread.
     */
    public FlybitsResult(Context context, ExecutorService service){
        this(context);
        setService(service);
    }

    /**
     * Cancels the network request to get more items.
     */
    public void cancel(){
        if (status == QueryStatus.PROCESSING){
            status  = QueryStatus.CANCELLED;
            cancelRequest();
        }
        cancelTimeout();
    }

    /**
     * Get the Context of the application used during this request.
     *
     * @return the context of the application.
     */
    public Context getContext(){
        return context;
    }

    /**
     * Get the {@link ExecutorService} that is processing the network request.
     *
     * @return The {@link ExecutorService} if it is still running, null otherwise.
     */
    public ExecutorService getService(){
        return service;
    }

    /**
     * Get the current status of the network request that is used to retrieve more items.
     *
     * @return The {@link QueryStatus} of the network request.
     */
    public QueryStatus getStatus(){
        return status;
    }

    /**
     * Sets the {@link FlybitsException} that was thrown when request more items to be loaded.
     *
     * @param e The {@link FlybitsException} that caused the network to fail.
     * @return true if the error has been set, false otherwise
     */
    protected boolean setFailed(FlybitsException e) {
        if (getStatus() != QueryStatus.CANCELLED && getStatus() != QueryStatus.TIMED_OUT) {
            this.status = QueryStatus.FAILED;
            cancelTimeout();
            cancelRequest();
            return true;
        }
        return false;
    }

    /**
     * Sets the {@link ExecutorService} which is used for making the network request within its own
     * thread.
     *
     * @param service The {@link ExecutorService} within which the network request takes place.
     */
    public void setService(ExecutorService service) {
        this.service    = service;
        this.status     = QueryStatus.PROCESSING;
    }

    /**
     * Sets the result of the network request used to retrieve more items. In all cases, this should
     * include the items retrieved along with the returned {@link Pagination} object.
     *
     * @return true if the request was successful, false otherwise
     */
    protected boolean setSuccess(){
        if (getStatus() != QueryStatus.CANCELLED && getStatus() != QueryStatus.TIMED_OUT) {
            this.status = QueryStatus.SUCCESS;
            cancelTimeout();
            cancelRequest();
            return true;
        }
        return false;
    }

    /**
     * Sets a timeout on the network request to stop the network request if it is taking too long to
     * execute.
     *
     * @param time The time when the {@link ExecutorService} should shutdown in the event that it
     *             has not been finished.
     * @param unit The unit of time used in associated with the {@code time} parameter.
     */
    public void setTimeout(long time, TimeUnit unit){
        scheduledFuture = sched.schedule(r, time, unit);
    }

    private void cancelRequest() {
        if (service != null && !service.isShutdown()){
            service.shutdownNow();
        }
    }

    private void cancelTimeout() {
        if (scheduledFuture != null) {
            scheduledFuture.cancel(true);
        }
    }

    private Runnable r = new Runnable() {
        @Override
        public void run() {
            if (status == QueryStatus.PROCESSING) {
                status = QueryStatus.TIMED_OUT;
                cancelRequest();
                cancelTimeout();
            }
        }
    };
}
