package com.flybits.commons.library.api

import android.content.Context
import com.flybits.commons.library.SharedElements
import com.flybits.commons.library.SharedElementsFactory
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.http.HttpDefaultClass
import com.flybits.commons.library.logging.Logger
import com.flybits.commons.library.models.internal.Result
import java.io.IOException

/**
 * The [FlyJWT] class is responsible for making network requests for refreshing a user's JWT token.
 * The JWT token is an important aspect of a User's session as it allows the logged in user to
 * contact the Flybits server and makes requests based on the user's permission level.
 *
 * @param sharedElements Retrieves saved information associated to the application
 * @param httpClient The HTTP client that is used to perform CRUD options on a specific Flybits
 * endpoint.
 */
class FlyJWT internal constructor(private val sharedElements: SharedElements, private val httpClient : HttpDefaultClass.Builder){

    companion object {
        val ENDPOINT = FlybitsManager.AUTHENTICATION_API + "/refreshToken"

        @Volatile
        internal var savedInstance : FlyJWT? = null

        /**
         * Creates a JWT object that is used to refresh the already authorized user.
         *
         * @param context The current [Context] of the application.
         *
         * @return [FlyJWT] object that is used to refresh the JWT.
         */
        fun get(context: Context): FlyJWT {
            return savedInstance ?: synchronized(this) {

                if (savedInstance == null){
                    val request = HttpDefaultClass.Builder(context, false, ENDPOINT)
                    savedInstance = FlyJWT(SharedElementsFactory.get(context), request)
                }
                return savedInstance as FlyJWT
            }
        }
    }

    /**
     * Refreshes the JWT associated to the application. If the user has not connect to Flybits
     * previously a 401 - Not Connected result will be returned.
     *
     * @return The [Result] which indicates whether or not the refreshing of the JWT was successful.
     */
    @Throws(FlybitsException::class)
    fun refresh() : Result<*> {
        if (sharedElements.getSavedJWTToken().isEmpty()){
            return Result<String>(401, "")
        }

        try {
            return httpClient.get().response
        } catch (e: IOException) {
            Logger.exception("FlyJWT.refreshJWT", e)
            throw FlybitsException("Error Connecting to Flybits Server: FlyException_JWT1")
        } catch (e: NullPointerException) {
            Logger.exception("FlyJWT.refreshJWT2", e)
            throw FlybitsException("Error Connecting to Flybits Server: FlyException_JWT2")
        }
    }
}