/*
 * Decompiled with CFR 0.152.
 */
package com.fizzed.jne;

import com.fizzed.jne.ABI;
import com.fizzed.jne.HardwareArchitecture;
import com.fizzed.jne.OperatingSystem;
import com.fizzed.jne.PlatformInfo;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

public class NativeTarget {
    private final OperatingSystem operatingSystem;
    private final HardwareArchitecture hardwareArchitecture;
    private final ABI abi;

    private NativeTarget(OperatingSystem operatingSystem, HardwareArchitecture hardwareArchitecture, ABI abi) {
        this.operatingSystem = operatingSystem;
        this.hardwareArchitecture = hardwareArchitecture;
        this.abi = abi;
    }

    public OperatingSystem getOperatingSystem() {
        return this.operatingSystem;
    }

    public HardwareArchitecture getHardwareArchitecture() {
        return this.hardwareArchitecture;
    }

    public ABI getAbi() {
        return this.abi;
    }

    private void checkHardwareArchitecture() {
        if (this.hardwareArchitecture == null) {
            throw new IllegalArgumentException("Hardware architecture was null");
        }
    }

    private void checkOperatingSystem() {
        if (this.operatingSystem == null) {
            throw new IllegalArgumentException("Operating system was null");
        }
    }

    public String getExecutableFileExtension() {
        this.checkOperatingSystem();
        switch (this.operatingSystem) {
            case WINDOWS: {
                return ".exe";
            }
        }
        return null;
    }

    public String resolveExecutableFileName(String name) {
        Objects.requireNonNull(name, "name was null");
        this.checkOperatingSystem();
        String ext = this.getExecutableFileExtension();
        if (ext != null) {
            return name + ext;
        }
        return name;
    }

    public String getLibraryFileExtension() {
        this.checkOperatingSystem();
        switch (this.operatingSystem) {
            case WINDOWS: {
                return ".dll";
            }
            case LINUX: 
            case FREEBSD: 
            case OPENBSD: 
            case SOLARIS: {
                return ".so";
            }
            case MACOS: {
                return ".dylib";
            }
        }
        return null;
    }

    public String resolveLibraryFileName(String name) {
        Objects.requireNonNull(name, "name was null");
        this.checkOperatingSystem();
        switch (this.operatingSystem) {
            case WINDOWS: {
                return name + ".dll";
            }
            case LINUX: 
            case FREEBSD: 
            case OPENBSD: 
            case SOLARIS: {
                return "lib" + name + ".so";
            }
            case MACOS: {
                return "lib" + name + ".dylib";
            }
        }
        return name;
    }

    public String toRustTarget() {
        this.checkHardwareArchitecture();
        this.checkOperatingSystem();
        String arch = null;
        String vendorOsEnv = null;
        switch (this.hardwareArchitecture) {
            case X64: {
                arch = "x86_64";
                break;
            }
            case X32: {
                arch = "i686";
                break;
            }
            case ARM64: {
                arch = "aarch64";
                break;
            }
            case RISCV64: {
                arch = "riscv64gc";
                break;
            }
            default: {
                throw new IllegalArgumentException("Unsupported rust target for hardware architecture " + (Object)((Object)this.hardwareArchitecture) + " (if it should be valid please add to " + this.getClass().getCanonicalName() + ")");
            }
        }
        switch (this.operatingSystem) {
            case WINDOWS: {
                if (this.abi == ABI.GNU) {
                    vendorOsEnv = "pc-windows-gnu";
                    break;
                }
                vendorOsEnv = "pc-windows-msvc";
                break;
            }
            case LINUX: {
                if (this.abi == ABI.MUSL) {
                    vendorOsEnv = "unknown-linux-musl";
                    break;
                }
                vendorOsEnv = "unknown-linux-gnu";
                break;
            }
            case MACOS: {
                vendorOsEnv = "apple-darwin";
                break;
            }
            case FREEBSD: {
                vendorOsEnv = "unknown-freebsd";
                break;
            }
            case OPENBSD: {
                vendorOsEnv = "unknown-openbsd";
                break;
            }
            case SOLARIS: {
                vendorOsEnv = "sun-solaris";
                break;
            }
            default: {
                throw new IllegalArgumentException("Unsupported rust target for operating system " + (Object)((Object)this.operatingSystem) + " (if it should be valid please add to " + this.getClass().getCanonicalName() + ")");
            }
        }
        return arch + "-" + vendorOsEnv;
    }

    public String toJneOsAbi() {
        this.checkOperatingSystem();
        return NativeTarget.toJneOsAbi(this.operatingSystem, this.abi, null);
    }

    public String toJneArch() {
        this.checkHardwareArchitecture();
        return this.hardwareArchitecture.name().toLowerCase();
    }

    public String toJneTarget() {
        return this.toJneOsAbi() + "-" + this.toJneArch();
    }

    private static String toJneOsAbi(OperatingSystem os, ABI abi, String osAlias) {
        NativeTarget.validateAbi(os, abi);
        if (os == OperatingSystem.LINUX) {
            if (abi == ABI.MUSL) {
                if (osAlias != null) {
                    return osAlias.toLowerCase() + "_" + abi.name().toLowerCase();
                }
                return os.name().toLowerCase() + "_" + abi.name().toLowerCase();
            }
        } else if (os == OperatingSystem.WINDOWS && abi == ABI.GNU) {
            if (osAlias != null) {
                return osAlias.toLowerCase() + "_" + abi.name().toLowerCase();
            }
            return os.name().toLowerCase() + "_" + abi.name().toLowerCase();
        }
        if (osAlias != null) {
            return osAlias.toLowerCase();
        }
        return os.name().toLowerCase();
    }

    private static void validateAbi(OperatingSystem os, ABI abi) {
        boolean invalidAbi = false;
        if (abi != null && abi != ABI.DEFAULT) {
            if (os == OperatingSystem.LINUX) {
                switch (abi) {
                    case GNU: 
                    case MUSL: {
                        break;
                    }
                    default: {
                        invalidAbi = true;
                        break;
                    }
                }
            } else if (os == OperatingSystem.WINDOWS) {
                switch (abi) {
                    case GNU: 
                    case MSVC: {
                        break;
                    }
                    default: {
                        invalidAbi = true;
                        break;
                    }
                }
            } else {
                invalidAbi = true;
            }
        }
        if (invalidAbi) {
            throw new IllegalArgumentException("ABI " + (Object)((Object)abi) + " is not valid for operating system " + (Object)((Object)os));
        }
    }

    public static String toJneArch(HardwareArchitecture hardwareArchitecture, String hardwareArchitectureAlias) {
        if (hardwareArchitectureAlias != null) {
            return hardwareArchitectureAlias.toLowerCase();
        }
        return hardwareArchitecture.name().toLowerCase();
    }

    public List<String> resolveResourcePaths(String resourcePrefix, String name) {
        ArrayList<String> jneOsAbis = new ArrayList<String>();
        if (this.operatingSystem != null) {
            jneOsAbis.add(NativeTarget.toJneOsAbi(this.operatingSystem, this.abi, null));
            if (this.operatingSystem.getAliases() != null) {
                for (String alias : this.operatingSystem.getAliases()) {
                    jneOsAbis.add(NativeTarget.toJneOsAbi(this.operatingSystem, this.abi, alias));
                }
            }
        } else {
            jneOsAbis.add(null);
        }
        ArrayList<String> jneArchs = new ArrayList<String>();
        if (this.hardwareArchitecture != null) {
            jneArchs.add(NativeTarget.toJneArch(this.hardwareArchitecture, null));
            if (this.hardwareArchitecture.getAliases() != null) {
                for (String alias : this.hardwareArchitecture.getAliases()) {
                    jneArchs.add(NativeTarget.toJneArch(this.hardwareArchitecture, alias));
                }
            }
        } else {
            jneArchs.add(null);
        }
        ArrayList<String> resourcePaths = new ArrayList<String>();
        for (String jneOsAbi : jneOsAbis) {
            for (String jneArch : jneArchs) {
                StringBuilder s = new StringBuilder();
                s.append(resourcePrefix);
                if (jneOsAbi != null) {
                    s.append("/");
                    s.append(jneOsAbi);
                }
                if (jneArch != null) {
                    s.append("/");
                    s.append(jneArch);
                }
                s.append("/");
                s.append(name);
                resourcePaths.add(s.toString());
            }
        }
        return resourcePaths;
    }

    public static NativeTarget of(OperatingSystem os, HardwareArchitecture arch, ABI abi) {
        return new NativeTarget(os, arch, abi);
    }

    public static NativeTarget detect() {
        OperatingSystem os = PlatformInfo.detectOperatingSystem();
        HardwareArchitecture arch = PlatformInfo.detectHardwareArchitecture();
        ABI abi = PlatformInfo.detectAbi(os);
        return new NativeTarget(os, arch, abi);
    }

    public static NativeTarget fromJneTarget(String jneTarget) {
        String abiStr;
        String osStr;
        if (jneTarget == null || jneTarget.trim().isEmpty()) {
            throw new IllegalArgumentException("JNE target was null or empty");
        }
        int hyphenPos = jneTarget.indexOf("-");
        if (hyphenPos < 2 || hyphenPos >= jneTarget.length() - 2) {
            throw new IllegalArgumentException("JNE target [" + jneTarget + "] was not of format <os+abi>/arch such as linux-x64 or windows-x64");
        }
        String osAbiStr = jneTarget.substring(0, hyphenPos);
        String archStr = jneTarget.substring(hyphenPos + 1);
        int underscorePos = osAbiStr.indexOf("_");
        if (underscorePos > 1) {
            osStr = osAbiStr.substring(0, underscorePos);
            abiStr = osAbiStr.substring(underscorePos + 1);
        } else {
            osStr = osAbiStr;
            abiStr = null;
        }
        OperatingSystem os = OperatingSystem.resolve(osStr);
        HardwareArchitecture arch = HardwareArchitecture.resolve(archStr);
        ABI abi = ABI.resolve(abiStr);
        if (os == null) {
            throw new IllegalArgumentException("JNE target [" + jneTarget + "] with an unsupported operating system [" + osStr + "]");
        }
        if (arch == null) {
            throw new IllegalArgumentException("JNE target [" + jneTarget + "] with an unsupported hardware architecture [" + archStr + "]");
        }
        if (abiStr != null && abi == null) {
            throw new IllegalArgumentException("JNE target [" + jneTarget + "] with an unsupported abi [" + abiStr + "]");
        }
        NativeTarget.validateAbi(os, abi);
        return new NativeTarget(os, arch, abi);
    }
}

