/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.plugin.base.security;

import com.facebook.airlift.log.Logger;
import com.facebook.airlift.security.pem.PemReader;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.StandardErrorCode;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.cert.Certificate;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509TrustManager;
import javax.security.auth.x500.X500Principal;

public class SslContextProvider {
    private static final Logger log = Logger.get(SslContextProvider.class);
    private static final String TLS_PROTOCOL = "TLS";
    private final Optional<File> keystorePath;
    private final Optional<String> keystorePassword;
    private final Optional<File> truststorePath;
    private final Optional<String> truststorePassword;

    public SslContextProvider(Optional<File> keystorePath, Optional<String> keystorePassword, Optional<File> truststorePath, Optional<String> truststorePassword) {
        this.keystorePath = keystorePath;
        this.keystorePassword = keystorePassword;
        this.truststorePath = truststorePath;
        this.truststorePassword = truststorePassword;
    }

    public Optional<SSLContext> buildSslContext() {
        if (!this.keystorePath.isPresent() && !this.truststorePath.isPresent()) {
            log.debug("No SSL configuration provided, returning empty SSL context");
            return Optional.empty();
        }
        try {
            log.debug("Creating SSL context with keystore: {}, truststore: {}", new Object[]{this.keystorePath.map(File::getPath).orElse("none"), this.truststorePath.map(File::getPath).orElse("none")});
            return Optional.of(this.createSSLContext());
        }
        catch (IOException | GeneralSecurityException e) {
            log.error("Failed to initialize SSL context", new Object[]{e});
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, "Failed to initialize SSL context", (Throwable)e);
        }
    }

    private SSLContext createSSLContext() throws GeneralSecurityException, IOException {
        KeyStore keystore = null;
        KeyManager[] keyManagers = null;
        if (this.keystorePath.isPresent()) {
            log.debug("Loading keystore from: {}", new Object[]{this.keystorePath.get().getPath()});
            keystore = this.loadKeyStore();
            keyManagers = this.createKeyManagers(keystore);
            log.debug("Keystore loaded successfully");
        }
        KeyStore truststore = null;
        if (this.truststorePath.isPresent()) {
            log.debug("Loading truststore from: {}", new Object[]{this.truststorePath.get().getPath()});
            truststore = SslContextProvider.loadTrustStore(this.truststorePath.get(), this.truststorePassword);
            log.debug("Truststore loaded successfully");
        } else if (keystore != null) {
            log.debug("No custom truststore configured, using keystore as truststore");
            truststore = keystore;
        } else {
            log.debug("No custom truststore or keystore configured, using system default");
        }
        X509TrustManager trustManager = this.createTrustManager(truststore);
        SSLContext sslContext = SSLContext.getInstance(TLS_PROTOCOL);
        sslContext.init(keyManagers, new TrustManager[]{trustManager}, null);
        log.debug("SSL Context initialized with TLS protocol");
        return sslContext;
    }

    private KeyStore loadKeyStore() throws GeneralSecurityException, IOException {
        KeyStore keystore;
        File keystoreFile = this.keystorePath.get();
        try {
            log.debug("Attempting to load keystore as PEM format");
            keystore = PemReader.loadKeyStore((File)keystoreFile, (File)keystoreFile, this.keystorePassword);
            log.debug("Successfully loaded keystore as PEM format");
        }
        catch (IOException | GeneralSecurityException e) {
            log.debug("Failed to load keystore as PEM, attempting JKS format: {}", new Object[]{e.getMessage()});
            keystore = KeyStore.getInstance(KeyStore.getDefaultType());
            try (FileInputStream in = new FileInputStream(keystoreFile);){
                keystore.load(in, this.keystorePassword.map(String::toCharArray).orElse(null));
            }
            log.debug("Successfully loaded keystore as JKS format");
        }
        SslContextProvider.validateCertificates(keystore);
        return keystore;
    }

    private KeyManager[] createKeyManagers(KeyStore keystore) throws GeneralSecurityException {
        char[] keyManagerPassword = this.keystorePassword.map(String::toCharArray).orElse(null);
        KeyManagerFactory keyManagerFactory = KeyManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
        keyManagerFactory.init(keystore, keyManagerPassword);
        return keyManagerFactory.getKeyManagers();
    }

    private X509TrustManager createTrustManager(KeyStore truststore) throws GeneralSecurityException {
        TrustManagerFactory trustManagerFactory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
        if (truststore != null) {
            try {
                ArrayList<String> aliases = Collections.list(truststore.aliases());
                if (aliases.isEmpty()) {
                    throw new GeneralSecurityException("Truststore is empty - no trusted certificates found");
                }
                log.debug("Truststore contains {} certificate(s): {}", new Object[]{aliases.size(), aliases});
            }
            catch (KeyStoreException e) {
                throw new GeneralSecurityException("Failed to read truststore", e);
            }
        }
        trustManagerFactory.init(truststore);
        Object[] trustManagers = trustManagerFactory.getTrustManagers();
        if (trustManagers.length != 1 || !(trustManagers[0] instanceof X509TrustManager)) {
            throw new RuntimeException("Unexpected default trust managers: " + Arrays.toString(trustManagers));
        }
        return (X509TrustManager)trustManagers[0];
    }

    private static KeyStore loadTrustStore(File trustStorePath, Optional<String> trustStorePassword) throws GeneralSecurityException {
        KeyStore trustStore = KeyStore.getInstance(KeyStore.getDefaultType());
        boolean loaded = false;
        Exception lastException = null;
        try {
            log.debug("Attempting to load truststore as PEM format");
            List certificateChain = PemReader.readCertificateChain((File)trustStorePath);
            if (!certificateChain.isEmpty()) {
                trustStore.load(null, null);
                for (int i = 0; i < certificateChain.size(); ++i) {
                    X509Certificate certificate = (X509Certificate)certificateChain.get(i);
                    X500Principal principal = certificate.getSubjectX500Principal();
                    String alias = "cert-" + i + "-" + principal.getName();
                    trustStore.setCertificateEntry(alias, certificate);
                }
                loaded = true;
                log.debug("Successfully loaded {} certificates from PEM truststore", new Object[]{certificateChain.size()});
            }
        }
        catch (IOException | GeneralSecurityException e) {
            log.debug("Failed to load truststore as PEM format: {}", new Object[]{e.getMessage()});
            lastException = e;
        }
        if (!loaded) {
            try {
                log.debug("Attempting to load truststore as JKS format");
                try (FileInputStream inputStream = new FileInputStream(trustStorePath);){
                    trustStore.load(inputStream, trustStorePassword.map(String::toCharArray).orElse(null));
                }
                log.debug("Successfully loaded truststore as JKS format");
            }
            catch (IOException | GeneralSecurityException e) {
                log.debug("Failed to load truststore as JKS format: {}", new Object[]{e.getMessage()});
                throw new GeneralSecurityException("Failed to load truststore as both PEM and KeyStore format. PEM error: " + (lastException != null ? lastException.getMessage() : "unknown") + ", KeyStore error: " + e.getMessage(), e);
            }
        }
        try {
            ArrayList<String> aliases = Collections.list(trustStore.aliases());
            if (aliases.isEmpty()) {
                throw new GeneralSecurityException("Loaded truststore is empty - no certificates found in: " + trustStorePath);
            }
            log.debug("Truststore loaded with {} certificate(s)", new Object[]{aliases.size()});
        }
        catch (KeyStoreException e) {
            throw new GeneralSecurityException("Failed to verify truststore contents", e);
        }
        return trustStore;
    }

    private static void validateCertificates(KeyStore keyStore) throws GeneralSecurityException {
        for (String alias : Collections.list(keyStore.aliases())) {
            Certificate certificate;
            if (!keyStore.isKeyEntry(alias) || !((certificate = keyStore.getCertificate(alias)) instanceof X509Certificate)) continue;
            try {
                ((X509Certificate)certificate).checkValidity();
                log.debug("Certificate '{}' is valid", new Object[]{alias});
            }
            catch (CertificateExpiredException e) {
                throw new CertificateExpiredException("KeyStore certificate '" + alias + "' is expired: " + e.getMessage());
            }
            catch (CertificateNotYetValidException e) {
                throw new CertificateNotYetValidException("KeyStore certificate '" + alias + "' is not yet valid: " + e.getMessage());
            }
        }
    }
}

