/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.cache.filemerge;

import alluxio.collections.ConcurrentHashSet;
import com.facebook.airlift.log.Logger;
import com.facebook.presto.cache.CacheConfig;
import com.facebook.presto.cache.CacheManager;
import com.facebook.presto.cache.CacheResult;
import com.facebook.presto.cache.CacheStats;
import com.facebook.presto.cache.FileReadRequest;
import com.facebook.presto.cache.filemerge.FileMergeCacheConfig;
import com.facebook.presto.hive.CacheQuota;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.StandardErrorCode;
import com.google.common.base.Preconditions;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterators;
import com.google.common.collect.Range;
import com.google.common.collect.RangeMap;
import com.google.common.collect.TreeRangeMap;
import io.airlift.slice.Slice;
import io.airlift.units.DataSize;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.stream.Collectors;
import javax.annotation.PreDestroy;
import javax.inject.Inject;
import org.apache.hadoop.fs.Path;

public class FileMergeCacheManager
implements CacheManager {
    private static final Logger log = Logger.get(FileMergeCacheManager.class);
    private static final String EXTENSION = ".cache";
    private static final int FILE_MERGE_BUFFER_SIZE = StrictMath.toIntExact(new DataSize(8.0, DataSize.Unit.MEGABYTE).toBytes());
    private final ThreadLocal<byte[]> buffers = ThreadLocal.withInitial(() -> new byte[FILE_MERGE_BUFFER_SIZE]);
    private final ExecutorService cacheFlushExecutor;
    private final ExecutorService cacheRemovalExecutor;
    private final ScheduledExecutorService cacheSizeCalculateExecutor;
    private final Map<Path, CacheRange> persistedRanges = new ConcurrentHashMap<Path, CacheRange>();
    private final Cache<Path, Long> cache;
    private final Map<Long, Set<Path>> cacheScopeFiles = new ConcurrentHashMap<Long, Set<Path>>();
    private final Map<Long, Long> cacheScopeSizeInBytes = new ConcurrentHashMap<Long, Long>();
    private final CacheStats stats;
    private final Path baseDirectory;
    private final long maxInflightBytes;

    @Inject
    public FileMergeCacheManager(CacheConfig cacheConfig, FileMergeCacheConfig fileMergeCacheConfig, CacheStats stats, ExecutorService cacheFlushExecutor, ExecutorService cacheRemovalExecutor, ScheduledExecutorService cacheSizeCalculateExecutor) {
        Objects.requireNonNull(cacheConfig, "directory is null");
        this.cacheFlushExecutor = cacheFlushExecutor;
        this.cacheRemovalExecutor = cacheRemovalExecutor;
        this.cacheSizeCalculateExecutor = cacheSizeCalculateExecutor;
        this.cache = CacheBuilder.newBuilder().maximumSize((long)fileMergeCacheConfig.getMaxCachedEntries()).expireAfterAccess(fileMergeCacheConfig.getCacheTtl().toMillis(), TimeUnit.MILLISECONDS).removalListener((RemovalListener)new CacheRemovalListener()).recordStats().build();
        this.stats = Objects.requireNonNull(stats, "stats is null");
        this.baseDirectory = new Path(cacheConfig.getBaseDirectory());
        Preconditions.checkArgument((fileMergeCacheConfig.getMaxInMemoryCacheSize().toBytes() >= 0L ? 1 : 0) != 0, (Object)"maxInflightBytes is negative");
        this.maxInflightBytes = fileMergeCacheConfig.getMaxInMemoryCacheSize().toBytes();
        File target = new File(this.baseDirectory.toUri());
        if (!target.exists()) {
            try {
                Files.createDirectories(target.toPath(), new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, "cannot create cache directory " + target, (Throwable)e);
            }
        } else {
            File[] files = target.listFiles();
            if (files == null) {
                return;
            }
            this.cacheRemovalExecutor.submit(() -> Arrays.stream(files).forEach(file -> {
                try {
                    Files.delete(file.toPath());
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }));
        }
        this.cacheSizeCalculateExecutor.scheduleAtFixedRate(() -> {
            try {
                this.cacheScopeFiles.keySet().forEach(cacheIdentifier -> this.cacheScopeSizeInBytes.put((Long)cacheIdentifier, this.getCacheScopeSizeInBytes((long)cacheIdentifier)));
                this.cacheScopeSizeInBytes.keySet().removeIf(key -> !this.cacheScopeFiles.containsKey(key));
            }
            catch (Throwable t) {
                log.error(t, "Error calculating cache size");
            }
        }, 0L, 15L, TimeUnit.SECONDS);
    }

    @PreDestroy
    public void destroy() {
        this.cacheFlushExecutor.shutdownNow();
        this.cacheRemovalExecutor.shutdownNow();
        this.cacheSizeCalculateExecutor.shutdownNow();
        this.buffers.remove();
    }

    @Override
    public CacheResult get(FileReadRequest request, byte[] buffer, int offset, CacheQuota cacheQuota) {
        boolean result = this.read(request, buffer, offset);
        if (!result && this.ifExceedQuota(cacheQuota, request)) {
            this.stats.incrementQuotaExceed();
            return CacheResult.CACHE_QUOTA_EXCEED;
        }
        try {
            this.cache.get((Object)request.getPath(), () -> ((CacheQuota)cacheQuota).getIdentifier());
        }
        catch (ExecutionException executionException) {
            // empty catch block
        }
        if (result) {
            this.stats.incrementCacheHit();
            return CacheResult.HIT;
        }
        this.stats.incrementCacheMiss();
        return CacheResult.MISS;
    }

    private boolean ifExceedQuota(CacheQuota cacheQuota, FileReadRequest request) {
        DataSize cacheSize = DataSize.succinctBytes((long)(this.cacheScopeSizeInBytes.getOrDefault(cacheQuota.getIdentifier(), 0L) + (long)request.getLength()));
        return cacheQuota.getQuota().map(quota -> cacheSize.compareTo(quota) > 0).orElse(false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private long getCacheScopeSizeInBytes(long cacheScopeIdentifier) {
        Set<Path> paths = this.cacheScopeFiles.get(cacheScopeIdentifier);
        if (paths == null) {
            return 0L;
        }
        long bytes = 0L;
        for (Path path : paths) {
            CacheRange cacheRange = this.persistedRanges.get(path);
            if (cacheRange == null) continue;
            Lock readLock = cacheRange.getLock().readLock();
            readLock.lock();
            try {
                for (Range range : cacheRange.getRange().asDescendingMapOfRanges().keySet()) {
                    bytes += (Long)range.upperEndpoint() - (Long)range.lowerEndpoint();
                }
            }
            finally {
                readLock.unlock();
            }
        }
        return bytes;
    }

    @Override
    public void put(FileReadRequest key, Slice data, CacheQuota cacheQuota) {
        if (this.stats.getInMemoryRetainedBytes() + (long)data.length() >= this.maxInflightBytes) {
            return;
        }
        Set paths = this.cacheScopeFiles.computeIfAbsent(cacheQuota.getIdentifier(), k -> new ConcurrentHashSet());
        paths.add(key.getPath());
        this.stats.addInMemoryRetainedBytes(data.length());
        byte[] copy = data.getBytes();
        this.cacheFlushExecutor.submit(() -> {
            Path newFilePath = new Path(this.baseDirectory.toUri() + "/" + UUID.randomUUID() + EXTENSION);
            if (!this.write(key, copy, newFilePath)) {
                log.warn("%s Fail to persist cache %s with length %s ", new Object[]{Thread.currentThread().getName(), newFilePath, key.getLength()});
            }
            this.stats.addInMemoryRetainedBytes(-copy.length);
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean read(FileReadRequest request, byte[] buffer, int offset) {
        LocalCacheFile cacheFile;
        if (request.getLength() <= 0) {
            return true;
        }
        CacheRange cacheRange = this.persistedRanges.get(request.getPath());
        if (cacheRange == null) {
            return false;
        }
        Lock readLock = cacheRange.getLock().readLock();
        readLock.lock();
        try {
            Map diskRanges = cacheRange.getRange().subRangeMap(Range.closedOpen((Comparable)Long.valueOf(request.getOffset()), (Comparable)Long.valueOf((long)request.getLength() + request.getOffset()))).asMapOfRanges();
            if (diskRanges.size() != 1) {
                boolean bl2 = false;
                return bl2;
            }
            cacheFile = (LocalCacheFile)((Map.Entry)Iterators.getOnlyElement(diskRanges.entrySet().iterator())).getValue();
        }
        finally {
            readLock.unlock();
        }
        try (RandomAccessFile file = new RandomAccessFile(new File(cacheFile.getPath().toUri()), "r");){
            file.seek(request.getOffset() - cacheFile.getOffset());
            file.readFully(buffer, offset, request.getLength());
            boolean bl = true;
            return bl;
        }
        catch (IOException e) {
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean write(FileReadRequest key, byte[] data, Path newFilePath) {
        boolean updated;
        long newFileOffset;
        int newFileLength;
        LocalCacheFile followingCacheFile;
        LocalCacheFile previousCacheFile;
        Path targetFile = key.getPath();
        this.persistedRanges.putIfAbsent(targetFile, new CacheRange());
        CacheRange cacheRange = this.persistedRanges.get(targetFile);
        if (cacheRange == null) {
            return false;
        }
        Lock readLock = cacheRange.getLock().readLock();
        readLock.lock();
        try {
            RangeMap<Long, LocalCacheFile> cache = cacheRange.getRange();
            previousCacheFile = (LocalCacheFile)cache.get((Comparable)Long.valueOf(key.getOffset() - 1L));
            followingCacheFile = (LocalCacheFile)cache.get((Comparable)Long.valueOf(key.getOffset() + (long)key.getLength()));
        }
        finally {
            readLock.unlock();
        }
        if (previousCacheFile != null && FileMergeCacheManager.cacheFileEquals(previousCacheFile, followingCacheFile)) {
            log.debug("%s found covered range %s", new Object[]{Thread.currentThread().getName(), previousCacheFile.getPath()});
            return true;
        }
        File newFile = new File(newFilePath.toUri());
        try {
            if (previousCacheFile == null) {
                Files.write(newFile.toPath(), data, StandardOpenOption.CREATE_NEW);
                newFileLength = data.length;
                newFileOffset = key.getOffset();
            } else {
                int previousFileLength = this.appendToFile(previousCacheFile, 0L, newFile);
                long previousFileOffset = previousCacheFile.getOffset();
                int remainingCacheFileOffset = StrictMath.toIntExact((long)previousFileLength + previousFileOffset - key.getOffset());
                int remainingCacheFileLength = StrictMath.toIntExact((long)key.getLength() + key.getOffset() - ((long)previousFileLength + previousFileOffset));
                try (RandomAccessFile randomAccessFile = new RandomAccessFile(newFile, "rw");){
                    randomAccessFile.seek(randomAccessFile.length());
                    randomAccessFile.write(data, remainingCacheFileOffset, remainingCacheFileLength);
                }
                newFileLength = previousFileLength + remainingCacheFileLength;
                newFileOffset = previousFileOffset;
            }
            if (followingCacheFile != null) {
                newFileLength += this.appendToFile(followingCacheFile, key.getOffset() + (long)key.getLength() - followingCacheFile.getOffset(), newFile);
            }
        }
        catch (IOException e) {
            log.warn((Throwable)e, "%s encountered an error while flushing file %s", new Object[]{Thread.currentThread().getName(), newFilePath});
            FileMergeCacheManager.tryDeleteFile(newFilePath);
            return false;
        }
        Object cacheFilesToDelete = new HashSet();
        Lock writeLock = this.persistedRanges.get(targetFile).getLock().writeLock();
        writeLock.lock();
        try {
            RangeMap<Long, LocalCacheFile> cache = this.persistedRanges.get(targetFile).getRange();
            LocalCacheFile newPreviousCacheFile = (LocalCacheFile)cache.get((Comparable)Long.valueOf(key.getOffset() - 1L));
            LocalCacheFile newFollowingCacheFile = (LocalCacheFile)cache.get((Comparable)Long.valueOf(key.getOffset() + (long)key.getLength()));
            if (!FileMergeCacheManager.cacheFileEquals(previousCacheFile, newPreviousCacheFile) || !FileMergeCacheManager.cacheFileEquals(followingCacheFile, newFollowingCacheFile)) {
                updated = false;
            } else {
                updated = true;
                cacheFilesToDelete = cache.subRangeMap(Range.closedOpen((Comparable)Long.valueOf(key.getOffset()), (Comparable)Long.valueOf(key.getOffset() + (long)key.getLength()))).asMapOfRanges().values().stream().map(LocalCacheFile::getPath).collect(Collectors.toSet());
                Range newRange = Range.closedOpen((Comparable)Long.valueOf(newFileOffset), (Comparable)Long.valueOf(newFileOffset + (long)newFileLength));
                cache.remove(newRange);
                cache.put(newRange, (Object)new LocalCacheFile(newFileOffset, newFilePath));
            }
        }
        finally {
            writeLock.unlock();
        }
        if (updated) {
            if (previousCacheFile != null) {
                cacheFilesToDelete.add(previousCacheFile.getPath());
            }
            if (followingCacheFile != null) {
                cacheFilesToDelete.add(followingCacheFile.getPath());
            }
        } else {
            cacheFilesToDelete = ImmutableSet.of((Object)newFilePath);
        }
        cacheFilesToDelete.forEach(FileMergeCacheManager::tryDeleteFile);
        return true;
    }

    private int appendToFile(LocalCacheFile source, long offset, File destination) throws IOException {
        int totalBytesRead = 0;
        try (FileInputStream fileInputStream = new FileInputStream(new File(source.getPath().toUri()));){
            int readBytes;
            fileInputStream.getChannel().position(offset);
            byte[] buffer = this.buffers.get();
            while ((readBytes = fileInputStream.read(buffer)) > 0) {
                if (!Files.exists(destination.toPath(), new LinkOption[0])) {
                    Files.createFile(destination.toPath(), new FileAttribute[0]);
                }
                totalBytesRead += readBytes;
                if (readBytes != FILE_MERGE_BUFFER_SIZE) {
                    RandomAccessFile randomAccessFile = new RandomAccessFile(destination, "rw");
                    Throwable throwable = null;
                    try {
                        randomAccessFile.seek(destination.length());
                        randomAccessFile.write(buffer, 0, readBytes);
                        continue;
                    }
                    catch (Throwable throwable2) {
                        throwable = throwable2;
                        throw throwable2;
                    }
                    finally {
                        if (randomAccessFile == null) continue;
                        if (throwable != null) {
                            try {
                                randomAccessFile.close();
                            }
                            catch (Throwable throwable3) {
                                throwable.addSuppressed(throwable3);
                            }
                            continue;
                        }
                        randomAccessFile.close();
                        continue;
                    }
                }
                Files.write(destination.toPath(), buffer, StandardOpenOption.APPEND);
            }
        }
        return totalBytesRead;
    }

    private static void tryDeleteFile(Path path) {
        try {
            File file = new File(path.toUri());
            if (file.exists()) {
                Files.delete(file.toPath());
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private static boolean cacheFileEquals(LocalCacheFile left, LocalCacheFile right) {
        if (left == null && right == null) {
            return true;
        }
        if (left == null || right == null) {
            return false;
        }
        return left.equals(right);
    }

    private class CacheRemovalListener
    implements RemovalListener<Path, Long> {
        private CacheRemovalListener() {
        }

        public void onRemoval(RemovalNotification<Path, Long> notification) {
            Path path = (Path)notification.getKey();
            CacheRange cacheRange = (CacheRange)FileMergeCacheManager.this.persistedRanges.remove(path);
            Set paths = (Set)FileMergeCacheManager.this.cacheScopeFiles.get(notification.getValue());
            if (paths != null) {
                paths.remove(path);
                if (paths.isEmpty()) {
                    FileMergeCacheManager.this.cacheScopeFiles.remove(notification.getValue(), Collections.emptySet());
                }
            }
            if (cacheRange == null) {
                return;
            }
            FileMergeCacheManager.this.cacheRemovalExecutor.submit(() -> {
                Collection files;
                cacheRange.lock.readLock().lock();
                try {
                    files = cacheRange.getRange().asMapOfRanges().values();
                }
                finally {
                    cacheRange.lock.readLock().unlock();
                }
                for (LocalCacheFile file : files) {
                    try {
                        Files.delete(new File(file.getPath().toUri()).toPath());
                    }
                    catch (IOException iOException) {}
                }
            });
        }
    }

    private static class CacheRange {
        private final RangeMap<Long, LocalCacheFile> range = TreeRangeMap.create();
        private final ReadWriteLock lock = new ReentrantReadWriteLock();

        private CacheRange() {
        }

        public RangeMap<Long, LocalCacheFile> getRange() {
            return this.range;
        }

        public ReadWriteLock getLock() {
            return this.lock;
        }
    }

    private static class LocalCacheFile {
        private final long offset;
        private final Path path;

        public LocalCacheFile(long offset, Path path) {
            this.offset = offset;
            this.path = path;
        }

        public long getOffset() {
            return this.offset;
        }

        public Path getPath() {
            return this.path;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            LocalCacheFile that = (LocalCacheFile)o;
            return Objects.equals(this.offset, that.offset) && Objects.equals(this.path, that.path);
        }

        public int hashCode() {
            return Objects.hash(this.offset, this.path);
        }
    }
}

