/*
 * Copyright (c) 2014 by EagleXad
 * Team: EagleXad
 * Create: 2014-08-29
 */

package com.eaglexad.lib.http.tool;

import android.net.Uri;
import android.os.Build;

import com.eaglexad.lib.http.ExHttp;
import com.eaglexad.lib.http.entry.ExRequest;
import com.eaglexad.lib.http.entry.ExResponse;
import com.eaglexad.lib.http.exception.ErrorConnection;
import com.eaglexad.lib.http.ible.IExHttpStack;
import com.eaglexad.lib.http.ible.IExRequestBody;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.Proxy;
import java.net.URL;
import java.net.URLDecoder;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocketFactory;

/**
 * @author Aloneter
 * @ClassName: ExHttpStackRequest
 * @Description:
 */
public class ExHttpStackRequest implements IExHttpStack {

    private ExRequest mRequest;
    private String mUrl;
    private Map<String, String> mAdditionalHeaders;
    private HttpURLConnection mConnection;
    private InputStream mInputStream;

    @Override
    public ExResponse performRequest(ExRequest request, Map<String, String> additionalHeaders) throws ErrorConnection {

        mRequest = request;
        mAdditionalHeaders = additionalHeaders;
        mUrl = initUrl();

        int responseCode;

        ExResponse.Builder builder = new ExResponse.Builder(request.what);
        builder.setReuest(request);

        try {

            prepare();
            openConnection();
            initHeader();
            writeBody();

            responseCode = getResponseCode();

            byte[] data = new byte[0];

            if (responseCode == HttpURLConnection.HTTP_OK) {
                mInputStream = mConnection.getInputStream();
                if (!ExHttpUtils.isEmpty(mInputStream)) {
                    data = ExHttpUtils.inputStreamToBytes(mInputStream, getContentLength());
                }

                builder.setIsSuccess();
            }
            if (responseCode >= 300) {
                mInputStream = mConnection.getErrorStream();
                if (mInputStream != null) {
                    data = ExHttpUtils.inputStreamToBytes(mInputStream, getContentLength());
                }
            }
            if (data.length > 0) {
                builder.setData(data);
            }

            builder.setContentLength(getContentLength());
            builder.setLastModified(getLastModified());
            builder.setExpiration(getExpiration());
            builder.setETag(getETag());
            builder.setResponseMessage(getResponseMessage());
            builder.setResponseCode(responseCode);
            builder.setResponseHeaders(getResponseHeaders());

        } catch (IOException e) {
            throw new ErrorConnection(e.getMessage(), e);
        } finally {
            close();
        }

        return builder.builder();
    }

    @Override
    public void close() {

        if (mInputStream != null) {
            ExHttpUtils.closeQuietly(mInputStream);
        }
        if (mConnection != null) {
            mConnection.disconnect();
        }
    }

    private String initUrl() {

        String uri = mRequest.url;
        StringBuilder queryBuilder = new StringBuilder(uri);

        if (!uri.contains("?")) {
            queryBuilder.append("?");
        } else if (!uri.endsWith("?")) {
            queryBuilder.append("&");
        }

        List<ExRequest.KeyValue> queryParams = mRequest.queryStringParams;

        if (!ExHttpUtils.isEmpty(queryParams)) {
            for (ExRequest.KeyValue kv : queryParams) {
                String name = kv.key;
                String value = kv.getValueStr();

                if (!ExHttpUtils.isEmpty(name) && value != null) {
                    queryBuilder.append(Uri.encode(name, mRequest.charset));
                    queryBuilder.append("=");
                    queryBuilder.append(Uri.encode(value, mRequest.charset));
                    queryBuilder.append("&");
                }
            }
        }
        if (queryBuilder.charAt(queryBuilder.length() - 1) == '&') {
            queryBuilder.deleteCharAt(queryBuilder.length() - 1);
        }
        if (queryBuilder.charAt(queryBuilder.length() - 1) == '?') {
            queryBuilder.deleteCharAt(queryBuilder.length() - 1);
        }

        return queryBuilder.toString();
    }

    private void prepare() {

    }

    private void openConnection() throws IOException {

        URL url = new URL(mUrl);
        Proxy proxy = mRequest.proxy;

        if (proxy != null) {
            mConnection = (HttpURLConnection) url.openConnection(proxy);
        } else {
            mConnection = (HttpURLConnection) url.openConnection();
        }

        mConnection.setReadTimeout(mRequest.connectTimeout);
        mConnection.setConnectTimeout(mRequest.connectTimeout);
        mConnection.setInstanceFollowRedirects(mRequest.getRedirectIEx() == null);

        if (mConnection instanceof HttpsURLConnection) {
            SSLSocketFactory sslSocketFactory = mRequest.sslSocketFactory;

            if (sslSocketFactory != null) {
                ((HttpsURLConnection) mConnection).setHostnameVerifier(new HostnameVerifier() {
                    @Override
                    public boolean verify(String hostname, SSLSession session) {

                        return mRequest.hostNameVerify;
                    }
                });

                ((HttpsURLConnection) mConnection).setSSLSocketFactory(sslSocketFactory);
            }
        }
    }

    private void initHeader() {

        List<ExRequest.Header> headers = mRequest.headers;

        if (headers != null) {
            for (ExRequest.Header header : headers) {
                String name = header.key;
                String value = header.getValueStr();

                if (!ExHttpUtils.isEmpty(name) && !ExHttpUtils.isEmpty(value)) {
                    if (header.setHeader) {
                        mConnection.setRequestProperty(name, value);
                    } else {
                        mConnection.addRequestProperty(name, value);
                    }
                }
            }
        }
        if (mAdditionalHeaders != null && mAdditionalHeaders.size() > 0) {
            for (String key : mAdditionalHeaders.keySet()) {
                String value = mAdditionalHeaders.get(key);

                if (!ExHttpUtils.isEmpty(value)) {
                    mConnection.addRequestProperty(key, value);
                }
            }
        }
    }

    private void writeBody() throws IOException {

        ExHttp.Method method = mRequest.method;
        mConnection.setRequestMethod(method.toString());

        if (ExHttp.Method.permitsRequestBody(method)) {
            IExRequestBody body = mRequest.getRequestBody();

            if (body != null) {
                String contentType = body.getContentType();

                if (!ExHttpUtils.isEmpty(contentType)) {
                    mConnection.setRequestProperty("Content-Type", contentType);
                }

                long contentLength = body.getContentLength();

                if (contentLength < 0) {
                    mConnection.setChunkedStreamingMode(256 * 1024);
                } else {
                    if (contentLength < Integer.MAX_VALUE) {
                        mConnection.setFixedLengthStreamingMode((int) contentLength);
                    } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                        mConnection.setFixedLengthStreamingMode(contentLength);
                    } else {
                        mConnection.setChunkedStreamingMode(256 * 1024);
                    }
                }

                mConnection.setRequestProperty("Content-Length", String.valueOf(contentLength));
                mConnection.setDoOutput(true);

                body.writeTo(mConnection.getOutputStream());
            }
        }
    }

    private long getContentLength() {

        long result = 0;

        if (mConnection != null) {
            try {
                result = mConnection.getContentLength();
            } catch (Throwable ex) {
            }
            if (result < 1) {
                try {
                    result = mInputStream.available();
                } catch (Throwable ignored) {
                }
            }
        } else {
            try {
                result = mInputStream.available();
            } catch (Throwable ignored) {
            }
        }

        return result;
    }

    private long getExpiration() {

        if (mConnection == null) {
            return -1L;
        }

        long expiration = -1L;

        // from max-age
        String cacheControl = mConnection.getHeaderField("Cache-Control");

        if (!ExHttpUtils.isEmpty(cacheControl)) {
            StringTokenizer tok = new StringTokenizer(cacheControl, ",");

            while (tok.hasMoreTokens()) {
                String token = tok.nextToken().trim().toLowerCase();

                if (token.startsWith("max-age")) {
                    int eqIdx = token.indexOf('=');

                    if (eqIdx > 0) {
                        try {
                            String value = token.substring(eqIdx + 1).trim();
                            long seconds = Long.parseLong(value);

                            if (seconds > 0L) {
                                expiration = System.currentTimeMillis() + seconds * 1000L;
                            }
                        } catch (Throwable ex) {
                        }
                    }
                    break;
                }
            }
        }

        // from expires
        if (expiration <= 0L) {
            expiration = mConnection.getExpiration();
        }
        if (expiration <= 0) {
            expiration = System.currentTimeMillis() + mRequest.cacheMaxAge;
        }
        if (expiration <= 0) {
            expiration = Long.MAX_VALUE;
        }

        return expiration;
    }

    private long getLastModified() {

        return mConnection == null ? System.currentTimeMillis() : mConnection.getHeaderFieldDate("Last-Modified", System.currentTimeMillis());
    }

    private String getETag() {

        return mConnection == null ? null : mConnection.getHeaderField("ETag");
    }

    private Map<String, List<String>> getResponseHeaders() {

        return mConnection == null ? null : mConnection.getHeaderFields();
    }

    private int getResponseCode() throws IOException {

        return mConnection == null ? -1 : mConnection.getResponseCode();
    }

    private String getResponseMessage() throws IOException {

        return mConnection == null ? null : URLDecoder.decode(mConnection.getResponseMessage(), mRequest.charset);
    }

}
