/*
 * Copyright (c) 2014 by EagleXad
 * Team: EagleXad
 * Create: 2014-08-29
 */

package com.eaglexad.lib.http.entry;

import com.eaglexad.lib.http.ExHttp;
import com.eaglexad.lib.http.body.ExFileBody;
import com.eaglexad.lib.http.body.ExInputStreamBody;
import com.eaglexad.lib.http.body.ExMultipartBody;
import com.eaglexad.lib.http.body.ExStringBody;
import com.eaglexad.lib.http.body.ExUrlEncodedParamsBody;
import com.eaglexad.lib.http.ible.IExCache;
import com.eaglexad.lib.http.ible.IExConvert;
import com.eaglexad.lib.http.ible.IExRedirect;
import com.eaglexad.lib.http.ible.IExRequestBody;
import com.eaglexad.lib.http.tool.ExHttpQueue;
import com.eaglexad.lib.http.tool.ExHttpUtils;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.Proxy;
import java.util.ArrayList;
import java.util.List;

import javax.net.ssl.SSLSocketFactory;

/**
 * @author Aloneter
 * @ClassName: ExRequest
 * @Description:
 */
public class ExRequest implements java.lang.Comparable {

    public static final String DEFAULT_CHARSET = "UTF-8";
    public static final int DEFAULT_WHAT = -1;

    public final String url;

    public final int what;
    public final ExHttp.Method method;
    public final Object remark;
    public final ExHttp.RequestCallback callback;

    public final List<Header> headers;

    public final List<KeyValue> queryStringParams;
    public final List<KeyValue> bodyParams;
    public final List<KeyValue> fileParams;

    public ExHttp.Priority priority;
    public ExHttp.CacheType cacheType;
    public boolean shouldCache;

    public String bodyContent;

    public Class<?> responseClazz;

    public boolean isMultipart;
    public boolean isCancel;
    public boolean isIntercept;

    public int cacheTimeout;
    public int cacheMaxAge;
    public int connectTimeout;
    public Proxy proxy;
    public SSLSocketFactory sslSocketFactory;
    public boolean hostNameVerify;
    public String charset = DEFAULT_CHARSET;

    public String cacheKey = "";
    public String cacheGroupKey = "";
    public String tag = "";

    private IExRequestBody requestBodyIEx;
    private IExRedirect redirectIEx;
    private IExConvert convertIEx;
    private IExCache cacheIEx;

    private ExDownload mDownload;
    private ExHttpQueue mQueue;
    private int mSequence;

    private ExRequest(Builder builder) {

        this.what = builder.what;
        this.url = builder.url;
        this.method = builder.method;
        this.remark = builder.remark;
        this.callback = builder.callback;

        this.headers = builder.headers;
        this.queryStringParams = builder.queryStringParams;
        this.bodyParams = builder.bodyParams;
        this.fileParams = builder.fileParams;

        this.priority = builder.priority;
        this.cacheType = builder.cacheType;
        this.shouldCache = builder.shouldCache;
        this.bodyContent = builder.bodyContent;
        this.responseClazz = builder.responseClazz;
        this.isMultipart = builder.isMultipart;
        this.cacheTimeout = builder.cacheTimeout;
        this.cacheMaxAge = builder.cacheMaxAge;
        this.connectTimeout = builder.connectTimeout;
        this.proxy = builder.proxy;
        this.sslSocketFactory = builder.sslSocketFactory;
        this.hostNameVerify = builder.hostNameVerify;
        this.charset = builder.charset;

        this.redirectIEx = builder.redirectIEx;
        this.convertIEx = builder.convertIEx;
        this.cacheIEx = builder.cacheIEx;
    }

    public void cancel() {

        isCancel = true;
    }

    public void finish() {

        if (mQueue != null) {
            mQueue.finish(this);
        }
    }

    public void setQueue(ExHttpQueue queue) {

        mQueue = queue;
    }

    public void setSequence(int sequence) {

        mSequence = sequence;
    }

    public int getSequence() {

        return mSequence;
    }

    public void setDownload(ExDownload download) {

        mDownload = download;
    }

    public ExDownload getDownload() {

        return mDownload;
    }

    public String getCacheKey() {

        if (ExHttpUtils.isEmpty(cacheKey)) {
            cacheKey = url;
        }

        return ExHttpUtils.getMD5(cacheKey);
    }

    public String getCacheGroupKey() {

        if (ExHttpUtils.isEmpty(cacheGroupKey)) {
            cacheGroupKey = url;
        }

        return ExHttpUtils.getMD5(cacheGroupKey);
    }

    public IExRedirect getRedirectIEx() {

        return redirectIEx;
    }

    public IExConvert getConvertIEx() {

        return convertIEx;
    }

    public IExCache getCacheIEx() {

        return cacheIEx;
    }

    public IExRequestBody getRequestBody() throws IOException {

        if (!ExHttpUtils.isEmpty(requestBodyIEx)) {
            return requestBodyIEx;
        }
        if (!ExHttpUtils.isEmpty(bodyContent)) {
            requestBodyIEx = new ExStringBody(bodyContent, charset);

            return requestBodyIEx;
        }
        if (!ExHttpUtils.isEmpty(fileParams)) {
            if (!isMultipart && fileParams.size() == 1) {
                KeyValue item = fileParams.get(0);
                Object value = item.value;

                if (value instanceof File) {
                    requestBodyIEx = new ExFileBody((File) value);
                }
                if (value instanceof InputStream) {
                    requestBodyIEx = new ExInputStreamBody((InputStream) value);
                }
                if (value instanceof byte[]) {
                    requestBodyIEx = new ExInputStreamBody(new ByteArrayInputStream((byte[]) value));
                }
                if (value instanceof String) {
                    requestBodyIEx = new ExStringBody(value.toString(), charset);
                }
            } else {
                requestBodyIEx = new ExMultipartBody(fileParams, charset);
            }

            return requestBodyIEx;
        }
        if (!ExHttpUtils.isEmpty(bodyParams)) {
            requestBodyIEx = new ExUrlEncodedParamsBody(bodyParams, charset);

            return requestBodyIEx;
        }

        return requestBodyIEx;
    }

    @Override
    public int compareTo(Object another) {

        if (another instanceof ExRequest) {
            ExRequest other = (ExRequest) another;

            return priority.compareTo(other.priority);
        }

        return 0;
    }

    public final static class Header extends KeyValue {

        public final boolean setHeader;

        public Header(String key, String value, boolean setHeader) {
            super(key, value);

            this.setHeader = setHeader;
        }
    }

    public static class KeyValue {

        public final String key;
        public final Object value;

        public KeyValue(String key, Object value) {

            this.key = key;
            this.value = value;
        }

        public String getValueStr() {

            return value == null ? null : value.toString();
        }

        @Override
        public boolean equals(Object o) {

            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }

            KeyValue keyValue = (KeyValue) o;

            return key == null ? keyValue.key == null : key.equals(keyValue.key);
        }

        @Override
        public int hashCode() {

            return key != null ? key.hashCode() : 0;
        }

    }

    public static class Builder {

        public int what;
        public String url;
        public ExHttp.Method method;
        public Object remark;
        public ExHttp.RequestCallback callback;

        public List<Header> headers;

        public List<KeyValue> queryStringParams;
        public List<KeyValue> bodyParams;
        public List<KeyValue> fileParams;

        public ExHttp.Priority priority;
        public ExHttp.CacheType cacheType;
        public boolean shouldCache;

        public String bodyContent;

        public Class<?> responseClazz;

        public boolean isMultipart;

        public int cacheTimeout;
        public int cacheMaxAge;
        public int connectTimeout;
        public Proxy proxy;
        public SSLSocketFactory sslSocketFactory;
        public boolean hostNameVerify;
        public String charset = DEFAULT_CHARSET;

        public IExRedirect redirectIEx;
        public IExConvert convertIEx;
        public IExCache cacheIEx;

        public Builder(int what, String url, ExHttp.Method method, Object remark, ExHttp.RequestCallback callback) {
            this.what = what;
            this.url = url;
            this.method = method;
            this.remark = remark;
            this.callback = callback;

            headers = new ArrayList<>();

            queryStringParams = new ArrayList<>();
            bodyParams = new ArrayList<>();
            fileParams = new ArrayList<>();
        }

        public Builder setPriority(ExHttp.Priority priority) {

            this.priority = priority;

            return this;
        }

        public Builder setCacheType(ExHttp.CacheType cacheType) {

            this.cacheType = cacheType;

            if (cacheType != ExHttp.CacheType.NONE) {
                shouldCache = true;
            }

            return this;
        }

        public Builder setBodyContent(String bodyContent) {

            this.bodyContent = bodyContent;

            return this;
        }

        public Builder setResponseClazz(Class<?> responseClazz) {

            this.responseClazz = responseClazz;

            return this;
        }

        public Builder setIsMultipart(boolean isMultipart) {

            this.isMultipart = isMultipart;

            return this;
        }

        public Builder setCacheTimeout(int cacheTimeout) {

            this.cacheTimeout = cacheTimeout;

            return this;
        }

        public Builder setConnectTimeout(int connectTimeout) {

            this.connectTimeout = connectTimeout;

            return this;
        }

        public Builder setCacheMaxAge(int cacheMaxAge) {

            this.cacheMaxAge = cacheMaxAge;

            return this;
        }

        public Builder setProxy(Proxy proxy) {

            this.proxy = proxy;

            return this;
        }

        public Builder setSSLSocketFactory(SSLSocketFactory sslSocketFactory) {

            this.sslSocketFactory = sslSocketFactory;

            return this;
        }

        public Builder setHostNameVerify(boolean hostNameVerify) {

            this.hostNameVerify = hostNameVerify;

            return this;
        }

        public Builder setCharset(String charset) {

            this.charset = charset;

            return this;
        }

        public Builder setRuleOfRedirect(IExRedirect redirectIEx) {

            this.redirectIEx = redirectIEx;

            return this;
        }

        public Builder setRuleOfConvert(IExConvert convertIEx) {

            this.convertIEx = convertIEx;

            return this;
        }

        public Builder setRuleOfCache(IExCache cacheIEx) {

            this.cacheIEx = cacheIEx;

            return this;
        }

        public Builder addHeader(String key, String value) {

            addHeader(key, value, false);

            return this;
        }

        public Builder addHeader(String key, String value, boolean setHeader) {

            headers.add(new Header(key, value, setHeader));

            return this;
        }

        public Builder addQueryParam(String key, String value) {

            queryStringParams.add(new KeyValue(key, value));

            return this;
        }

        public Builder addBodyParam(String key, Object value) {

            bodyParams.add(new KeyValue(key, value));

            return this;
        }

        public Builder addFileParam(String key, Object value) {

            fileParams.add(new KeyValue(key, value));

            return this;
        }

        public ExRequest build() {

            return new ExRequest(this);
        }
    }

}
