/*
 * Decompiled with CFR 0.152.
 */
package com.diffplug.spotless.biome;

import com.diffplug.spotless.biome.Architecture;
import com.diffplug.spotless.biome.BiomeFlavor;
import com.diffplug.spotless.biome.OS;
import com.diffplug.spotless.biome.Platform;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigInteger;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class BiomeExecutableDownloader {
    private static final Logger logger = LoggerFactory.getLogger(BiomeExecutableDownloader.class);
    private static final String CHECKSUM_ALGORITHM = "SHA-256";
    private static final String PLATFORM_PATTERN = "%s-%s%s";
    private static final OpenOption[] READ_OPTIONS = new OpenOption[]{StandardOpenOption.READ};
    private static final OpenOption[] WRITE_OPTIONS = new OpenOption[]{StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE};
    private final Path downloadDir;
    private final BiomeFlavor flavor;

    public BiomeExecutableDownloader(BiomeFlavor flavor, Path downloadDir) {
        this.flavor = flavor;
        this.downloadDir = downloadDir;
    }

    public Path download(String version) throws IOException, InterruptedException {
        Platform platform = Platform.guess();
        String url = this.getDownloadUrl(version, platform);
        Path executablePath = this.getExecutablePath(version, platform);
        Path checksumPath = this.getChecksumPath(executablePath);
        Path executableDir = executablePath.getParent();
        if (executableDir != null) {
            Files.createDirectories(executableDir, new FileAttribute[0]);
        }
        logger.info("Attempting to download Biome from '{}' to '{}'", (Object)url, (Object)executablePath);
        HttpRequest request = HttpRequest.newBuilder(URI.create(url)).GET().build();
        HttpResponse.BodyHandler<Path> handler = HttpResponse.BodyHandlers.ofFile(executablePath, WRITE_OPTIONS);
        HttpResponse<Path> response = HttpClient.newBuilder().followRedirects(HttpClient.Redirect.NORMAL).build().send(request, handler);
        if (response.statusCode() != 200) {
            throw new IOException("Failed to download file from " + url + ", server returned " + response.statusCode());
        }
        Path downloadedFile = response.body();
        if (!Files.exists(downloadedFile, new LinkOption[0]) || Files.size(downloadedFile) == 0L) {
            throw new IOException("Failed to download file from " + url + ", file is empty or does not exist");
        }
        this.writeChecksumFile(downloadedFile, checksumPath);
        logger.debug("Biome was downloaded successfully to '{}'", (Object)downloadedFile);
        return downloadedFile;
    }

    public Path ensureDownloaded(String version) throws IOException, InterruptedException {
        Platform platform = Platform.guess();
        logger.debug("Ensuring that Biome for platform '{}' is downloaded", (Object)platform);
        Optional<Path> existing = this.findDownloaded(version);
        if (existing.isPresent()) {
            logger.debug("Biome was already downloaded, using executable at '{}'", (Object)existing.get());
            return existing.get();
        }
        logger.debug("Biome was not yet downloaded, attempting to download executable");
        return this.download(version);
    }

    public Optional<Path> findDownloaded(String version) throws IOException {
        Platform platform = Platform.guess();
        Path executablePath = this.getExecutablePath(version, platform);
        logger.debug("Checking Biome executable at {}", (Object)executablePath);
        return this.checkFileWithChecksum(executablePath) ? Optional.ofNullable(executablePath) : Optional.empty();
    }

    private boolean checkFileWithChecksum(Path filePath) {
        if (!Files.exists(filePath, new LinkOption[0])) {
            logger.debug("File '{}' does not exist yet", (Object)filePath);
            return false;
        }
        if (Files.isDirectory(filePath, new LinkOption[0])) {
            logger.debug("File '{}' exists, but is a directory", (Object)filePath);
            return false;
        }
        Path checksumPath = this.getChecksumPath(filePath);
        if (!Files.exists(checksumPath, new LinkOption[0])) {
            logger.debug("File '{}' exists, but checksum file '{}' does not", (Object)filePath, (Object)checksumPath);
            return false;
        }
        if (Files.isDirectory(checksumPath, new LinkOption[0])) {
            logger.debug("Checksum file '{}' exists, but is a directory", (Object)checksumPath);
            return false;
        }
        try {
            String actualChecksum = this.computeChecksum(filePath, CHECKSUM_ALGORITHM);
            String expectedChecksum = this.readTextFile(checksumPath, StandardCharsets.ISO_8859_1);
            logger.debug("Expected checksum: {}, actual checksum: {}", (Object)expectedChecksum, (Object)actualChecksum);
            return Objects.equals(expectedChecksum, actualChecksum);
        }
        catch (IOException ignored) {
            return false;
        }
    }

    private String computeChecksum(Path file, String algorithm) throws IOException {
        String string;
        block9: {
            byte[] buffer = new byte[4192];
            InputStream in = Files.newInputStream(file, READ_OPTIONS);
            try {
                int result;
                MessageDigest digest = MessageDigest.getInstance(algorithm);
                while ((result = in.read(buffer, 0, buffer.length)) != -1) {
                    digest.update(buffer, 0, result);
                }
                byte[] bytes = digest.digest();
                string = String.format("%0" + bytes.length * 2 + "X", new BigInteger(1, bytes));
                if (in == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (in != null) {
                        try {
                            in.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (NoSuchAlgorithmException e) {
                    throw new RuntimeException(e);
                }
            }
            in.close();
        }
        return string;
    }

    private String getArchitectureCodeName(Architecture architecture) throws IOException {
        switch (architecture) {
            case ARM64: {
                return "arm64";
            }
            case X64: {
                return "x64";
            }
        }
        throw new IOException("Unsupported architecture: " + architecture);
    }

    private Path getChecksumPath(Path file) {
        Path parent = file.getParent();
        Path base = parent != null ? parent : file;
        Path fileName = file.getFileName();
        String checksumName = fileName != null ? fileName.toString() + ".sha256" : "checksum.sha256";
        return base.resolve(checksumName);
    }

    private String getDownloadUrl(String version, Platform platform) throws IOException {
        String osCodeName = this.getOsCodeName(platform.getOs());
        String architectureCodeName = this.getArchitectureCodeName(platform.getArchitecture());
        String extension = this.getDownloadUrlExtension(platform.getOs());
        String platformString = String.format(PLATFORM_PATTERN, osCodeName, architectureCodeName, extension);
        return String.format(this.flavor.getUrlPattern(), version, platformString);
    }

    private String getDownloadUrlExtension(OS os) throws IOException {
        switch (os) {
            case LINUX: {
                return "";
            }
            case MAC_OS: {
                return "";
            }
            case WINDOWS: {
                return ".exe";
            }
        }
        throw new IOException("Unsupported OS: " + os);
    }

    private Path getExecutablePath(String version, Platform platform) {
        String os = platform.getOs().name().toLowerCase(Locale.ROOT);
        String arch = platform.getArchitecture().name().toLowerCase(Locale.ROOT);
        String fileName = String.format(this.flavor.getDownloadFilePattern(), os, arch, version);
        return this.downloadDir.resolve(fileName);
    }

    private String getOsCodeName(OS os) throws IOException {
        switch (os) {
            case LINUX: {
                return "linux";
            }
            case MAC_OS: {
                return "darwin";
            }
            case WINDOWS: {
                return "win32";
            }
        }
        throw new IOException("Unsupported OS: " + os);
    }

    private String readTextFile(Path file, Charset charset) throws IOException {
        try (InputStream in = Files.newInputStream(file, READ_OPTIONS);){
            String string = new String(in.readAllBytes(), charset);
            return string;
        }
    }

    private void writeChecksumFile(Path file, Path checksumPath) throws IOException {
        String checksum = this.computeChecksum(file, CHECKSUM_ALGORITHM);
        try (OutputStream out = Files.newOutputStream(checksumPath, WRITE_OPTIONS);){
            out.write(checksum.getBytes(StandardCharsets.ISO_8859_1));
        }
    }
}

