package com.devicenative.dna.ads;

import static android.content.pm.LauncherApps.ShortcutQuery.FLAG_MATCH_DYNAMIC;
import static android.content.pm.LauncherApps.ShortcutQuery.FLAG_MATCH_MANIFEST;
import static android.content.pm.LauncherApps.ShortcutQuery.FLAG_MATCH_PINNED;

import android.content.ActivityNotFoundException;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.LauncherActivityInfo;
import android.content.pm.LauncherApps;
import android.content.pm.ShortcutInfo;
import android.net.Uri;
import android.os.Build;
import android.os.Process;
import android.os.UserHandle;


import com.devicenative.dna.DeviceNativeClickHandler;
import com.devicenative.dna.utils.DNAConstants;
import com.devicenative.dna.utils.DNALogger;
import com.devicenative.dna.utils.DNAPreferences;
import com.devicenative.dna.utils.DNAStatsLogger;
import com.devicenative.dna.utils.DNAUtils;

import org.json.JSONObject;

import java.lang.reflect.Method;
import java.net.HttpURLConnection;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Collections;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class DNAdLinkHandler {
    DNALinkResultCacheManager linkResultCacheManager_;

    public DNAdLinkHandler(int adCacheTimeLimitMs) {
        linkResultCacheManager_ = new DNALinkResultCacheManager(adCacheTimeLimitMs);
    }

    public void asyncLinkProcessor(Context context, String urlString, String adId, String adAppName, String adPackageName, String clickId, String productSource, DeviceNativeClickHandler clickHandler) {
        // Follow all redirects and close the connection. Do nothing with the response. Do this on a background thread.
        DNAPreferences prefs = DNAPreferences.getInstance(context);
        long startTime = System.currentTimeMillis();

        final String completedUrlString = completeMacrosInURL(prefs, urlString, clickId, productSource);

        HttpURLConnection connection = null;
        try {
            URL url = new URL(completedUrlString);

            DNALogger.i("about to start link request sync: " + completedUrlString);

            connection = (HttpURLConnection) url.openConnection();
            // Optional: set timeout, request method, etc.

            String userAgent = prefs.getUserAgent(context);
            if (userAgent != null && !userAgent.isEmpty()) {
                connection.setRequestProperty("User-Agent", userAgent);
            } else {
                connection.setRequestProperty("User-Agent", System.getProperty("http.agent"));
            }

            connection.setRequestMethod("GET");
            connection.setInstanceFollowRedirects(true);
            connection.setConnectTimeout(30000);
            connection.setReadTimeout(30000);

            // Just connecting to the URL, not processing the result
            connection.connect();
            int responseCode = connection.getResponseCode();

            // if response code != 40X or 500, then we're good
            if (responseCode != HttpURLConnection.HTTP_BAD_REQUEST &&
                    responseCode != HttpURLConnection.HTTP_INTERNAL_ERROR &&
                    responseCode != HttpURLConnection.HTTP_NOT_FOUND) {
                if (clickHandler != null) {
                    clickHandler.onClickServerCompleted();
                }
                JSONObject statsMetadata = new JSONObject();
                try {
                    statsMetadata.put("sId", prefs.getSessionID());
                    statsMetadata.put("url", urlString);
                    statsMetadata.put("clickId", clickId);
                    statsMetadata.put("adId", adId);
                    statsMetadata.put("adAppName", adAppName);
                    statsMetadata.put("adPackageName", adPackageName);
                } catch (Exception je) {
                    DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
                }
                long endTime = System.currentTimeMillis();
                DNAStatsLogger.logInternal(context, "Async Link Fetch", (int) (endTime - startTime), statsMetadata);
            }
        } catch (Exception e) {
            if (clickHandler != null) {
                clickHandler.onFailure(0, e.getMessage());
            }
            DNALogger.e("DeviceNativeAds: Error in asyncLinkProcessor: " + e.getMessage());
            JSONObject statsMetadata = new JSONObject();
            try {
                statsMetadata.put("sId", prefs.getSessionID());
                statsMetadata.put("loc", "asyncLinkProcessor");
                statsMetadata.put("url", urlString);
                statsMetadata.put("clickId", clickId);
                statsMetadata.put("adId", adId);
                statsMetadata.put("adAppName", adAppName);
                statsMetadata.put("adPackageName", adPackageName);
                statsMetadata.put("m", e.getMessage());
            } catch (Exception je) {
                DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
            }
            long endTime = System.currentTimeMillis();
            DNAStatsLogger.logInternal(context, "Error", (int) (endTime - startTime), statsMetadata);
        } finally {
            if (connection != null) {
                connection.disconnect();
            }
        }

    }

    public void linkRouter(Context context, String urlString, String destinationUrl, String adId, String adAppName, String adPackageName, String component, String userId, String clickId, String productSource, boolean doRoute, boolean earlyOpenPlayStore, String destinationUrlOverride, DeviceNativeClickHandler clickHandler) {
        DNAPreferences prefs = DNAPreferences.getInstance(context);
        if (linkResultCacheManager_.isRedirectionActive()) {
            if (clickHandler != null && doRoute) {
                clickHandler.onFailure(0, "Redirection already in progress");
            }
            return;
        }
        if (doRoute) {
            linkResultCacheManager_.setRedirectionActive(true);
        }
        long startTime = System.currentTimeMillis();
        boolean destinationUrlIsOverridden = destinationUrlOverride != null && !destinationUrlOverride.isEmpty();
        boolean didRouteAlready = false;

        if (doRoute) {
            DNALinkResultCacheManager.CacheLinkResult cacheItem = linkResultCacheManager_.getLinkResultCache(adId);
            // If link is cached but we're not going to override the destination URL with early open or dev provided URL
            if (cacheItem != null && !earlyOpenPlayStore & !destinationUrlIsOverridden) {
                // use cache result
                DNALogger.i("Using cached link result for ad: " + adPackageName);
                JSONObject statsMetadata = cacheItem.getLastStatsMetadata();
                try {
                    if (clickHandler != null) {
                        clickHandler.onClickRouterCompleted(true);
                    }

                    triggerLinkRoute(context, adPackageName, component, userId, cacheItem.getFinalUrl());
                    didRouteAlready = true;

                    if (clickHandler != null) {
                        clickHandler.onClickRouterCompleted(true);
                    }
                    long endTime = System.currentTimeMillis();
                    statsMetadata.put("cached", true);
                    DNAStatsLogger.logInternal(context, "Link Route", (int) (endTime - startTime), statsMetadata);
                    linkResultCacheManager_.setRedirectionActive(false);
                } catch (Exception e) {
                    fallbackLinkRouter(context, adPackageName, component, adId, adAppName, userId, clickHandler);

                    long endTime = System.currentTimeMillis();
                    try {
                        statsMetadata.put("loc", "linkRouter");
                        statsMetadata.put("cached", true);
                        statsMetadata.put("m", e.getMessage());
                    } catch (Exception je) {
                        DNALogger.e("DeviceNativeAds: Unable to edit stats metadata");
                    }
                    DNAStatsLogger.logInternal(context, "Error", (int) (endTime - startTime), statsMetadata);

                    linkResultCacheManager_.setRedirectionActive(false);
                }
                return;
            // If we're going to early open the play store, we can fire that before we process the tracking link
            } else if (earlyOpenPlayStore) {
                // force open Play Store early and process ad link behind the scenes
                String playStoreUrl = "market://details?id=" + adPackageName;
                if (destinationUrlIsOverridden) {
                    playStoreUrl = destinationUrlOverride;
                }
                DNALogger.i("Early opening destination store for ad: " + adPackageName + " with URL: " + playStoreUrl);
                try {
                    triggerLinkRoute(context, adPackageName, component, userId, playStoreUrl);
                } catch (Exception ex) {
                    DNALogger.e("DeviceNativeAds: Error early opening Play Store: " + ex.getMessage());
                    fallbackLinkRouter(context, adPackageName, component, adId, adAppName, userId, clickHandler);
                }
                didRouteAlready = true;
            }
        }

        final String completedUrlString = completeMacrosInURL(prefs, urlString, clickId, productSource);
        HttpURLConnection connection = null;
        String currentUrl = completedUrlString;
        boolean redirect = true;
        int redirectCount = 0;
        int status = 0;

        try {
            do {
                DNALogger.i("about to start link request sync: " + currentUrl);
                URL url = new URL(currentUrl);
                connection = (HttpURLConnection) url.openConnection();

                String userAgent = prefs.getUserAgent(context);
                if (userAgent != null && !userAgent.isEmpty()) {
                    connection.setRequestProperty("User-Agent", userAgent);
                } else {
                    connection.setRequestProperty("User-Agent", System.getProperty("http.agent"));
                }

                connection.setConnectTimeout(prefs.getConnectTimeout());
                connection.setReadTimeout(prefs.getTimeout());
                connection.setInstanceFollowRedirects(false);
                connection.connect();
                status = connection.getResponseCode();

                DNALogger.i("link request sync response code: " + status);
                if (status != HttpURLConnection.HTTP_OK) {
                    if (status == HttpURLConnection.HTTP_MOVED_TEMP
                            || status == HttpURLConnection.HTTP_MOVED_PERM
                            || status == HttpURLConnection.HTTP_SEE_OTHER
                            || status == 307
                            || status == 308) {
                        redirectCount++;

                        String redirectUrl = connection.getHeaderField("Location");
                        if (redirectUrl.startsWith("/")) {
                            currentUrl = url.getProtocol() + "://" + url.getHost() + redirectUrl;
                        } else if (redirectUrl.startsWith("http://")) {
                            currentUrl = redirectUrl.replace("http://", "https://");
                        } else {
                            currentUrl = redirectUrl;
                        }

                        DNALogger.i("link request sync redirect url: " + currentUrl);
                    } else if (status == HttpURLConnection.HTTP_BAD_REQUEST ||
                            status == HttpURLConnection.HTTP_INTERNAL_ERROR ||
                            status == HttpURLConnection.HTTP_NOT_FOUND) {
                        // Non-successful, non-redirect status code
                        redirect = false;
                        if (clickHandler != null && doRoute) {
                            clickHandler.onFailure(status, "Non-successful, non-redirect status code");
                        }
                    }
                }

                String newHostname = "";
                if (currentUrl.startsWith("http")) {
                    URL newUrl = new URL(currentUrl);
                    newHostname = newUrl.getHost();
                }

                String destinationUrlRootHost = destinationUrl;
                if (destinationUrlRootHost != null && !destinationUrlRootHost.isEmpty()) {
                    destinationUrlRootHost = getRootDomain(destinationUrl);
                }

                // if we're about to do another redirect, and the current URL is to the Play Store or is a non HTTP URL, then launch the intent and stop processing
                if ((redirect && (currentUrl.startsWith("market://") ||
                        currentUrl.contains("play.google.com") ||
                        (destinationUrlRootHost != null && !destinationUrlRootHost.isEmpty() && !newHostname.isEmpty() && newHostname.contains(destinationUrlRootHost)) ||
                        !currentUrl.startsWith("http"))) ||
                        status == HttpURLConnection.HTTP_OK) {
                    DNALogger.i("Hit terminal URL at " + currentUrl);
                    if (clickHandler != null && doRoute) {
                        clickHandler.onClickServerCompleted();
                    }

                    if (doRoute && !didRouteAlready) {
                        triggerLinkRoute(context, adPackageName, component, userId, currentUrl);
                    }

                    if (clickHandler != null && doRoute) {
                        clickHandler.onClickRouterCompleted(true);
                    }
                    redirect = false;

                    JSONObject statsMetadata = new JSONObject();
                    try {
                        statsMetadata.put("sId", prefs.getSessionID());
                        statsMetadata.put("url", urlString);
                        statsMetadata.put("lastUrl", currentUrl);
                        statsMetadata.put("redirectCount", redirectCount);
                        statsMetadata.put("doRoute", doRoute);
                        statsMetadata.put("clickId", clickId);
                        statsMetadata.put("cached", false);
                        statsMetadata.put("adId", adId);
                        statsMetadata.put("adAppName", adAppName);
                        statsMetadata.put("adPackageName", adPackageName);
                    } catch (Exception je) {
                        DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
                    }
                    long endTime = System.currentTimeMillis();
                    DNAStatsLogger.logInternal(context, "Link Route", (int) (endTime - startTime), statsMetadata);

                    linkResultCacheManager_.setLinkResultCache(adId, currentUrl, statsMetadata);
                }
            } while (redirect);

            linkResultCacheManager_.setRedirectionActive(false);
        } catch (Exception e) {
            DNALogger.e("DeviceNativeAds: Error in linkRouter: " + e.getMessage());
            JSONObject statsMetadata = new JSONObject();
            try {
                statsMetadata.put("sId", prefs.getSessionID());
                statsMetadata.put("loc", "linkRouter");
                statsMetadata.put("url", urlString);
                statsMetadata.put("clickId", clickId);
                statsMetadata.put("redirectCount", redirectCount);
                statsMetadata.put("cached", false);
                statsMetadata.put("doRoute", doRoute);
                statsMetadata.put("adId", adId);
                statsMetadata.put("adAppName", adAppName);
                statsMetadata.put("adPackageName", adPackageName);
                statsMetadata.put("m", e.getMessage());
            } catch (Exception je) {
                DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
            }
            long endTime = System.currentTimeMillis();
            DNAStatsLogger.logInternal(context, "Error", (int) (endTime - startTime), statsMetadata);

            // Trigger the fallback link router
            if (doRoute && !earlyOpenPlayStore) {
                fallbackLinkRouter(context, adPackageName, component, adId, adAppName, userId, clickHandler);
            }

            linkResultCacheManager_.setRedirectionActive(false);
        } finally {
            if (connection != null) {
                connection.disconnect();
            }
        }
    }

    private void triggerLinkRoute(Context context, String adPackageName, String component, String userId, String finalUrl) throws Exception {
        DNAPreferences prefs = DNAPreferences.getInstance(context);
        DNALogger.i("trying to launch intent: " + finalUrl);
        boolean isMarketUrl = false;
        if (finalUrl.startsWith("market://") || finalUrl.contains("play.google.com")) {
            isMarketUrl = true;
        }

        if (prefs.getReplaceGooglePlayUrl()) {
            String googlePlayUrlReplacement = prefs.getGooglePlayUrlReplacement();
            if (googlePlayUrlReplacement != null &&
                    googlePlayUrlReplacement.contains(DNAConstants.UrlMacros.APP_PACKAGE_NAME)) {

                // Check if the current URL is a Google Play URL
                if (isMarketUrl) {
                    try {
                        Uri currentUri = Uri.parse(finalUrl);
                        String packageName = currentUri.getQueryParameter("id");

                        if (packageName != null && !packageName.isEmpty()) {
                            // Replace macro with the package name
                            String remappedBaseUrl = googlePlayUrlReplacement
                                    .replace(DNAConstants.UrlMacros.APP_PACKAGE_NAME, packageName);

                            Uri.Builder remappedUriBuilder = Uri.parse(remappedBaseUrl).buildUpon();

                            // Copy all query parameters from the original URL
                            for (String paramName : currentUri.getQueryParameterNames()) {
                                String paramValue = currentUri.getQueryParameter(paramName);
                                if (paramValue != null) {
                                    remappedUriBuilder.appendQueryParameter(paramName, paramValue);
                                }
                            }

                            Uri remappedUri = remappedUriBuilder.build();

                            // Ensure the remapped URL is absolute
                            if (remappedUri.isAbsolute()) {
                                finalUrl = remappedUri.toString();
                                DNALogger.i("Successfully remapped Google Play URL: " + finalUrl);
                            } else {
                                DNALogger.e("Remapped URL is not absolute: " + remappedUri.toString());
                            }
                        } else {
                            DNALogger.w("No package name found in the original URL: " + finalUrl);
                        }
                    } catch (Exception e) {
                        DNALogger.e("Error remapping Google Play URL: " + e.getMessage());
                    }
                }
            }
        }

        if (isMarketUrl && prefs.getAutoDeepLinkEnabled()) {
            // check if app is installed, and send the user there instead of the Play Store
            boolean shouldFallback = false;
            LauncherApps la = context.getSystemService(LauncherApps.class);
            if (la == null) {
                shouldFallback = true;
            }

            UserHandle profile;
            if (userId != null && !userId.isEmpty()) {
                profile  = DNAUtils.getUserHandleFromFormattedString(userId);
            } else {
                profile      = Process.myUserHandle();
            }

            if (!shouldFallback && !la.isPackageEnabled(adPackageName, profile)) {
                shouldFallback = true;
            }

            List<LauncherActivityInfo> launcherActivities = null;
            try {
                if (!shouldFallback) {
                    launcherActivities = la.getActivityList(adPackageName, profile);

                    if (launcherActivities == null || launcherActivities.isEmpty()) {
                        shouldFallback = true;
                    }
                }
            } catch (Exception e) {
                shouldFallback = true;
            }

            if (!shouldFallback) {
                ApplicationInfo appInfo = launcherActivities.get(0).getApplicationInfo();
                if (DNAUtils.isArchived(appInfo)) {
                    shouldFallback = true;
                }
            }

            if (!shouldFallback) {
                if (component != null && !component.isEmpty()) {
                    la.startMainActivity(
                            new ComponentName(adPackageName, component),
                            profile,
                            null,
                            null
                    );
                } else {
                    la.startMainActivity(
                            launcherActivities.get(0).getComponentName(),
                            profile,
                            null,
                            null
                    );
                }
            } else {
                Intent intent = new Intent(Intent.ACTION_VIEW, Uri.parse(finalUrl));
                intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                context.startActivity(intent);
            }
        } else {
            Intent intent;
            if (finalUrl.startsWith("intent://")) {
                intent = Intent.parseUri(finalUrl, Intent.URI_INTENT_SCHEME);

                intent.addCategory(Intent.CATEGORY_BROWSABLE);
                intent.setSelector(null);
                intent.setComponent(null);

                // If there's no browser to handle this intent scheme directly, fallback to default browser URL
                if (intent.getData() != null) {
                    Intent fallbackIntent = new Intent(Intent.ACTION_VIEW, intent.getData());
                    fallbackIntent.addCategory(Intent.CATEGORY_BROWSABLE);
                    fallbackIntent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                    context.startActivity(fallbackIntent);
                } else {
                    intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                    context.startActivity(intent);
                }

            } else {
                // Non-intent URLs, handle normally.
                intent = new Intent(Intent.ACTION_VIEW, Uri.parse(finalUrl));
                intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                context.startActivity(intent);
            }
        }
    }

    public void settingShortcutRouter(Context context, String intentUri, String packageName, DeviceNativeClickHandler clickHandler) {
        DNAPreferences prefs = DNAPreferences.getInstance(context);
        // check if package is installed and open app directly if so, otherwise open Play Store
        try {
            Intent intent = new Intent(intentUri);
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
            context.startActivity(intent);
            if (clickHandler != null) {
                clickHandler.onClickRouterCompleted(true);
            }
        } catch (Exception e) {
            Intent intent = context.getPackageManager().getLaunchIntentForPackage(packageName);
            if (intent != null) {
                intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                context.startActivity(intent);
                if (clickHandler != null) {
                    clickHandler.onClickRouterCompleted(true);
                }
                return;
            }
            if (clickHandler != null) {
                clickHandler.onFailure(0, e.getMessage());
            }
            JSONObject statsMetadata = new JSONObject();
            try {
                statsMetadata.put("sId", prefs.getSessionID());
                statsMetadata.put("loc", "settingShortcutRouter");
                statsMetadata.put("url", packageName);
                statsMetadata.put("appName", "Settings");
                statsMetadata.put("packageName", packageName);
                statsMetadata.put("m", e.getMessage());
            } catch (Exception je) {
                DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
            }
            DNAStatsLogger.logInternal(context, "Error", null, statsMetadata);
        }
    }

    public void shortcutRouter(Context context, String intentUri, String appName, String packageName, DeviceNativeClickHandler clickHandler) {
        long startTime = System.currentTimeMillis();
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.O) {
            fallbackLinkRouter(context, packageName, null, "shortcut", appName, null, clickHandler);
            return;
        }
        try {
            final LauncherApps launcherApps = (LauncherApps) context.getSystemService(Context.LAUNCHER_APPS_SERVICE);

            if (!launcherApps.hasShortcutHostPermission()) {
                fallbackLinkRouter(context, packageName, null, "shortcut", appName, null, clickHandler);
                return;
            }

            LauncherApps.ShortcutQuery query = new LauncherApps.ShortcutQuery();
            query.setPackage(packageName);
            query.setShortcutIds(Collections.singletonList(intentUri));
            query.setQueryFlags(FLAG_MATCH_DYNAMIC | FLAG_MATCH_MANIFEST | FLAG_MATCH_PINNED);

            List<ShortcutInfo> shortcuts = launcherApps.getShortcuts(query, android.os.Process.myUserHandle());
            if (shortcuts != null) {
                for (ShortcutInfo shortcut : shortcuts) {
                    if (shortcut.isEnabled()) {
                        launcherApps.startShortcut(shortcut, null, null);
                        if (clickHandler != null) {
                            clickHandler.onClickRouterCompleted(true);
                        }
                        return;
                    }
                }
            }
            fallbackLinkRouter(context, packageName, null, "shortcut", appName, null, clickHandler);
        } catch (Exception e) {
            if (clickHandler != null) {
                clickHandler.onFailure(0, e.getMessage());
            }
            DNALogger.e("DeviceNativeAds: Error in shortcutRouter: " + e.getMessage());
            DNAPreferences prefs = DNAPreferences.getInstance(context);
            JSONObject statsMetadata = new JSONObject();
            try {
                statsMetadata.put("sId", prefs.getSessionID());
                statsMetadata.put("loc", "shortcutRouter");
                statsMetadata.put("url", intentUri);
                statsMetadata.put("appName", appName);
                statsMetadata.put("packageName", packageName);
                statsMetadata.put("m", e.getMessage());
            } catch (Exception je) {
                DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
            }
            long endTime = System.currentTimeMillis();
            DNAStatsLogger.logInternal(context, "Error", (int) (endTime - startTime), statsMetadata);
        }
    }

    public void fallbackLinkRouter(Context context, String packageName, String component, String adId, String adAppName, String userId, DeviceNativeClickHandler clickHandler) {
        DNAPreferences prefs = DNAPreferences.getInstance(context);
        // check if package is installed and open app directly if so, otherwise open Play Store
        try {
            UserHandle userHandle = getUserHandle(userId);
            if (userHandle != null) {
                LauncherApps launcherApps = (LauncherApps) context.getSystemService(Context.LAUNCHER_APPS_SERVICE);

                if (launcherApps != null) {
                    List<LauncherActivityInfo> activities = launcherApps.getActivityList(packageName, userHandle);
                    if (!activities.isEmpty()) {
                        ApplicationInfo appInfo = activities.get(0).getApplicationInfo();
                        if (!DNAUtils.isArchived(appInfo)) {
                            if (component != null && !component.isEmpty()) {
                                launcherApps.startMainActivity(
                                        new ComponentName(packageName, component),
                                        userHandle,
                                        null,
                                        null
                                );
                                if (clickHandler != null) {
                                    clickHandler.onClickRouterCompleted(true);
                                }
                                return;
                            } else {
                                launcherApps.startMainActivity(
                                        activities.get(0).getComponentName(),
                                        userHandle,
                                        null,
                                        null
                                );
                                if (clickHandler != null) {
                                    clickHandler.onClickRouterCompleted(true);
                                }
                                return;
                            }
                        }
                    }
                }
            }

            Intent intent = context.getPackageManager().getLaunchIntentForPackage(packageName);
            if (intent != null) {
                intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                context.startActivity(intent);
            } else {
                try {
                    Intent playStoreIntent = new Intent(Intent.ACTION_VIEW, Uri.parse("market://details?id=" + packageName));
                    playStoreIntent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                    context.startActivity(playStoreIntent);
                } catch (ActivityNotFoundException anfe) {
                    Intent playStoreIntent = new Intent(Intent.ACTION_VIEW, Uri.parse("https://play.google.com/store/apps/details?id=" + packageName));
                    playStoreIntent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                    context.startActivity(playStoreIntent);
                }

            }
            if (clickHandler != null) {
                clickHandler.onClickRouterCompleted(true);
            }
        } catch (Exception e) {
            if (clickHandler != null) {
                clickHandler.onFailure(0, e.getMessage());
            }
            JSONObject statsMetadata = new JSONObject();
            try {
                statsMetadata.put("sId", prefs.getSessionID());
                statsMetadata.put("loc", "fallbackLinkRouter");
                statsMetadata.put("url", packageName);
                statsMetadata.put("adId", adId);
                statsMetadata.put("adAppName", adAppName);
                statsMetadata.put("adPackageName", packageName);
                statsMetadata.put("m", e.getMessage());
            } catch (Exception je) {
                DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
            }
            DNAStatsLogger.logInternal(context, "Error", null, statsMetadata);
        }
    }

    private static String completeMacrosInURL(DNAPreferences prefs, String url, String clickId, String productSource) {
        if (prefs.getAdidUsageEnabled() && !prefs.getAdid().isEmpty() && !prefs.getConfigGAIDDisabled()) {
            url = url.replace(DNAConstants.UrlMacros.AD_ID, prefs.getAdid());
        } else {
            url = url.replace(DNAConstants.UrlMacros.AD_ID, "");
        }
        url = url.replace(DNAConstants.UrlMacros.APP_PACKAGE_NAME, prefs.getAppPackageName());
        url = url.replace(DNAConstants.UrlMacros.CLICK_ID, clickId);
        url = url.replace(DNAConstants.UrlMacros.PRODUCT_SOURCE, productSource);
        return url;
    }

    public static UserHandle getUserHandle(String userHandle) {
        int userId = DNAUtils.readPidFromUserHandleString(userHandle);

        UserHandle targetUserHandle = null;
        if (userId != -1) {
            try {
                Class<?> userHandleClass = Class.forName("android.os.UserHandle");
                Method ofMethod = userHandleClass.getMethod("of", int.class);
                targetUserHandle = (UserHandle) ofMethod.invoke(null, userId);
                DNALogger.i("DeviceNativeAds: successfully fetched UserHandle: " + targetUserHandle);
                return targetUserHandle;
            } catch (Exception e) {
                DNALogger.i("DeviceNativeAds: Error getting UserHandle: " + e.getMessage());
            }
        }
        return null;
    }

    public static String getRootDomain(String url) {
        String[] parts = url.split("\\.");
        if (parts.length < 2) {
            return url;
        }
        return parts[parts.length - 2] + "." + parts[parts.length - 1];
    }
}
