package com.devicenative.dna.ads;

import org.json.JSONObject;
import java.util.concurrent.ConcurrentHashMap;

public class DNALinkResultCacheManager {
    private final long adCacheTimeLimitMs;

    private boolean isRedirectionActive = false;
    private final ConcurrentHashMap<String, DNALinkResultCacheManager.CacheLinkResult> linkResultCache = new ConcurrentHashMap<>();

    public DNALinkResultCacheManager(long adCacheTimeLimitMs) {
        this.adCacheTimeLimitMs = adCacheTimeLimitMs;
    }

    public synchronized void setRedirectionActive(boolean isActive) {
        isRedirectionActive = isActive;
    }
    public synchronized boolean isRedirectionActive() {
        return isRedirectionActive;
    }

    public DNALinkResultCacheManager.CacheLinkResult getLinkResultCache(String adId) {
        DNALinkResultCacheManager.CacheLinkResult item = linkResultCache.get(adId);
        if (item != null && item.isExpired()) {
            linkResultCache.remove(adId);
            return null;
        }
        if (item != null && !item.isExpired()) {
            return item;
        }
        return null; // Return null if expired or not found
    }

    public void setLinkResultCache(String adId, String finalUrl, JSONObject lastStatsMetadata) {
        cleanupExpiredItems(linkResultCache);
        DNALinkResultCacheManager.CacheLinkResult item = new DNALinkResultCacheManager.CacheLinkResult(finalUrl, lastStatsMetadata, adCacheTimeLimitMs);
        linkResultCache.put(adId, item);
    }

    private void cleanupExpiredItems(ConcurrentHashMap<String, DNALinkResultCacheManager.CacheLinkResult> cache) {
        cache.entrySet().removeIf(entry -> entry.getValue().isExpired());
    }

    public static class CacheLinkResult {
        private final long cacheTimeLimitMs;
        private final long lastLoadTime;
        private final String finalUrl;
        private JSONObject lastStatsMetadata;

        public CacheLinkResult(String finalUrl, JSONObject lastStatsMetadata, long cacheTimeLimitMs) {
            this.cacheTimeLimitMs = cacheTimeLimitMs;
            this.lastLoadTime = System.currentTimeMillis();
            this.finalUrl = finalUrl;
            this.lastStatsMetadata = lastStatsMetadata;
        }

        public boolean isExpired() {
            return System.currentTimeMillis() - lastLoadTime > this.cacheTimeLimitMs;
        }

        public String getFinalUrl() {
            return finalUrl;
        }

        public JSONObject getLastStatsMetadata() {
            return lastStatsMetadata;
        }
    }
}
