package com.devicenative.dna.ads;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.os.Process;
import android.os.UserHandle;
import android.content.pm.LauncherActivityInfo;
import android.content.pm.LauncherApps;

import com.devicenative.dna.DNAResultItem;
import com.devicenative.dna.utils.DNALogger;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class DNAAppSearchFallback {

    private static final Object CACHE_LOCK = new Object();
    private static List<DNAResultItem> cachedAppResults = null;

    public static void clearCache() {
        synchronized (CACHE_LOCK) {
            if (cachedAppResults != null) {
                cachedAppResults.clear();
            }
            cachedAppResults = null;
        }
    }

    public static List<DNAResultItem> getAppResultsForDisplayFallback(Context context, int count) {
        // Initialize the cache if necessary.
        if (cachedAppResults == null) {
            synchronized (CACHE_LOCK) {
                if (cachedAppResults == null) {
                    cachedAppResults = loadInstalledApps(context);
                }
            }
        }

        List<String> defaultPackages = Arrays.asList(
                "com.android.vending",
                "com.android.chrome",
                "com.android.dialer",
                "com.android.camera",
                "com.android.settings",
                "com.google.android.apps.maps",
                "com.google.android.youtube"
        );

        // Create a list for default apps.
        List<DNAResultItem> defaultApps = new ArrayList<>();

        // First, add apps that match the default packages.
        for (DNAResultItem app : cachedAppResults) {
            String packageName = app.packageName;
            if (defaultPackages.contains(packageName)) {
                defaultApps.add(app);
            }
        }

        if (defaultApps.size() > count) {
            return defaultApps.subList(0, count);
        }

        if (defaultApps.size() < count) {
            for (DNAResultItem app : cachedAppResults) {
                if (defaultApps.size() >= count) break;
                String packageName = app.packageName;
                if (!defaultPackages.contains(packageName)) {
                    defaultApps.add(app);
                }
            }
        }

        return defaultApps;
    }


    public static List<DNAResultItem> getAppResultsForSearchFallback(Context context, String query) {
        if (query == null || query.length() == 0) {
            return new ArrayList<>();
        }

        if (cachedAppResults == null) {
            synchronized (CACHE_LOCK) {
                if (cachedAppResults == null) {
                    cachedAppResults = loadInstalledApps(context);
                }
            }
        }

        // ---- Filter and rank search results using fallback logic ----
        List<String> queryParts = processQueryIntoParts(query);

        List<DNAResultItem> startPrefixMatchedApps = new ArrayList<>();
        List<DNAResultItem> laterPrefixMatchedApps = new ArrayList<>();
        List<DNAResultItem> containsMatchedApps = new ArrayList<>();

        for (DNAResultItem item : cachedAppResults) {
            String appName = item.appName;
            List<String> appNameParts = processAppNameIntoParts(appName);

            boolean matched = false;
            // Exact start-prefix match.
            for (int j = 0; j < appNameParts.size() && !matched; j++) {
                String part = appNameParts.get(j);
                for (String queryPart : queryParts) {
                    if (part.startsWith(queryPart)) {
                        if (j == 0) {
                            startPrefixMatchedApps.add(item);
                        } else {
                            laterPrefixMatchedApps.add(item);
                        }
                        matched = true;
                        break;
                    }
                }
            }

            // Contains match if query length > 2.
            if (!matched && query.length() > 2) {
                for (String part : appNameParts) {
                    for (String queryPart : queryParts) {
                        if (part.contains(queryPart)) {
                            containsMatchedApps.add(item);
                            matched = true;
                            break;
                        }
                    }
                    if (matched) {
                        break;
                    }
                }
            }
        }

        List<DNAResultItem> filteredResults = new ArrayList<>();
        filteredResults.addAll(startPrefixMatchedApps);
        filteredResults.addAll(laterPrefixMatchedApps);
        filteredResults.addAll(containsMatchedApps);

        return filteredResults;
    }

    private static List<DNAResultItem> loadInstalledApps(Context context) {
        List<DNAResultItem> appResults = new ArrayList<>();
        UserHandle currentUser = Process.myUserHandle();
        LauncherApps launcher = (LauncherApps) context.getSystemService(Context.LAUNCHER_APPS_SERVICE);

        if (launcher == null) {
            return appResults;
        }

        for (LauncherActivityInfo activityInfo : launcher.getActivityList(null, currentUser)) {
            if (activityInfo == null) {
                continue;
            }
            ApplicationInfo appInfo = activityInfo.getApplicationInfo();
            JSONObject appData = new JSONObject();
            try {
                appData.put("id", appInfo.packageName + "-" + currentUser.toString());
                appData.put("packageName", appInfo.packageName);
                if (activityInfo.getLabel() != null) {
                    appData.put("appName", String.valueOf(activityInfo.getLabel()));
                } else {
                    appData.put("appName", appInfo.loadLabel(context.getPackageManager()).toString());
                }
                appData.put("component", activityInfo.getName());
                appData.put("userId", currentUser.toString());
            } catch (Exception e) {
                DNALogger.e("Error while creating app data: " + e.getMessage());
            }
            DNAResultItem app = new DNAResultItem(appData, DNAResultItem.TYPE_APP);
            app.updateAppInstalledState(context);
            appResults.add(app);
        }
        return appResults;
    }

    private static List<String> processQueryIntoParts(String query) {
        List<String> queryParts = new ArrayList<>();
        // Split on spaces.
        String[] parts = query.split(" ");
        for (String part : parts) {
            part = part.trim();
            if (part.length() > 0) {
                queryParts.add(part.toLowerCase());
            }
        }
        // Also split on periods if there is more than one part.
        String[] dotParts = query.split("\\.");
        if (dotParts.length > 1) {
            for (String part : dotParts) {
                part = part.trim();
                if (part.length() > 0) {
                    queryParts.add(part.toLowerCase());
                }
            }
        }
        return queryParts;
    }

    private static List<String> processAppNameIntoParts(String appName) {
        List<String> parts = new ArrayList<>();
        String lowerAppName = appName.toLowerCase();
        parts.add(lowerAppName);
        String[] splitParts = appName.split("[ \\u00A0]+");
        for (String part : splitParts) {
            String lowerPart = part.toLowerCase();
            if (!parts.contains(lowerPart) && lowerPart.length() > 0) {
                String[] capParts = part.split("(?<=[a-z])(?=[A-Z])");
                if (capParts.length > 1) {
                    for (String capPart : capParts) {
                        if (capPart.length() > 0) {
                            String capLower = capPart.toLowerCase();
                            if (!parts.contains(capLower)) {
                                parts.add(capLower);
                            }
                        }
                    }
                }
                parts.add(lowerPart);
            }
        }
        return parts;
    }
}
