// Copyright 2007-2016 Google Inc., David Ehrmann
// Author: Lincoln Smith, David Ehrmann
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Classes to implement an Encoder for the format described in
// RFC 3284 - The VCDIFF Generic Differencing and Compression Data Format.
// The RFC text can be found at http://www.faqs.org/rfcs/rfc3284.html
//
// The RFC describes the possibility of using a secondary compressor
// to further reduce the size of each section of the VCDIFF output.
// That feature is not supported in this implementation of the encoder
// and decoder.
// No secondary compressor types have been publicly registered with
// the IANA at http://www.iana.org/assignments/vcdiff-comp-ids
// in the more than five years since the registry was created, so there
// is no standard set of compressor IDs which would be generated by other
// encoders or accepted by other decoders.

package com.davidehrmann.vcdiff.engine;

import com.davidehrmann.vcdiff.VCDiffCodeTableWriter;
import com.davidehrmann.vcdiff.VCDiffFormatExtension;
import com.davidehrmann.vcdiff.VCDiffStreamingEncoder;
import com.davidehrmann.vcdiff.util.ZeroInitializedAdler32;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.zip.Adler32;

import static com.davidehrmann.vcdiff.VCDiffFormatExtension.GOOGLE_CHECKSUM;

public class VCDiffStreamingEncoderImpl<OUT> implements VCDiffStreamingEncoder<OUT> {
    private static final Logger LOGGER = LoggerFactory.getLogger(VCDiffStreamingEncoderImpl.class);

    protected final VCDiffEngine engine;
    protected final EnumSet<VCDiffFormatExtension> formatExtensions;

    // Determines whether to look for matches within the previously encoded
    // target data, or just within the source (dictionary) data.  Please see
    // vcencoder.h for a full explanation of this parameter.
    protected final boolean lookForTargetMatches;

    protected final VCDiffCodeTableWriter<OUT> coder;

    // This state variable is used to ensure that startEncoding(), encodeChunk(),
    // and finishEncoding() are called in the correct order.  It will be true
    // if startEncoding() has been called, followed by zero or more calls to
    // encodeChunk(), but finishEncoding() has not yet been called.  It will
    // be false initially, and also after finishEncoding() has been called.
    protected boolean encodeChunkAllowed;

    public VCDiffStreamingEncoderImpl(VCDiffCodeTableWriter<OUT> coder,
                                      HashedDictionary dictionary,
                                      EnumSet<VCDiffFormatExtension> format_extensions,
                                      boolean look_for_target_matches) {
        this.engine = dictionary.engine();
        this.formatExtensions = format_extensions.clone();
        this.lookForTargetMatches = look_for_target_matches;
        this.coder = coder;
    }

    // These functions are identical to their counterparts
    // in VCDiffStreamingEncoder.
    public void startEncoding(OUT out) throws IOException {
        coder.init(engine.dictionary_size());
        coder.writeHeader(out, formatExtensions);
        encodeChunkAllowed = true;
    }

    public void encodeChunk(byte[] data, int offset, int length, OUT out) throws IOException {
        if (!encodeChunkAllowed) {
            throw new IllegalStateException("encodeChunk called before startEncoding");
        }
        if ((formatExtensions.contains(GOOGLE_CHECKSUM))) {
            Adler32 adler32 = new ZeroInitializedAdler32();
            adler32.update(data, offset, length);
            coder.addChecksum((int) adler32.getValue());
        }
        engine.Encode(ByteBuffer.wrap(data, offset, length).slice(), lookForTargetMatches, out, coder);
    }

    public void encodeChunk(byte[] data, OUT out) throws IOException {
        encodeChunk(data, 0, data.length, out);
    }

    public void finishEncoding(OUT out) throws IOException {
        if (!encodeChunkAllowed) {
            throw new IllegalStateException("finishEncoding called before startEncoding");
        }
        encodeChunkAllowed = false;
        coder.finishEncoding(out);
    }
}
