// Copyright 2008-2016 Google Inc., David Ehrmann
// Authors: Lincoln Smith, David Ehrmann
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Implements a Decoder for the format described in
// RFC 3284 - The VCDIFF Generic Differencing and Compression Data Format.
// The RFC text can be found at http://www.faqs.org/rfcs/rfc3284.html
//
// The RFC describes the possibility of using a secondary compressor
// to further reduce the size of each section of the VCDIFF output.
// That feature is not supported in this implementation of the encoder
// and decoder.
// No secondary compressor types have been publicly registered with
// the IANA at http://www.iana.org/assignments/vcdiff-comp-ids
// in the more than five years since the registry was created, so there
// is no standard set of compressor IDs which would be generated by other

package com.davidehrmann.vcdiff.engine;

import com.davidehrmann.vcdiff.VCDiffStreamingDecoder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;

import static com.davidehrmann.vcdiff.engine.VCDiffHeaderParser.*;

@SuppressWarnings("ALL")
public class VCDiffStreamingDecoderImpl implements VCDiffStreamingDecoder {
    private static final Logger LOGGER = LoggerFactory.getLogger(VCDiffStreamingDecoderImpl.class);

    /**
     * The default maximum target file size (and target window size) if
     * setMaximumTargetFileSize() is not called.
     */
    public static final int DEFAULT_MAXIMUM_TARGET_FILE_SIZE = 67108864;  // 64 MB

    /**
     * The largest value that can be passed to setMaximumTargetWindowSize().
     * Using a larger value will result in an error.
     */
    public static final int TARGET_SIZE_LIMIT = Integer.MAX_VALUE;

    /**
     * A constant that is the default value for plannedTargetFileSize,
     * indicating that the decoder does not have an expected length
     * for the target data.
     */
    public static final int UNLIMITED_BYTES = -3;

    // Contents and length of the source (dictionary) data.
    private ByteBuffer dictionary;

    // This string will be used to store any unparsed bytes left over when
    // decodeChunk() reaches the end of its input and returns RESULT_END_OF_DATA.
    // It will also be used to concatenate those unparsed bytes with the data
    // supplied to the next call to decodeChunk(), so that they appear in
    // contiguous memory.
    private ByteBuffer unparsedBytes = ByteBuffer.allocate(0);

    // The portion of the target file that has been decoded so far.  This will be
    // used to fill the output string for decodeChunk(), and will also be used to
    // execute COPY instructions that reference target data.  Since the source
    // window can come from a range of addresses in the previously decoded target
    // data, the entire target file needs to be available to the decoder, not just
    // the current target window.
    private final DecoratedByteArrayOutputStream decodedTarget = new DecoratedByteArrayOutputStream(512); //IoBuffer.allocate(512);

    // The VCDIFF version byte (also known as "header4") from the
    // delta file header.
    private byte vcdiffVersionCode;

    private VCDiffDeltaFileWindow deltaWindow;

    private VCDiffAddressCache addrCache;

    // Will be NULL unless a custom code table has been defined.
    private VCDiffCodeTableData custom_code_table_;

    // Used to receive the decoded custom code table.
    private final ByteArrayOutputStream custom_code_table_string_ = new ByteArrayOutputStream(1024);

    // If a custom code table is specified, it will be expressed
    // as an embedded VCDIFF delta file which uses the default code table
    // as the source file (dictionary).  Use a child decoder object
    // to decode that delta file.
    private VCDiffStreamingDecoderImpl custom_code_table_decoder_;

    // If set, then the decoder is expecting *exactly* this number of
    // target bytes to be decoded from one or more delta file windows.
    // If this number is exceeded while decoding a window, but was not met
    // before starting on that window, an error will be reported.
    // If finishDecoding() is called before this number is met, an error
    // will also be reported.  This feature is used for decoding the
    // embedded code table data within a VCDIFF delta file; we want to
    // stop processing the embedded data once the entire code table has
    // been decoded, and treat the rest of the available data as part
    // of the enclosing delta file.
    private int plannedTargetFileSize;

    private long maximumTargetFileSize = DEFAULT_MAXIMUM_TARGET_FILE_SIZE;

    private int maximumTargetWindowSize = DEFAULT_MAXIMUM_TARGET_FILE_SIZE;

    // Contains the sum of the decoded sizes of all target windows seen so far,
    // including the expected total size of the current target window in progress
    // (even if some of the current target window has not yet been decoded.)
    private long totalOfTargetWindowSizes;

    // Contains the byte position within decodedTarget of the first data that
    // has not yet been output by appendNewOutputText().
    private int decodedTargetOutputPosition;

    // This value is used to ensure the correct order of calls to the interface
    // functions, i.e., a single call to startDecoding(), followed by zero or
    // more calls to decodeChunk(), followed by a single call to
    // finishDecoding().
    private boolean startDecodingWasCalled;

    // If this value is true then the VCD_TARGET flag can be specified to allow
    // the source segment to be chosen from the previously-decoded target data.
    // (This is the default behavior.)  If it is false, then specifying the
    // VCD_TARGET flag is considered an error, and the decoder does not need to
    // keep in memory any decoded target data prior to the current window.
    private boolean allowVcdTarget = true;

    public VCDiffStreamingDecoderImpl() {
        deltaWindow = new VCDiffDeltaFileWindow(this);
        reset();
    }

    // Resets all member variables to their initial states.
    public void reset() {
        startDecodingWasCalled = false;
        dictionary = null;
        vcdiffVersionCode = 0;
        plannedTargetFileSize = UNLIMITED_BYTES;
        totalOfTargetWindowSizes = 0;
        addrCache = null;
        custom_code_table_ = null;
        custom_code_table_decoder_ = null;
        deltaWindow.Reset();
        decodedTargetOutputPosition = 0;
    }

    public void startDecoding(byte[] dictionary) {
        startDecoding(ByteBuffer.wrap(dictionary));
    }

    public void startDecoding(ByteBuffer dictionary) {
        if (startDecodingWasCalled) {
            throw new IllegalStateException("startDecoding() called twice without finishDecoding()");
        }

        unparsedBytes = ByteBuffer.allocate(0);
        decodedTarget.reset();  // deltaWindow.reset() depends on this
        reset();
        this.dictionary = dictionary;
        startDecodingWasCalled = true;
    }

    public void decodeChunk(byte[] data, int offset, int len, OutputStream out) throws IOException {
        decodeChunk(ByteBuffer.wrap(data, offset, len), out);
    }

    public void decodeChunk(ByteBuffer data, OutputStream out) throws IOException {
        if (!startDecodingWasCalled) {
            reset();
            throw new IOException("decodeChunk() called without startDecoding()");
        }
        // TODO: there's a lot of room for optimization here
        ByteBuffer parseable_chunk = ByteBuffer.allocate(unparsedBytes.remaining() + data.remaining());
        parseable_chunk.put(unparsedBytes);
        parseable_chunk.put(data);
        parseable_chunk.flip();
        unparsedBytes = parseable_chunk.duplicate();

        try {
            int result = readDeltaFileHeader(parseable_chunk);
            if (RESULT_SUCCESS == result) {
                result = readCustomCodeTable(parseable_chunk);
            }
            if (RESULT_SUCCESS == result) {
                while (parseable_chunk.hasRemaining()) {
                    result = deltaWindow.DecodeWindow(parseable_chunk);
                    if (RESULT_SUCCESS != result) {
                        break;
                    }
                    if (reachedPlannedTargetFileSize()) {
                        // Found exactly the length we expected.  Stop decoding.
                        break;
                    }
                    if (!allowVcdTarget()) {
                        // VCD_TARGET will never be used to reference target data before the
                        // start of the current window, so flush and clear the contents of
                        // decodedTarget.
                        flushDecodedTarget(out);
                    }
                }
            }
        } catch (IOException e) {
            reset();  // Don't allow further decodeChunk calls
            throw e;
        }

        unparsedBytes = parseable_chunk;
        appendNewOutputText(out);
    }

    public void decodeChunk(byte[] data, OutputStream out) throws IOException {
        decodeChunk(ByteBuffer.wrap(data), out);
    }

    public void finishDecoding() throws IOException {
        try {
            if (!startDecodingWasCalled) {
                throw new IOException("finishDecoding() called before startDecoding(), or called after decodeChunk() returned false");
            } else if (!isDecodingComplete()) {
                throw new IOException("finishDecoding() called before parsing entire delta file window");
            }
        } finally {
            // reset the object state for the next decode operation
            reset();
        }
    }

    // If true, the version of VCDIFF used in the current delta file allows
    // for the interleaved format, in which instructions, addresses and data
    // are all sent interleaved in the instructions section of each window
    // rather than being sent in separate sections.  This is not part of
    // the VCDIFF draft standard, so we've defined a special version code
    // 'S' which implies that this feature is available.  Even if interleaving
    // is supported, it is not mandatory; interleaved format will be implied
    // if the address and data sections are both zero-length.
    //
    public boolean allowInterleaved() { return vcdiffVersionCode == 'S'; }

    // If true, the version of VCDIFF used in the current delta file allows
    // each delta window to contain an Adler32 checksum of the target window data.
    // If the bit 0x08 (VCD_CHECKSUM) is set in the Win_Indicator flags, then
    // this checksum will appear as a variable-length integer, just after the
    // "length of addresses for COPYs" value and before the window data sections.
    // It is possible for some windows in a delta file to use the checksum feature
    // and for others not to use it (and leave the flag bit set to 0.)
    // Just as with allowInterleaved(), this extension is not part of the draft
    // standard and is only available when the version code 'S' is specified.
    public boolean allowChecksum() { return vcdiffVersionCode == 'S'; }

    public boolean setMaximumTargetFileSize(long newMaximumTargetFileSize) {
        maximumTargetFileSize = newMaximumTargetFileSize;
        return true;
    }

    public boolean setMaximumTargetWindowSize(int newMaximumTargetWindowSize) {
        maximumTargetWindowSize = newMaximumTargetWindowSize;
        return true;
    }

    // See description of plannedTargetFileSize, below.
    public boolean hasPlannedTargetFileSize() {
        return plannedTargetFileSize != UNLIMITED_BYTES;
    }

    public void setPlannedTargetFileSize(int planned_target_file_size) {
        plannedTargetFileSize = planned_target_file_size;
    }

    public void addToTotalTargetWindowSize(int window_size) {
        totalOfTargetWindowSizes += window_size;
    }

    // Checks to see whether the decoded target data has reached its planned size.
    public boolean reachedPlannedTargetFileSize() {
        if (!hasPlannedTargetFileSize()) {
            return false;
        }
        // The planned target file size should not have been exceeded.
        // targetWindowWouldExceedSizeLimits() ensures that the advertised size of
        // each target window would not make the target file exceed that limit, and
        // DecodeBody() will return RESULT_ERROR if the actual decoded output ever
        // exceeds the advertised target window size.
        if (totalOfTargetWindowSizes > plannedTargetFileSize) {
            throw new IllegalStateException(String.format(
                    "Internal error: Decoded data size %d exceeds planned target file size %d",
                    totalOfTargetWindowSizes, plannedTargetFileSize
            ));
        }
        return totalOfTargetWindowSizes == plannedTargetFileSize;
    }

    // Checks to see whether adding a new target window of the specified size
    // would exceed the planned target file size, the maximum target file size,
    // or the maximum target window size.  If so, logs an error and returns true;
    // otherwise, returns false.
    public void targetWindowWouldExceedSizeLimits(int window_size) throws IOException {
        if (window_size > maximumTargetWindowSize) {
            throw new IOException(String.format(
                    "Length of target window (%d) exceeds limit of %d bytes",
                    window_size, maximumTargetWindowSize
            ));
        }
        if (hasPlannedTargetFileSize()) {
            // The logical expression to check would be:
            //
            //   totalOfTargetWindowSizes + window_size > plannedTargetFileSize
            //
            // but the addition might cause an integer overflow if target_bytes_to_add
            // is very large.  So it is better to check target_bytes_to_add against
            // the remaining planned target bytes.
            long remaining_planned_target_file_size = plannedTargetFileSize - totalOfTargetWindowSizes;
            if (window_size > remaining_planned_target_file_size) {
                throw new IOException(String.format(
                        "Length of target window (%d bytes) plus previous windows (%d bytes) would exceed planned size of %d bytes",
                        window_size, totalOfTargetWindowSizes, plannedTargetFileSize
                ));
            }
        }
        long remaining_maximum_target_bytes = maximumTargetFileSize - totalOfTargetWindowSizes;
        if (window_size > remaining_maximum_target_bytes) {
            throw new IOException(String.format(
                    "Length of target window (%d bytes) plus previous windows (%d bytes) would exceed maximum target file size of %d bytes",
                    window_size, totalOfTargetWindowSizes, maximumTargetFileSize
            ));
        }
    }

    // Returns the amount of input data passed to the last decodeChunk()
    // that was not consumed by the decoder.  This is essential if
    // setPlannedTargetFileSize() is being used, in order to preserve the
    // remaining input data stream once the planned target file has been decoded.
    private int getUnconsumedDataSize() {
        return unparsedBytes.remaining();
    }

    // This function will return true if the decoder has parsed a complete delta
    // file header plus zero or more delta file windows, with no data left over.
    // It will also return true if no delta data at all was decoded.  If these
    // conditions are not met, then finishDecoding() should not be called.
    @SuppressWarnings("SimplifiableIfStatement")
    private boolean isDecodingComplete() {
        if (!FoundFileHeader()) {
            // No complete delta file header has been parsed yet.  decodeChunk()
            // may have received some data that it hasn't yet parsed, in which case
            // decoding is incomplete.
            return !unparsedBytes.hasRemaining();
        } else if (custom_code_table_decoder_ != null) {
            // The decoder is in the middle of parsing a custom code table.
            return false;
        } else if (deltaWindow.FoundWindowHeader()) {
            // The decoder is in the middle of parsing an interleaved format delta
            // window.
            return false;
        } else if (reachedPlannedTargetFileSize()) {
            // The decoder found exactly the planned number of bytes.  In this case
            // it is OK for unparsedBytes to be non-empty; it contains the leftover
            // data after the end of the delta file.
            return true;
        } else {
            // No complete delta file window has been parsed yet.  decodeChunk()
            // may have received some data that it hasn't yet parsed, in which case
            // decoding is incomplete.
            return !unparsedBytes.hasRemaining();
        }
    }

    public ByteBuffer dictionary_ptr() { return dictionary; }

    VCDiffAddressCache addrCache() { return addrCache; }

    DecoratedByteArrayOutputStream decodedTarget() { return decodedTarget; }

    public boolean allowVcdTarget() { return allowVcdTarget; }

    public void setAllowVcdTarget(boolean allowVcdTarget) {
        if (startDecodingWasCalled) {
            throw new IllegalStateException("setAllowVcdTarget() called after startDecoding()");
        }
        this.allowVcdTarget = allowVcdTarget;
    }

    // Reads the VCDiff delta file header section as described in RFC section 4.1,
    // except the custom code table data.  Returns RESULT_ERROR if an error
    // occurred, or RESULT_END_OF_DATA if the end of available data was reached
    // before the entire header could be read.  (The latter may be an error
    // condition if there is no more data available.)  Otherwise, advances
    // data->position_ past the header and returns RESULT_SUCCESS.

    // Reads the VCDiff delta file header section as described in RFC section 4.1:
    //
    //	     Header1                                  - byte = 0xD6 (ASCII 'V' | 0x80)
    //	     Header2                                  - byte = 0xC3 (ASCII 'C' | 0x80)
    //	     Header3                                  - byte = 0xC4 (ASCII 'D' | 0x80)
    //	     Header4                                  - byte
    //	     Hdr_Indicator                            - byte
    //	     [Secondary compressor ID]                - byte
    //	     [Length of code table data]              - integer
    //	     [Code table data]
    //
    // Initializes the code table and address cache objects.  Returns RESULT_ERROR
    // if an error occurred, and RESULT_END_OF_DATA if the end of available data was
    // reached before the entire header could be read.  (The latter may be an error
    // condition if there is no more data available.)  Otherwise, returns
    // RESULT_SUCCESS, and removes the header bytes from the data string.
    //
    // It's relatively inefficient to expect this function to parse any number of
    // input bytes available, down to 1 byte, but it is necessary in case the input
    // is not a properly formatted VCDIFF delta file.  If the entire input consists
    // of two bytes "12", then we should recognize that it does not match the
    // initial VCDIFF magic number "VCD" and report an error, rather than waiting
    // indefinitely for more input that will never arrive.
    private int readDeltaFileHeader(ByteBuffer data) throws IOException {
        if (FoundFileHeader()) {
            return RESULT_SUCCESS;
        }
        int data_size = data.remaining();

        ByteBuffer paddedHeaderData = ByteBuffer.allocate(DeltaFileHeader.SERIALIZED_SIZE);
        paddedHeaderData.put((ByteBuffer) data.slice().limit(Math.min(DeltaFileHeader.SERIALIZED_SIZE, data.remaining())));
        paddedHeaderData.rewind();

        final DeltaFileHeader header = new DeltaFileHeader(paddedHeaderData);
        boolean wrong_magic_number = false;
        switch (data_size) {
            // Verify only the bytes that are available.
            default:
                // Found header contents up to and including VCDIFF version
                vcdiffVersionCode = header.header4;
                if ((vcdiffVersionCode != 0x00) &&  // Draft standard VCDIFF (RFC 3284)
                        (vcdiffVersionCode != 'S')) {   // Enhancements for SDCH protocol
                    throw new IOException("Unrecognized VCDIFF format version");
                }
                // fall through
            case 3:
                if (header.header3 != (byte) 0xC4) {  // magic value 'D' | 0x80
                    wrong_magic_number = true;
                }
                // fall through
            case 2:
                if (header.header2 != (byte) 0xC3) {  // magic value 'C' | 0x80
                    wrong_magic_number = true;
                }
                // fall through
            case 1:
                if (header.header1 != (byte) 0xD6) {  // magic value 'V' | 0x80
                    wrong_magic_number = true;
                }
                // fall through
            case 0:
                if (wrong_magic_number) {
                    throw new IOException("Did not find VCDIFF header bytes; input is not a VCDIFF delta file");
                }
                if (data_size < DeltaFileHeader.SERIALIZED_SIZE) return RESULT_END_OF_DATA;
        }

        int unrecognizedFlags = header.hdr_indicator & 0xff & ~(VCD_DECOMPRESS | VCD_CODETABLE);
        if (unrecognizedFlags != 0) {
            throw new IOException(String.format("Unrecognized hdr_indicator flags: %02x", unrecognizedFlags));
        }

        // Secondary compressor not supported.
        if ((header.hdr_indicator & VCD_DECOMPRESS) != 0) {
            throw new IOException("Secondary compression is not supported");
        }

        if ((header.hdr_indicator & VCD_CODETABLE) != 0) {
            int bytes_parsed = InitCustomCodeTable(data.array(), data.arrayOffset() + data.position() + DeltaFileHeader.SERIALIZED_SIZE,
                    data.remaining() - DeltaFileHeader.SERIALIZED_SIZE);
            if (bytes_parsed == RESULT_END_OF_DATA) {
                return RESULT_END_OF_DATA;
            }
            data.position(data.position() + DeltaFileHeader.SERIALIZED_SIZE + bytes_parsed);
            // TODO unknown flags on hdr_indicator
        } else {
            addrCache = new VCDiffAddressCacheImpl();
            // addrCache->init() will be called
            // from VCDiffStreamingDecoderImpl::decodeChunk()
            data.position(data.position() + DeltaFileHeader.SERIALIZED_SIZE);
        }
        return RESULT_SUCCESS;
    }

    // Indicates whether or not the header has already been read.
    private boolean FoundFileHeader() { return addrCache != null; }

    // If readDeltaFileHeader() finds the VCD_CODETABLE flag set within the delta
    // file header, this function parses the custom cache sizes and initializes
    // a nested VCDiffStreamingDecoderImpl object that will be used to parse the
    // custom code table in readCustomCodeTable().  Returns RESULT_ERROR if an
    // error occurred, or RESULT_END_OF_DATA if the end of available data was
    // reached before the custom cache sizes could be read.  Otherwise, returns
    // the number of bytes read.
    //
    private int InitCustomCodeTable(byte[] data_start, int offset, int length) throws IOException {
        // A custom code table is being specified.  Parse the variable-length
        // cache sizes and begin parsing the encoded custom code table.
        Integer near_cache_size;
        Integer same_cache_size;

        VCDiffHeaderParser header_parser = new VCDiffHeaderParser(ByteBuffer.wrap(data_start, offset, length).slice());
        if ((near_cache_size = header_parser.parseInt32("size of near cache")) == null) {
            LOGGER.warn("Failed to parse size of near cache");
            return header_parser.getResult();
        }
        if ((same_cache_size = header_parser.parseInt32("size of same cache")) == null) {
            LOGGER.warn("Failed to parse size of same cache");
            return header_parser.getResult();
        }

        custom_code_table_ = new VCDiffCodeTableData();

        custom_code_table_string_.reset();
        addrCache = new VCDiffAddressCacheImpl(near_cache_size.shortValue(), same_cache_size.shortValue());

        // addrCache->init() will be called
        // from VCDiffStreamingDecoderImpl::decodeChunk()

        // If we reach this point (the start of the custom code table)
        // without encountering a RESULT_END_OF_DATA condition, then we won't call
        // readDeltaFileHeader() again for this delta file.
        //
        // Instantiate a recursive decoder to interpret the custom code table
        // as a VCDIFF encoding of the default code table.
        custom_code_table_decoder_ = new VCDiffStreamingDecoderImpl();

        byte[] codeTableBytes = VCDiffCodeTableData.kDefaultCodeTableData.getBytes();
        custom_code_table_decoder_.startDecoding(codeTableBytes);
        custom_code_table_decoder_.setPlannedTargetFileSize(codeTableBytes.length);

        return header_parser.unparsedData().position();
    }

    // If a custom code table was specified in the header section that was parsed
    // by readDeltaFileHeader(), this function makes a recursive call to another
    // VCDiffStreamingDecoderImpl object (custom_code_table_decoder_), since the
    // custom code table is expected to be supplied as an embedded VCDIFF
    // encoding that uses the standard code table.  Returns RESULT_ERROR if an
    // error occurs, or RESULT_END_OF_DATA if the end of available data was
    // reached before the entire custom code table could be read.  Otherwise,
    // returns RESULT_SUCCESS and sets *data_ptr to the position after the encoded
    // custom code table.  If the function returns RESULT_SUCCESS or
    // RESULT_END_OF_DATA, it advances data->position_ past the parsed bytes.
    private int readCustomCodeTable(ByteBuffer data) throws IOException {
        if (custom_code_table_decoder_ == null) {
            return RESULT_SUCCESS;
        }
        if (custom_code_table_ == null) {
            throw new IllegalStateException("Internal error: custom_code_table_decoder_ is set, but custom_code_table_ is null");
        }

        try {
            custom_code_table_decoder_.decodeChunk(data.array(),
                    data.arrayOffset() + data.position(), data.remaining(), custom_code_table_string_);
        } catch (IOException cause) {
            IOException e = new IOException("Failed to write to custom_code_table_string_");
            e.initCause(cause);
            throw e;
        }
        if (custom_code_table_string_.size() < VCDiffCodeTableData.SERIALIZED_BYTE_SIZE) {
            // Skip over the consumed data.
            data.position(data.limit());
            return RESULT_END_OF_DATA;
        }

        custom_code_table_decoder_.finishDecoding();

        if (custom_code_table_string_.size() != VCDiffCodeTableData.SERIALIZED_BYTE_SIZE) {
            throw new IOException(String.format(
                    "Decoded custom code table size (%d) does not match size of a code table (%d)",
                    custom_code_table_string_.size(), VCDiffCodeTableData.SERIALIZED_BYTE_SIZE
            ));
        }

        custom_code_table_ = new VCDiffCodeTableData(custom_code_table_string_.toByteArray());
        custom_code_table_string_.reset();

        // Skip over the consumed data.
        data.position(data.limit() - custom_code_table_decoder_.getUnconsumedDataSize());
        custom_code_table_decoder_ = null;
        deltaWindow.useCodeTable(custom_code_table_, addrCache.LastMode());
        return RESULT_SUCCESS;
    }

    // Called after the decoder exhausts all input data.  This function
    // copies from decodedTarget into out all the data that
    // has not yet been output.  It sets decodedTargetOutputPosition
    // to mark the start of the next data that needs to be output.
    private void appendNewOutputText(OutputStream out) throws IOException {
        ByteBuffer decodedTargetBuffer = decodedTarget.toByteBuffer();
        decodedTargetBuffer.position(decodedTargetOutputPosition);

        // TODO: optimize
        while (decodedTargetBuffer.hasRemaining()) {
            out.write(decodedTargetBuffer.get());
        }

        decodedTargetOutputPosition = decodedTargetBuffer.limit();
    }

    // Appends to out the portion of decodedTarget that has
    // not yet been output, then clears decodedTarget.  This function is
    // called after each complete target window has been decoded if
    // allowVcdTarget is false.  In that case, there is no need to retain
    // target data from any window except the current window.
    private void flushDecodedTarget(OutputStream out) throws IOException {
        out.write(
                decodedTarget.getBuffer(),
                decodedTargetOutputPosition,
                decodedTarget.size() - decodedTargetOutputPosition
        );

        decodedTarget.reset();
        deltaWindow.setTargetWindowStartPos(0);
        decodedTargetOutputPosition = 0;
    }

    protected static class DecoratedByteArrayOutputStream extends ByteArrayOutputStream {
        public DecoratedByteArrayOutputStream() {
            super();
        }

        public DecoratedByteArrayOutputStream(int size) {
            super(size);
        }

        public synchronized ByteBuffer toByteBuffer() {
            return ByteBuffer.wrap(buf, 0, count).asReadOnlyBuffer();
        }

        public byte[] getBuffer() {
            return buf;
        }
    }
}
