package com.davidehrmann.vcdiff.engine;

import com.davidehrmann.vcdiff.VCDiffFormatExtension;
import com.davidehrmann.vcdiff.io.CountingOutputStream;
import com.davidehrmann.vcdiff.mina_buffer.IoBuffer;
import com.davidehrmann.vcdiff.util.VarInt;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.OutputStream;
import java.util.EnumSet;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * @author David Ehrmann
 * The method calls after construction *must* conform
 * to the following pattern:
 *    {{add|copy|run}* [addChecksum] output}*
 *
 * When output has been called in this sequence, a complete target window
 * (as defined in RFC 3284 section 4.3) will have been appended to
 * out (unless no calls to add, run, or copy were made, in which
 * case output will do nothing.)  The output will not be available for use
 * until after each call to output().
 *
 * NOT threadsafe.
 */
public class VCDiffCodeTableWriterImpl implements com.davidehrmann.vcdiff.VCDiffCodeTableWriter<OutputStream> {

    private static final Logger LOGGER = LoggerFactory.getLogger(VCDiffCodeTableWriterImpl.class);

    private static final byte[] HEADER_STANDARD_FORMAT = new byte[] {
        (byte)0xD6, // Header1: "V" | 0x80
        (byte)0xC3, // Header2: "C" | 0x80
        (byte)0xC4, // Header3: "D" | 0x80
        (byte)0x00, // Header4: Draft standard format
        (byte)0x00	// Hdr_Indicator: No compression, no custom code table
    };

    private static final byte[] HEADER_EXTENDED_FORMAT = new byte[] {
        (byte)0xD6, // Header1: "V" | 0x80
        (byte)0xC3, // Header2: "C" | 0x80
        (byte)0xC4, // Header3: "D" | 0x80
        (byte)'S',  // Header4: VCDIFF/SDCH, extensions used
        (byte)0x00	// Hdr_Indicator: No compression, no custom code table
    };

    static final int VCD_SOURCE = 0x01;
    static final int VCD_TARGET = 0x02;
    static final int VCD_CHECKSUM = 0x04;

    /**
     * The maximum value for the mode of a COPY instruction.
     */
    private final int maxMode;

    // A series of instruction opcodes, each of which may be followed
    // by one or two Varint values representing the size parameters
    // of the first and second instruction in the opcode.
    private IoBuffer instructionsAndSizes = IoBuffer.allocate(1024);

    // A series of data arguments (byte values) used for ADD and RUN
    // instructions.  Depending on whether interleaved output is used
    // for streaming or not, the pointer may point to
    // separateDataForAddAndRun or to instructionsAndSizes.
    private IoBuffer dataForAddAndRun;
    private final IoBuffer separateDataForAddAndRun = IoBuffer.allocate(1024);

    // A series of Varint addresses used for COPY instructions.
    // For the SAME mode, a byte value is stored instead of a Varint.
    // Depending on whether interleaved output is used
    // for streaming or not, the pointer may point to
    // separateAddressesForCopy or to instructionsAndSizes.
    private IoBuffer addressesForCopy;
    private final IoBuffer separateAddressesForCopy = IoBuffer.allocate(1024);

    private final VCDiffAddressCache addraddressCachess_cache_;

    private int dictionarySize;

    // The number of bytes of target data that has been encoded so far.
    // Each time add(), copy(), or run() is called, this will be incremented.
    // The target length is used to compute HERE mode addresses
    // for COPY instructions, and is also written into the header
    // of the delta window when output() is called.
    private int targetLength;

    private final VCDiffCodeTableData codeTableData;

    // The instruction map facilitates finding an opcode quickly given an
    // instruction inst, size, and mode.  This is an alternate representation
    // of the same information that is found in codeTableData.
    private VCDiffInstructionMap instructionMap;

    // The zero-based index within instructionsAndSizes of the byte
    // that contains the last single-instruction opcode generated by
    // encodeInstruction().  (See that function for exhaustive details.)
    // It is necessary to use an index rather than a pointer for this value
    // because instructionsAndSizes may be resized, which would invalidate
    // any pointers into its data buffer.  The value -1 is reserved to mean that
    // either no opcodes have been generated yet, or else the last opcode
    // generated was a double-instruction opcode.
    private int lastOpcodeIndex;

    // If true, an Adler32 checksum of the target window data will be written as
    // a variable-length integer, just after the size of the addresses section.
    private boolean addChecksum;

    // The checksum to be written to the current target window,
    // if addChecksum is true.
    // This will not be calculated based on the individual calls to add(), run(),
    // and copy(), which would be unnecessarily expensive.  Instead, the code
    // that uses the VCDiffCodeTableWriter object is expected to calculate
    // the checksum all at once and to call addChecksum() with that value.
    // Must be called sometime before calling output(), though it can be called
    // either before or after the calls to add(), run(), and copy().
    private long checksum;

    /**
     * This constructor uses the default code table.
     * If interleaved is true, the encoder writes each delta file window
     * by interleaving instructions and sizes with their corresponding
     * addresses and data, rather than placing these elements into three
     * separate sections.  This facilitates providing partially
     * decoded results when only a portion of a delta file window
     * is received (e.g. when HTTP over TCP is used as the
     * transmission protocol.)  The interleaved format is
     * not consistent with the VCDIFF draft standard.
     *
     * @param interleaved Whether or not to interleave the output data
     */
    public VCDiffCodeTableWriterImpl(boolean interleaved) {
        maxMode = VCDiffAddressCache.DefaultLastMode();
        dictionarySize = 0;
        targetLength = 0;
        codeTableData = VCDiffCodeTableData.kDefaultCodeTableData;
        instructionMap = null;
        lastOpcodeIndex = -1;
        addChecksum = false;
        checksum = 0;
        addraddressCachess_cache_ = new VCDiffAddressCacheImpl();

        instructionsAndSizes.setAutoExpand(true);
        separateDataForAddAndRun.setAutoExpand(true);
        separateAddressesForCopy.setAutoExpand(true);

        initSectionPointers(interleaved);
    }

    // TODO: expose this constructor safely
    /**
     *
     * Uses a non-standard code table and non-standard cache sizes.  The caller
     * must guarantee that codeTableData remains allocated for the lifetime of
     * the VCDiffCodeTableWriter object.  Note that this is different from how
     * VCDiffCodeTableReader::useCodeTable works.  It is assumed that a given
     * encoder will use either the default code table or a statically-defined
     * non-standard code table, whereas the decoder must have the ability to read
     * an arbitrary non-standard code table from a delta file and discard it once
     * the file has been decoded.
     *
     * @param interleaved Whether or not to interleave the output data
     * @param nearCacheSize size of the near cache
     * @param sameCacheSize size of the same cache
     * @param codeTableData custom code table data
     * @param maxMode maximum value for the mode of a COPY instruction.
     */
    VCDiffCodeTableWriterImpl(boolean interleaved, short nearCacheSize, short sameCacheSize, VCDiffCodeTableData codeTableData, short maxMode) {
        addraddressCachess_cache_ = new VCDiffAddressCacheImpl(nearCacheSize, sameCacheSize);
        dictionarySize = 0;
        targetLength = 0;
        this.codeTableData = codeTableData;
        instructionMap = null;
        lastOpcodeIndex = -1;
        addChecksum = false;
        checksum = 0;
        this.maxMode = maxMode;
        initSectionPointers(interleaved);
    }

    /**
     * Initializes the constructed object for use.
     * This method must be called after a VCDiffCodeTableWriter is constructed
     * and before any of its other methods can be called.  It will return
     * false if there was an error initializing the object, or true if it
     *  was successful.  After the object has been initialized and used,
     * init() can be called again to restore the initial state of the object.
     */
    public void init(int dictionarySize) {
        this.dictionarySize = dictionarySize;
        if (instructionMap == null) {
            if (codeTableData == VCDiffCodeTableData.kDefaultCodeTableData) {
                instructionMap = VCDiffInstructionMap.DEFAULT_INSTRUCTION_MAP;
            } else {
                instructionMap = new VCDiffInstructionMap(codeTableData, (byte) maxMode);
            }
        }

        addraddressCachess_cache_.Init();

        targetLength = 0;
        lastOpcodeIndex = -1;
    }

    /**
     *  encode an ADD opcode with the "size" bytes starting at data
     */
    public void add(byte[] data, int offset, int length) {
        if (offset + length > data.length || length < 0) {
            throw new IllegalArgumentException();
        }

        encodeInstruction(VCDiffCodeTableData.VCD_ADD, length);
        dataForAddAndRun.put(data, offset, length);
        targetLength += length;
    }

    public void addChecksum(int checksum) {
        addChecksum = true;
        this.checksum = checksum & 0xffffffffL;
    }

    /**
     *  encode a COPY opcode with args "offset" (into dictionary) and "size" bytes.
     */
    public void copy(int offset, int size) {
        if (instructionMap == null) {
            throw new IllegalStateException("copy called without calling init().");
        }

        // If a single interleaved stream of encoded values is used
        // instead of separate sections for instructions, addresses, and data,
        // then the string instructionsAndSizes may be the same as
        // addressesForCopy.  The address should therefore be encoded
        // *after* the instruction and its size.
        AtomicInteger encoded_addr = new AtomicInteger(0);
        final byte mode = (byte) addraddressCachess_cache_.EncodeAddress(offset, dictionarySize + targetLength, encoded_addr);
        encodeInstruction(VCDiffCodeTableData.VCD_COPY, size, mode);
        if (addraddressCachess_cache_.WriteAddressAsVarintForMode(mode)) {
            addressesForCopy.expand(VarInt.calculateIntLength(encoded_addr.get()));
            VarInt.putInt(addressesForCopy.buf(), encoded_addr.get());
        } else {
            addressesForCopy.put((byte)encoded_addr.get());
        }
        targetLength += size;
    }

    /**
     * There should not be any need to output more data
     * since encodeChunk() encodes a complete target window
     * and there is no end-of-delta-file marker.
     *
     * @param out
     */
    public void finishEncoding(OutputStream out) throws IOException {

    }

    public int getDeltaWindowSize() {
        final int length_of_the_delta_encoding = calculateLengthOfTheDeltaEncoding();
        return
                length_of_the_delta_encoding +
                        1 +  // Win_Indicator
                        VarInt.calculateIntLength(dictionarySize) +
                        VarInt.calculateIntLength(0) +
                        VarInt.calculateIntLength(length_of_the_delta_encoding)
                ;
    }

    /**
     * Appends the encoded delta window to the output
     * string.  The output string is not null-terminated and may contain embedded
     * '\0' characters.
     */
    public void output(OutputStream out) throws IOException {
        if (instructionsAndSizes.position() == 0) {
            LOGGER.warn("Empty input; no delta window produced");
        } else {
            CountingOutputStream countedOut = new CountingOutputStream(out);

            // add first element: Win_Indicator
            if (addChecksum) {
                countedOut.write(VCD_SOURCE | VCD_CHECKSUM);
            } else {
                countedOut.write(VCD_SOURCE);
            }

            // Source segment size: dictionary size
            VarInt.writeInt(countedOut, dictionarySize);

            // Source segment position: 0 (start of dictionary)
            VarInt.writeInt(countedOut, 0);

            // [Here is where a secondary compressor would be used
            //  if the encoder and decoder supported that feature.]

            final int length_of_the_delta_encoding = this.calculateLengthOfTheDeltaEncoding();

            VarInt.writeInt(countedOut, length_of_the_delta_encoding);

            // Start of Delta Encoding
            final int size_before_delta_encoding = (int) countedOut.getBytesWritten();

            VarInt.writeInt(countedOut, targetLength);
            countedOut.write(0x00);  // Delta_Indicator: no compression
            VarInt.writeInt(countedOut, separateDataForAddAndRun.position());
            VarInt.writeInt(countedOut, instructionsAndSizes.position());
            VarInt.writeInt(countedOut, separateAddressesForCopy.position());
            if (addChecksum) {
                // The checksum is a 32-bit *unsigned* integer.  VarintBE requires a
                // signed type, so use a 64-bit signed integer to store the checksum.
                VarInt.writeLong(countedOut, checksum);
            }

            countedOut.write(separateDataForAddAndRun.array(), separateDataForAddAndRun.arrayOffset(), separateDataForAddAndRun.position());
            countedOut.write(instructionsAndSizes.array(), instructionsAndSizes.arrayOffset(), instructionsAndSizes.position());
            countedOut.write(separateAddressesForCopy.array(), separateAddressesForCopy.arrayOffset(), separateAddressesForCopy.position());

            // End of Delta Encoding
            final int size_after_delta_encoding = (int) countedOut.getBytesWritten();
            if (length_of_the_delta_encoding != (size_after_delta_encoding - size_before_delta_encoding)) {
                throw new IllegalStateException(String.format(
                        "Internal error: calculated length of the delta encoding (%d) does not match actual length (%d)",
                        length_of_the_delta_encoding, size_after_delta_encoding - size_before_delta_encoding
                ));
            }
            separateDataForAddAndRun.clear();
            instructionsAndSizes.clear();
            separateAddressesForCopy.clear();
            if (targetLength == 0) {
                LOGGER.warn("Empty target window");
            }
        }

        // reset state for next window; assume we are using same code table
        // and dictionary.  The caller will have to invoke init() if a different
        // dictionary is used.
        //
        // Notably, init() calls addraddressCachess_cache_.init().  This resets the address
        // cache between delta windows, as required by RFC section 5.1.
        init(dictionarySize);
    }

    /**
     *  encode a RUN opcode for "size" copies of the value "byte".
     */
    public void run(int size, byte b) {
        encodeInstruction(VCDiffCodeTableData.VCD_RUN, size);
        dataForAddAndRun.put(b);
        targetLength += size;
    }

    /**
     * Write the header (as defined in section 4.1 of the RFC) to *out.
     * This includes information that can be gathered
     * before the first chunk of input is available.
     */
    public void writeHeader(OutputStream out, EnumSet<VCDiffFormatExtension> formatExtensions) throws IOException {
        if (formatExtensions.isEmpty()) {
            out.write(HEADER_STANDARD_FORMAT);
        } else {
            out.write(HEADER_EXTENDED_FORMAT);
        }

        // If custom cache table sizes or a custom code table were used
        // for encoding, here is where they would be appended to *output.
        // This implementation of the encoder does not use those features,
        // although the decoder can understand and interpret them.
    }

    public int targetLength() {
        return targetLength;
    }

    /**
     * If interleaved is true, sets dataForAddAndRun and
     * addressesForCopy to point at instructionsAndSizes,
     * so that instructions, sizes, addresses and data will be
     * combined into a single interleaved stream.
     * If interleaved is false, sets dataForAddAndRun and
     * addressesForCopy to point at their corresponding
     * separate_... strings, so that the three sections will
     * be generated separately from one another.
     */
    void initSectionPointers(boolean interleaved) {
        if (interleaved) {
            dataForAddAndRun = instructionsAndSizes;
            addressesForCopy = instructionsAndSizes;
        } else {
            dataForAddAndRun = separateDataForAddAndRun;
            addressesForCopy = separateAddressesForCopy;
        }
    }

    // Determines the best opcode to encode an instruction, and appends
    // or substitutes that opcode and its size into the
    // instructionsAndSizes string.
    private void encodeInstruction(byte inst, int size, byte mode) {
        if (instructionMap == null) {
            throw new IllegalStateException("encodeInstruction() called without calling init()");
        }

        if (lastOpcodeIndex >= 0) {
            final byte last_opcode = instructionsAndSizes.get(lastOpcodeIndex);
            // The encoding engine should not generate two ADD instructions in a row.
            // This won't cause a failure, but it's inefficient encoding and probably
            // represents a bug in the higher-level logic of the encoder.
            if (inst == VCDiffCodeTableData.VCD_ADD && codeTableData.inst1[last_opcode & 0xff] == VCDiffCodeTableData.VCD_ADD) {
                LOGGER.warn("encodeInstruction() called for two ADD instructions in a row");
            }

            short compound_opcode;
            if (size <= 255) {
                compound_opcode = instructionMap.LookupSecondOpcode(last_opcode, inst, (byte)size, mode);
                if (compound_opcode != VCDiffCodeTableData.kNoOpcode) {
                    instructionsAndSizes.put(lastOpcodeIndex, (byte)compound_opcode);
                    lastOpcodeIndex = -1;
                    return;
                }
            }

            // Try finding a compound opcode with size 0.
            compound_opcode = instructionMap.LookupSecondOpcode(last_opcode, inst, (byte)0, mode);
            if (compound_opcode != VCDiffCodeTableData.kNoOpcode) {
                instructionsAndSizes.put(lastOpcodeIndex, (byte)compound_opcode);
                lastOpcodeIndex = -1;
                instructionsAndSizes.expand(VarInt.calculateIntLength(size));
                VarInt.putInt(instructionsAndSizes.buf(), size);
                return;
            }
        }

        short opcode;
        if (size <= 255) {
            opcode = instructionMap.LookupFirstOpcode(inst, (byte)size, mode);
            if (opcode != VCDiffCodeTableData.kNoOpcode) {
                instructionsAndSizes.put((byte) opcode);
                lastOpcodeIndex = instructionsAndSizes.position() - 1;
                return;
            }
        }

        // There should always be an opcode with size 0.
        opcode = instructionMap.LookupFirstOpcode(inst, (byte)0, mode);
        if (opcode == VCDiffCodeTableData.kNoOpcode) {
            throw new IllegalStateException(String.format(
                    "No matching opcode found for inst %d, mode %d, size 0", inst, mode
            ));
        }

        instructionsAndSizes.put((byte)opcode);
        lastOpcodeIndex = instructionsAndSizes.position() - 1;
        instructionsAndSizes.expand(VarInt.calculateIntLength(size));
        VarInt.putInt(instructionsAndSizes.buf(), size);
    }

    private void encodeInstruction(byte inst, int size) {
        encodeInstruction(inst, size, (byte)0);
    }

    // Calculates the "Length of the delta encoding" field for the delta window
    // header, based on the sizes of the sections and of the other header
    // elements.
    private int calculateLengthOfTheDeltaEncoding() {
        int length_of_the_delta_encoding =
            VarInt.calculateIntLength(targetLength) +
            1 +  // Delta_Indicator
                    VarInt.calculateIntLength(separateDataForAddAndRun.position()) +
                    VarInt.calculateIntLength(instructionsAndSizes.position()) +
                    VarInt.calculateIntLength(separateAddressesForCopy.position()) +
            separateDataForAddAndRun.position() +
            instructionsAndSizes.position() +
            separateAddressesForCopy.position();
        if (addChecksum) {
            length_of_the_delta_encoding += VarInt.calculateLongLength(checksum);
        }

        return length_of_the_delta_encoding;
    }
}
