/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.core;

import com.datastax.driver.core.DataType;
import com.datastax.driver.core.TupleType;
import com.datastax.driver.core.TupleValue;
import com.datastax.driver.core.TypeCodec;
import com.datastax.driver.core.UDTValue;
import com.datastax.driver.core.UserType;
import com.datastax.driver.core.exceptions.CodecNotFoundException;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.reflect.TypeToken;
import com.google.common.util.concurrent.UncheckedExecutionException;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class CodecRegistry {
    private static final Logger logger = LoggerFactory.getLogger(CodecRegistry.class);
    private static final ImmutableSet<TypeCodec<?>> PRIMITIVE_CODECS = ImmutableSet.of((Object)TypeCodec.BlobCodec.instance, (Object)TypeCodec.BooleanCodec.instance, (Object)TypeCodec.SmallIntCodec.instance, (Object)TypeCodec.TinyIntCodec.instance, (Object)TypeCodec.IntCodec.instance, (Object)TypeCodec.BigintCodec.instance, (Object[])new TypeCodec[]{TypeCodec.CounterCodec.instance, TypeCodec.DoubleCodec.instance, TypeCodec.FloatCodec.instance, TypeCodec.VarintCodec.instance, TypeCodec.DecimalCodec.instance, TypeCodec.VarcharCodec.instance, TypeCodec.AsciiCodec.instance, TypeCodec.TimestampCodec.instance, TypeCodec.DateCodec.instance, TypeCodec.TimeCodec.instance, TypeCodec.UUIDCodec.instance, TypeCodec.TimeUUIDCodec.instance, TypeCodec.InetCodec.instance});
    static final ImmutableList<TypeCodec<?>> DEFAULT_CODECS;
    public static final CodecRegistry IMMUTABLE_INSTANCE;
    private final boolean immutable;
    private final CopyOnWriteArrayList<TypeCodec<?>> codecs;
    private final LoadingCache<CacheKey, TypeCodec<?>> cache;

    public CodecRegistry() {
        this(false);
    }

    private CodecRegistry(boolean immutable) {
        this.immutable = immutable;
        this.codecs = new CopyOnWriteArrayList((Collection<TypeCodec<?>>)DEFAULT_CODECS);
        this.cache = CacheBuilder.newBuilder().build(new CacheLoader<CacheKey, TypeCodec<?>>(){

            public TypeCodec<?> load(CacheKey cacheKey) {
                return CodecRegistry.this.findCodec(cacheKey.cqlType, (Object)cacheKey.javaType);
            }
        });
    }

    public CodecRegistry register(TypeCodec<?> codec) {
        return this.register(codec, false);
    }

    public CodecRegistry register(TypeCodec<?> codec, boolean includeDerivedCollectionCodecs) {
        return this.register(Collections.singleton(codec), includeDerivedCollectionCodecs);
    }

    public CodecRegistry register(TypeCodec<?> ... codecs) {
        return this.register(Arrays.asList(codecs), false);
    }

    public CodecRegistry register(Iterable<? extends TypeCodec<?>> codecs) {
        return this.register(codecs, false);
    }

    public CodecRegistry register(Iterable<? extends TypeCodec<?>> codecs, boolean includeDerivedCollectionCodecs) {
        if (this.immutable) {
            throw new UnsupportedOperationException("This CodecRegistry instance is immutable");
        }
        for (TypeCodec<?> codec : codecs) {
            this.codecs.add(codec);
            if (!includeDerivedCollectionCodecs) continue;
            this.codecs.add(new TypeCodec.ListCodec(codec));
            this.codecs.add(new TypeCodec.SetCodec(codec));
            this.codecs.add(new TypeCodec.MapCodec(codec, codec));
            for (TypeCodec primitiveCodec : PRIMITIVE_CODECS) {
                this.codecs.add(new TypeCodec.MapCodec(primitiveCodec, codec));
                this.codecs.add(new TypeCodec.MapCodec(codec, primitiveCodec));
            }
        }
        return this;
    }

    public <T> TypeCodec<T> codecFor(T value) {
        Preconditions.checkNotNull(value, (Object)"Parameter value cannot be null");
        TypeCodec<T> codec = this.findCodec(null, value);
        return codec;
    }

    public <T> TypeCodec<T> codecFor(DataType cqlType) throws CodecNotFoundException {
        return this.lookupCache(cqlType, null);
    }

    public <T> TypeCodec<T> codecFor(DataType cqlType, Class<T> javaType) throws CodecNotFoundException {
        return this.codecFor(cqlType, (T)TypeToken.of(javaType));
    }

    public <T> TypeCodec<T> codecFor(DataType cqlType, TypeToken<T> javaType) throws CodecNotFoundException {
        Preconditions.checkNotNull((Object)cqlType, (Object)"Parameter cqlType cannot be null");
        Preconditions.checkNotNull(javaType, (Object)"Parameter javaType cannot be null");
        return this.lookupCache(cqlType, javaType);
    }

    public <T> TypeCodec<T> codecFor(DataType cqlType, T value) {
        Preconditions.checkNotNull((Object)cqlType, (Object)"Parameter cqlType cannot be null");
        Preconditions.checkNotNull(value, (Object)"Parameter value cannot be null");
        return this.findCodec(cqlType, value);
    }

    private <T> TypeCodec<T> lookupCache(DataType cqlType, TypeToken<T> javaType) {
        Preconditions.checkNotNull((Object)cqlType, (Object)"Parameter cqlType cannot be null");
        if (logger.isTraceEnabled()) {
            logger.trace("Looking up codec for {} <-> {}", (Object)cqlType, javaType);
        }
        CacheKey cacheKey = new CacheKey(javaType, cqlType);
        try {
            return (TypeCodec)this.cache.get((Object)cacheKey);
        }
        catch (UncheckedExecutionException e) {
            if (e.getCause() instanceof CodecNotFoundException) {
                throw (CodecNotFoundException)e.getCause();
            }
            throw new CodecNotFoundException(e.getCause(), cqlType, javaType);
        }
        catch (ExecutionException e) {
            throw new CodecNotFoundException(e.getCause(), cqlType, javaType);
        }
    }

    private <T> TypeCodec<T> findCodec(DataType cqlType, TypeToken<T> javaType) {
        Preconditions.checkNotNull((Object)cqlType, (Object)"Parameter cqlType cannot be null");
        if (logger.isTraceEnabled()) {
            logger.trace("Looking for codec [{} <-> {}]", cqlType == null ? "ANY" : cqlType, javaType == null ? "ANY" : javaType);
        }
        for (TypeCodec<?> codec : this.codecs) {
            if (cqlType != null && !codec.accepts(cqlType) || javaType != null && !codec.accepts(javaType)) continue;
            if (logger.isTraceEnabled()) {
                logger.trace("Codec found: {}", codec);
            }
            return codec;
        }
        TypeCodec<TypeToken<T>> codec = this.maybeCreateCodec(cqlType, (T)javaType);
        if (codec == null) {
            throw CodecRegistry.logErrorAndThrow(cqlType, javaType);
        }
        if (!codec.accepts(cqlType) || javaType != null && !codec.accepts(javaType)) {
            throw CodecRegistry.logErrorAndThrow(cqlType, javaType);
        }
        if (logger.isTraceEnabled()) {
            logger.trace("Codec found: {}", codec);
        }
        this.codecs.addIfAbsent(codec);
        return codec;
    }

    private <T> TypeCodec<T> findCodec(DataType cqlType, T value) {
        Preconditions.checkNotNull(value, (Object)"Parameter value cannot be null");
        if (logger.isTraceEnabled()) {
            logger.trace("Looking for codec [{} <-> {}]", cqlType == null ? "ANY" : cqlType, value.getClass());
        }
        for (TypeCodec<?> codec : this.codecs) {
            if (cqlType != null && !codec.accepts(cqlType) || !codec.accepts(value)) continue;
            if (logger.isTraceEnabled()) {
                logger.trace("Codec found: {}", codec);
            }
            return codec;
        }
        TypeCodec<T> codec = this.maybeCreateCodec(cqlType, value);
        if (codec == null) {
            throw CodecRegistry.logErrorAndThrow(cqlType, TypeToken.of(value.getClass()));
        }
        if (cqlType != null && !codec.accepts(cqlType) || !codec.accepts(value)) {
            throw CodecRegistry.logErrorAndThrow(cqlType, TypeToken.of(value.getClass()));
        }
        if (logger.isTraceEnabled()) {
            logger.trace("Codec found: {}", codec);
        }
        this.codecs.addIfAbsent(codec);
        return codec;
    }

    private <T> TypeCodec<T> maybeCreateCodec(DataType cqlType, TypeToken<T> javaType) {
        if (this.immutable) {
            return null;
        }
        Preconditions.checkNotNull((Object)cqlType);
        if ((cqlType.getName() == DataType.Name.VARCHAR || cqlType.getName() == DataType.Name.TEXT) && javaType != null && Enum.class.isAssignableFrom(javaType.getRawType())) {
            return new TypeCodec.EnumStringCodec(javaType.getRawType());
        }
        if (cqlType.getName() == DataType.Name.LIST && (javaType == null || List.class.isAssignableFrom(javaType.getRawType()))) {
            TypeToken elementType = null;
            if (javaType != null && javaType.getType() instanceof ParameterizedType) {
                Type[] typeArguments = ((ParameterizedType)javaType.getType()).getActualTypeArguments();
                elementType = TypeToken.of((Type)typeArguments[0]);
            }
            TypeCodec<Object> eltCodec = this.findCodec(cqlType.getTypeArguments().get(0), (T)elementType);
            return new TypeCodec.ListCodec<Object>(eltCodec);
        }
        if (cqlType.getName() == DataType.Name.SET && (javaType == null || Set.class.isAssignableFrom(javaType.getRawType()))) {
            TypeToken elementType = null;
            if (javaType != null && javaType.getType() instanceof ParameterizedType) {
                Type[] typeArguments = ((ParameterizedType)javaType.getType()).getActualTypeArguments();
                elementType = TypeToken.of((Type)typeArguments[0]);
            }
            TypeCodec<Object> eltCodec = this.findCodec(cqlType.getTypeArguments().get(0), (T)elementType);
            return new TypeCodec.SetCodec<Object>(eltCodec);
        }
        if (cqlType.getName() == DataType.Name.MAP && (javaType == null || Map.class.isAssignableFrom(javaType.getRawType()))) {
            TypeToken keyType = null;
            TypeToken valueType = null;
            if (javaType != null && javaType.getType() instanceof ParameterizedType) {
                Type[] typeArguments = ((ParameterizedType)javaType.getType()).getActualTypeArguments();
                keyType = TypeToken.of((Type)typeArguments[0]);
                valueType = TypeToken.of((Type)typeArguments[1]);
            }
            TypeCodec<Object> keyCodec = this.findCodec(cqlType.getTypeArguments().get(0), (T)keyType);
            TypeCodec<Object> valueCodec = this.findCodec(cqlType.getTypeArguments().get(1), (T)valueType);
            return new TypeCodec.MapCodec<Object, Object>(keyCodec, valueCodec);
        }
        if (cqlType instanceof TupleType && (javaType == null || TupleValue.class.isAssignableFrom(javaType.getRawType()))) {
            return new TypeCodec.TupleCodec((TupleType)cqlType);
        }
        if (cqlType instanceof UserType && (javaType == null || UDTValue.class.isAssignableFrom(javaType.getRawType()))) {
            return new TypeCodec.UDTCodec((UserType)cqlType);
        }
        return null;
    }

    private <T> TypeCodec<T> maybeCreateCodec(DataType cqlType, T value) {
        if (this.immutable) {
            return null;
        }
        Preconditions.checkNotNull(value);
        if ((cqlType == null || cqlType.getName() == DataType.Name.VARCHAR || cqlType.getName() == DataType.Name.TEXT) && value instanceof Enum) {
            return new TypeCodec.EnumStringCodec(value.getClass());
        }
        if ((cqlType == null || cqlType.getName() == DataType.Name.LIST) && value instanceof List) {
            List list = (List)value;
            if (list.isEmpty()) {
                DataType elementType = cqlType == null || cqlType.getTypeArguments().isEmpty() ? DataType.blob() : cqlType.getTypeArguments().get(0);
                return new TypeCodec.ListCodec<TypeToken>(this.findCodec(elementType, (T)null));
            }
            DataType elementType = cqlType == null || cqlType.getTypeArguments().isEmpty() ? null : cqlType.getTypeArguments().get(0);
            return new TypeCodec.ListCodec(this.findCodec(elementType, list.iterator().next()));
        }
        if ((cqlType == null || cqlType.getName() == DataType.Name.SET) && value instanceof Set) {
            Set set = (Set)value;
            if (set.isEmpty()) {
                DataType elementType = cqlType == null || cqlType.getTypeArguments().isEmpty() ? DataType.blob() : cqlType.getTypeArguments().get(0);
                return new TypeCodec.SetCodec<TypeToken>(this.findCodec(elementType, (T)null));
            }
            DataType elementType = cqlType == null || cqlType.getTypeArguments().isEmpty() ? null : cqlType.getTypeArguments().get(0);
            return new TypeCodec.SetCodec(this.findCodec(elementType, set.iterator().next()));
        }
        if ((cqlType == null || cqlType.getName() == DataType.Name.MAP) && value instanceof Map) {
            Map map = (Map)value;
            if (map.isEmpty()) {
                DataType keyType = cqlType == null || cqlType.getTypeArguments().size() < 1 ? DataType.blob() : cqlType.getTypeArguments().get(0);
                DataType valueType = cqlType == null || cqlType.getTypeArguments().size() < 2 ? DataType.blob() : cqlType.getTypeArguments().get(1);
                return new TypeCodec.MapCodec<TypeToken, TypeToken>(this.findCodec(keyType, (T)null), this.findCodec(valueType, (T)null));
            }
            DataType keyType = cqlType == null || cqlType.getTypeArguments().size() < 1 ? null : cqlType.getTypeArguments().get(0);
            DataType valueType = cqlType == null || cqlType.getTypeArguments().size() < 2 ? null : cqlType.getTypeArguments().get(1);
            Map.Entry entry = map.entrySet().iterator().next();
            return new TypeCodec.MapCodec(this.findCodec(keyType, entry.getKey()), this.findCodec(valueType, entry.getValue()));
        }
        if ((cqlType == null || cqlType.getName() == DataType.Name.TUPLE) && value instanceof TupleValue) {
            return new TypeCodec.TupleCodec(cqlType == null ? ((TupleValue)value).getType() : (TupleType)cqlType);
        }
        if ((cqlType == null || cqlType.getName() == DataType.Name.UDT) && value instanceof UDTValue) {
            return new TypeCodec.UDTCodec(cqlType == null ? ((UDTValue)value).getType() : (UserType)cqlType);
        }
        return null;
    }

    private static CodecNotFoundException logErrorAndThrow(DataType cqlType, TypeToken<?> javaType) {
        String msg = String.format("Codec not found for requested operation: [%s <-> %s]", cqlType == null ? "ANY" : cqlType, javaType == null ? "ANY" : javaType);
        logger.error(msg);
        return new CodecNotFoundException(msg, cqlType, javaType);
    }

    static {
        ImmutableList.Builder builder = new ImmutableList.Builder();
        builder.addAll(PRIMITIVE_CODECS);
        for (TypeCodec primitiveCodec1 : PRIMITIVE_CODECS) {
            builder.add(new TypeCodec.ListCodec(primitiveCodec1));
            builder.add(new TypeCodec.SetCodec(primitiveCodec1));
            for (TypeCodec primitiveCodec2 : PRIMITIVE_CODECS) {
                builder.add(new TypeCodec.MapCodec(primitiveCodec1, primitiveCodec2));
            }
        }
        DEFAULT_CODECS = builder.build();
        IMMUTABLE_INSTANCE = new CodecRegistry(true);
    }

    private static final class CacheKey {
        private final TypeToken<?> javaType;
        private final DataType cqlType;

        public CacheKey(TypeToken<?> javaType, DataType cqlType) {
            this.javaType = javaType;
            this.cqlType = cqlType;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            CacheKey cacheKey = (CacheKey)o;
            return Objects.equal(this.javaType, cacheKey.javaType) && Objects.equal((Object)this.cqlType, (Object)cacheKey.cqlType);
        }

        public int hashCode() {
            return Objects.hashCode((Object[])new Object[]{this.javaType, this.cqlType});
        }
    }
}

