/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.core;

import com.datastax.driver.core.AuthProvider;
import com.datastax.driver.core.Authenticator;
import com.datastax.driver.core.BusyConnectionException;
import com.datastax.driver.core.CloseFuture;
import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.ClusterNameMismatchException;
import com.datastax.driver.core.Configuration;
import com.datastax.driver.core.ConnectionException;
import com.datastax.driver.core.DefaultResultSetFuture;
import com.datastax.driver.core.ExceptionCode;
import com.datastax.driver.core.ExecutionInfo;
import com.datastax.driver.core.Frame;
import com.datastax.driver.core.FrameCompressor;
import com.datastax.driver.core.Host;
import com.datastax.driver.core.HostConnectionPool;
import com.datastax.driver.core.Message;
import com.datastax.driver.core.NettyOptions;
import com.datastax.driver.core.OperationTimedOutException;
import com.datastax.driver.core.ProtocolOptions;
import com.datastax.driver.core.ProtocolV1Authenticator;
import com.datastax.driver.core.RequestHandler;
import com.datastax.driver.core.Requests;
import com.datastax.driver.core.Responses;
import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.SSLOptions;
import com.datastax.driver.core.SocketOptions;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.StreamIdGenerator;
import com.datastax.driver.core.SystemProperties;
import com.datastax.driver.core.TransportException;
import com.datastax.driver.core.UnsupportedProtocolVersionException;
import com.datastax.driver.core.exceptions.AuthenticationException;
import com.datastax.driver.core.exceptions.DriverException;
import com.datastax.driver.core.exceptions.DriverInternalError;
import com.datastax.driver.core.utils.MoreFutures;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Lists;
import com.google.common.collect.MapMaker;
import com.google.common.util.concurrent.AbstractFuture;
import com.google.common.util.concurrent.AsyncFunction;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.FutureFallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.SettableFuture;
import com.google.common.util.concurrent.Uninterruptibles;
import io.netty.bootstrap.Bootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInitializer;
import io.netty.channel.ChannelOption;
import io.netty.channel.ChannelPipeline;
import io.netty.channel.EventLoop;
import io.netty.channel.EventLoopGroup;
import io.netty.channel.SimpleChannelInboundHandler;
import io.netty.channel.group.ChannelGroup;
import io.netty.channel.group.DefaultChannelGroup;
import io.netty.channel.socket.SocketChannel;
import io.netty.handler.ssl.SslHandler;
import io.netty.handler.timeout.IdleState;
import io.netty.handler.timeout.IdleStateEvent;
import io.netty.handler.timeout.IdleStateHandler;
import io.netty.util.Timeout;
import io.netty.util.Timer;
import io.netty.util.TimerTask;
import io.netty.util.concurrent.EventExecutor;
import io.netty.util.concurrent.GenericFutureListener;
import io.netty.util.concurrent.GlobalEventExecutor;
import java.lang.ref.WeakReference;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.nio.channels.ClosedChannelException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import javax.net.ssl.SSLEngine;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class Connection {
    private static final Logger logger = LoggerFactory.getLogger(Connection.class);
    private static final byte[] EMPTY_BYTE_ARRAY = new byte[0];
    private static final boolean DISABLE_COALESCING = SystemProperties.getBoolean("com.datastax.driver.DISABLE_COALESCING", false);
    final AtomicReference<State> state = new AtomicReference<State>(State.OPEN);
    volatile long maxIdleTime;
    public final InetSocketAddress address;
    private final String name;
    @VisibleForTesting
    volatile Channel channel;
    private final Factory factory;
    @VisibleForTesting
    final Dispatcher dispatcher = new Dispatcher();
    public final AtomicInteger inFlight = new AtomicInteger(0);
    private final AtomicInteger writer = new AtomicInteger(0);
    private volatile String keyspace;
    private volatile boolean isInitialized;
    private final AtomicBoolean isDefunct = new AtomicBoolean();
    private final AtomicBoolean signaled = new AtomicBoolean();
    private final AtomicReference<ConnectionCloseFuture> closeFuture = new AtomicReference();
    private final AtomicReference<Owner> ownerRef = new AtomicReference();
    private static final ConcurrentMap<EventLoop, Flusher> flusherLookup = new MapMaker().concurrencyLevel(16).weakKeys().makeMap();
    private static final ResponseCallback HEARTBEAT_CALLBACK = new ResponseCallback(){

        @Override
        public Message.Request request() {
            return new Requests.Options();
        }

        @Override
        public int retryCount() {
            return 0;
        }

        @Override
        public void onSet(Connection connection, Message.Response response, long latency, int retryCount) {
            switch (response.type) {
                case SUPPORTED: {
                    logger.debug("{} heartbeat query succeeded", (Object)connection);
                    break;
                }
                default: {
                    this.fail(connection, new ConnectionException(connection.address, "Unexpected heartbeat response: " + response));
                }
            }
        }

        @Override
        public void onException(Connection connection, Exception exception, long latency, int retryCount) {
        }

        @Override
        public boolean onTimeout(Connection connection, long latency, int retryCount) {
            this.fail(connection, new ConnectionException(connection.address, "Heartbeat query timed out"));
            return true;
        }

        private void fail(Connection connection, Exception e) {
            connection.defunct(e);
        }
    };

    protected Connection(String name, InetSocketAddress address, Factory factory, Owner owner) {
        this.address = address;
        this.factory = factory;
        this.name = name;
        this.ownerRef.set(owner);
    }

    Connection(String name, InetSocketAddress address, Factory factory) {
        this(name, address, factory, null);
    }

    public ListenableFuture<Void> initAsync() {
        if (this.factory.isShutdown) {
            return Futures.immediateFailedFuture((Throwable)new ConnectionException(this.address, "Connection factory is shut down"));
        }
        int protocolVersion = this.factory.protocolVersion == 1 ? 1 : 2;
        final SettableFuture channelReadyFuture = SettableFuture.create();
        try {
            Bootstrap bootstrap = this.factory.newBootstrap();
            ProtocolOptions protocolOptions = this.factory.configuration.getProtocolOptions();
            bootstrap.handler((ChannelHandler)new Initializer(this, protocolVersion, protocolOptions.getCompression().compressor(), protocolOptions.getSSLOptions(), this.factory.configuration.getPoolingOptions().getHeartbeatIntervalSeconds(), this.factory.configuration.getNettyOptions()));
            ChannelFuture future = bootstrap.connect((SocketAddress)this.address);
            this.writer.incrementAndGet();
            future.addListener((GenericFutureListener)new ChannelFutureListener(){

                public void operationComplete(ChannelFuture future) throws Exception {
                    Connection.this.writer.decrementAndGet();
                    Connection.this.channel = future.channel();
                    if (Connection.this.isClosed()) {
                        Connection.this.channel.close().addListener((GenericFutureListener)new ChannelFutureListener(){

                            public void operationComplete(ChannelFuture future) throws Exception {
                                channelReadyFuture.setException((Throwable)new TransportException(Connection.this.address, "Connection closed during initialization."));
                            }
                        });
                    } else {
                        Connection.this.factory.allChannels.add((Object)Connection.this.channel);
                        if (!future.isSuccess()) {
                            if (logger.isDebugEnabled()) {
                                logger.debug(String.format("%s Error connecting to %s%s", Connection.this, Connection.this.address, Connection.extractMessage(future.cause())));
                            }
                            channelReadyFuture.setException((Throwable)new TransportException(Connection.this.address, "Cannot connect", future.cause()));
                        } else {
                            logger.debug("{} Connection established, initializing transport", (Object)Connection.this);
                            Connection.this.channel.closeFuture().addListener((GenericFutureListener)new ChannelCloseListener());
                            channelReadyFuture.set(null);
                        }
                    }
                }
            });
        }
        catch (RuntimeException e) {
            this.closeAsync().force();
            throw e;
        }
        Executor initExecutor = this.factory.manager.configuration.getPoolingOptions().getInitializationExecutor();
        ListenableFuture initializeTransportFuture = Futures.transform((ListenableFuture)channelReadyFuture, this.onChannelReady(protocolVersion, initExecutor), (Executor)initExecutor);
        ListenableFuture initFuture = Futures.withFallback((ListenableFuture)initializeTransportFuture, (FutureFallback)new FutureFallback<Void>(){

            public ListenableFuture<Void> create(Throwable t) throws Exception {
                SettableFuture future = SettableFuture.create();
                if (t instanceof ClusterNameMismatchException || t instanceof UnsupportedProtocolVersionException) {
                    Connection.this.closeAsync().force();
                    future.setException(t);
                } else {
                    Exception e = t instanceof ConnectionException || t instanceof DriverException || t instanceof InterruptedException ? (Exception)t : new ConnectionException(Connection.this.address, String.format("Unexpected error during transport initialization (%s)", t), t);
                    future.setException((Throwable)Connection.this.defunct(e));
                }
                return future;
            }
        }, (Executor)initExecutor);
        Futures.addCallback((ListenableFuture)initFuture, (FutureCallback)new MoreFutures.FailureCallback<Void>(){

            public void onFailure(Throwable t) {
                if (!Connection.this.isClosed()) {
                    Connection.this.closeAsync().force();
                }
            }
        }, (Executor)initExecutor);
        return initFuture;
    }

    private static String extractMessage(Throwable t) {
        if (t == null) {
            return "";
        }
        String msg = t.getMessage() == null || t.getMessage().isEmpty() ? t.toString() : t.getMessage();
        return " (" + msg + ')';
    }

    private AsyncFunction<Void, Void> onChannelReady(final int protocolVersion, final Executor initExecutor) {
        return new AsyncFunction<Void, Void>(){

            public ListenableFuture<Void> apply(Void input) throws Exception {
                ProtocolOptions.Compression compression = ((Connection)Connection.this).factory.configuration.getProtocolOptions().getCompression();
                Future startupResponseFuture = Connection.this.write(new Requests.Startup(compression));
                return Futures.transform((ListenableFuture)startupResponseFuture, (AsyncFunction)Connection.this.onStartupResponse(protocolVersion, initExecutor), (Executor)initExecutor);
            }
        };
    }

    private AsyncFunction<Message.Response, Void> onStartupResponse(final int protocolVersion, final Executor initExecutor) {
        return new AsyncFunction<Message.Response, Void>(){

            public ListenableFuture<Void> apply(Message.Response response) throws Exception {
                switch (response.type) {
                    case READY: {
                        return Connection.this.checkClusterName(initExecutor);
                    }
                    case ERROR: {
                        Responses.Error error = (Responses.Error)response;
                        if (error.code == ExceptionCode.PROTOCOL_ERROR && error.message.contains("Invalid or unsupported protocol version")) {
                            throw Connection.this.unsupportedProtocolVersionException(protocolVersion);
                        }
                        throw new TransportException(Connection.this.address, String.format("Error initializing connection: %s", error.message));
                    }
                    case AUTHENTICATE: {
                        Authenticator authenticator = ((Connection)Connection.this).factory.authProvider.newAuthenticator(Connection.this.address);
                        if (protocolVersion == 1) {
                            if (authenticator instanceof ProtocolV1Authenticator) {
                                return Connection.this.authenticateV1(authenticator, initExecutor);
                            }
                            return Connection.this.authenticateV2(authenticator, initExecutor);
                        }
                        return Connection.this.authenticateV2(authenticator, initExecutor);
                    }
                }
                throw new TransportException(Connection.this.address, String.format("Unexpected %s response message from server to a STARTUP message", new Object[]{response.type}));
            }
        };
    }

    private ListenableFuture<Void> checkClusterName(Executor executor) {
        final String expected = this.factory.manager.metadata.clusterName;
        if (expected == null) {
            this.markInitialized();
            return MoreFutures.VOID_SUCCESS;
        }
        DefaultResultSetFuture clusterNameFuture = new DefaultResultSetFuture(null, new Requests.Query("select cluster_name from system.local"));
        try {
            this.write(clusterNameFuture);
            return Futures.transform((ListenableFuture)clusterNameFuture, (AsyncFunction)new AsyncFunction<ResultSet, Void>(){

                public ListenableFuture<Void> apply(ResultSet rs) throws Exception {
                    Row row = rs.one();
                    String actual = row.getString("cluster_name");
                    if (!expected.equals(actual)) {
                        throw new ClusterNameMismatchException(Connection.this.address, actual, expected);
                    }
                    Connection.this.markInitialized();
                    return MoreFutures.VOID_SUCCESS;
                }
            }, (Executor)executor);
        }
        catch (Exception e) {
            return Futures.immediateFailedFuture((Throwable)e);
        }
    }

    private void markInitialized() {
        this.isInitialized = true;
        Host.statesLogger.debug("[{}] {} Transport initialized, connection ready", (Object)this.address, (Object)this);
    }

    private ListenableFuture<Void> authenticateV1(Authenticator authenticator, final Executor executor) {
        Requests.Credentials creds = new Requests.Credentials(((ProtocolV1Authenticator)((Object)authenticator)).getCredentials());
        try {
            Future authResponseFuture = this.write(creds);
            return Futures.transform((ListenableFuture)authResponseFuture, (AsyncFunction)new AsyncFunction<Message.Response, Void>(){

                public ListenableFuture<Void> apply(Message.Response authResponse) throws Exception {
                    switch (authResponse.type) {
                        case READY: {
                            return Connection.this.checkClusterName(executor);
                        }
                        case ERROR: {
                            throw new AuthenticationException(Connection.this.address, ((Responses.Error)authResponse).message);
                        }
                    }
                    throw new TransportException(Connection.this.address, String.format("Unexpected %s response message from server to a CREDENTIALS message", new Object[]{authResponse.type}));
                }
            }, (Executor)executor);
        }
        catch (Exception e) {
            return Futures.immediateFailedFuture((Throwable)e);
        }
    }

    private ListenableFuture<Void> authenticateV2(Authenticator authenticator, Executor executor) {
        byte[] initialResponse = authenticator.initialResponse();
        if (null == initialResponse) {
            initialResponse = EMPTY_BYTE_ARRAY;
        }
        try {
            Future authResponseFuture = this.write(new Requests.AuthResponse(initialResponse));
            return Futures.transform((ListenableFuture)authResponseFuture, this.onV2AuthResponse(authenticator, executor), (Executor)executor);
        }
        catch (Exception e) {
            return Futures.immediateFailedFuture((Throwable)e);
        }
    }

    private AsyncFunction<Message.Response, Void> onV2AuthResponse(final Authenticator authenticator, final Executor executor) {
        return new AsyncFunction<Message.Response, Void>(){

            public ListenableFuture<Void> apply(Message.Response authResponse) throws Exception {
                switch (authResponse.type) {
                    case AUTH_SUCCESS: {
                        logger.trace("{} Authentication complete", (Object)this);
                        authenticator.onAuthenticationSuccess(((Responses.AuthSuccess)authResponse).token);
                        return Connection.this.checkClusterName(executor);
                    }
                    case AUTH_CHALLENGE: {
                        byte[] responseToServer = authenticator.evaluateChallenge(((Responses.AuthChallenge)authResponse).token);
                        if (responseToServer == null) {
                            logger.trace("{} Authentication complete (No response to server)", (Object)this);
                            return Connection.this.checkClusterName(executor);
                        }
                        logger.trace("{} Sending Auth response to challenge", (Object)this);
                        Future nextResponseFuture = Connection.this.write(new Requests.AuthResponse(responseToServer));
                        return Futures.transform((ListenableFuture)nextResponseFuture, (AsyncFunction)Connection.this.onV2AuthResponse(authenticator, executor), (Executor)executor);
                    }
                    case ERROR: {
                        String message = ((Responses.Error)authResponse).message;
                        if (message.startsWith("java.lang.ArrayIndexOutOfBoundsException: 15")) {
                            message = String.format("Cannot use authenticator %s with protocol version 1, only plain text authentication is supported with this protocol version", authenticator);
                        }
                        throw new AuthenticationException(Connection.this.address, message);
                    }
                }
                throw new TransportException(Connection.this.address, String.format("Unexpected %s response message from server to authentication message", new Object[]{authResponse.type}));
            }
        };
    }

    private UnsupportedProtocolVersionException unsupportedProtocolVersionException(int triedVersion) {
        logger.debug("Got unsupported protocol version error from {} for version {}", (Object)this.address, (Object)triedVersion);
        return new UnsupportedProtocolVersionException(this.address, triedVersion);
    }

    public boolean isDefunct() {
        return this.isDefunct.get();
    }

    public int maxAvailableStreams() {
        return this.dispatcher.streamIdHandler.maxAvailableStreams();
    }

    <E extends Exception> E defunct(E e) {
        if (this.isDefunct.compareAndSet(false, true)) {
            if (Host.statesLogger.isTraceEnabled()) {
                Host.statesLogger.trace("Defuncting " + this, e);
            } else if (Host.statesLogger.isDebugEnabled()) {
                Host.statesLogger.debug("Defuncting {} because: {}", (Object)this, (Object)e.getMessage());
            }
            Host host = this.factory.manager.metadata.getHost(this.address);
            if (host != null) {
                boolean decrement = this.signaled.compareAndSet(false, true);
                boolean hostDown = host.convictionPolicy.signalConnectionFailure(this, decrement);
                if (hostDown) {
                    this.factory.manager.signalHostDown(host, host.wasJustAdded());
                } else {
                    this.notifyOwnerWhenDefunct();
                }
            }
            this.closeAsync().force();
        }
        return e;
    }

    private void notifyOwnerWhenDefunct() {
        if (!this.isInitialized) {
            return;
        }
        Owner owner = this.ownerRef.get();
        if (owner != null) {
            owner.onConnectionDefunct(this);
        }
    }

    public String keyspace() {
        return this.keyspace;
    }

    public void setKeyspace(String keyspace) throws ConnectionException {
        if (keyspace == null) {
            return;
        }
        if (this.keyspace != null && this.keyspace.equals(keyspace)) {
            return;
        }
        try {
            Uninterruptibles.getUninterruptibly(this.setKeyspaceAsync(keyspace));
        }
        catch (ConnectionException e) {
            throw this.defunct(e);
        }
        catch (BusyConnectionException e) {
            logger.warn("Tried to set the keyspace on busy {}. This should not happen but is not critical (it will be retried)", (Object)this);
            throw new ConnectionException(this.address, "Tried to set the keyspace on busy connection");
        }
        catch (ExecutionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof OperationTimedOutException) {
                logger.warn("Timeout while setting keyspace on {}. This should not happen but is not critical (it will be retried)", (Object)this);
                throw new ConnectionException(this.address, "Timeout while setting keyspace on connection");
            }
            throw this.defunct(new ConnectionException(this.address, "Error while setting keyspace", cause));
        }
    }

    ListenableFuture<Void> setKeyspaceAsync(String keyspace) throws ConnectionException, BusyConnectionException {
        logger.trace("{} Setting keyspace {}", (Object)this, (Object)keyspace);
        Future future = this.write(new Requests.Query("USE \"" + keyspace + '\"'));
        return Futures.transform((ListenableFuture)future, (AsyncFunction)new AsyncFunction<Message.Response, Void>(){

            public ListenableFuture<Void> apply(Message.Response response) throws Exception {
                if (response instanceof Responses.Result.SetKeyspace) {
                    Connection.this.keyspace = ((Responses.Result.SetKeyspace)response).keyspace;
                    return MoreFutures.VOID_SUCCESS;
                }
                if (response.type == Message.Response.Type.ERROR) {
                    Responses.Error error = (Responses.Error)response;
                    throw Connection.this.defunct(error.asException(Connection.this.address));
                }
                throw Connection.this.defunct(new DriverInternalError("Unexpected response while setting keyspace: " + response));
            }
        }, (Executor)this.factory.manager.configuration.getPoolingOptions().getInitializationExecutor());
    }

    public Future write(Message.Request request) throws ConnectionException, BusyConnectionException {
        Future future = new Future(request);
        this.write(future);
        return future;
    }

    public ResponseHandler write(ResponseCallback callback) throws ConnectionException, BusyConnectionException {
        return this.write(callback, true);
    }

    public ResponseHandler write(ResponseCallback callback, boolean startTimeout) throws ConnectionException, BusyConnectionException {
        ResponseHandler handler = new ResponseHandler(this, callback);
        this.dispatcher.add(handler);
        Message.Request request = callback.request().setStreamId(handler.streamId);
        if (this.isDefunct.get()) {
            this.dispatcher.removeHandler(handler, true);
            throw new ConnectionException(this.address, "Write attempt on defunct connection");
        }
        if (this.isClosed()) {
            this.dispatcher.removeHandler(handler, true);
            throw new ConnectionException(this.address, "Connection has been closed");
        }
        logger.trace("{}, stream {}, writing request {}", new Object[]{this, request.getStreamId(), request});
        this.writer.incrementAndGet();
        if (DISABLE_COALESCING) {
            this.channel.writeAndFlush((Object)request).addListener((GenericFutureListener)this.writeHandler(request, handler));
        } else {
            this.flush(new FlushItem(this.channel, request, this.writeHandler(request, handler)));
        }
        if (startTimeout) {
            handler.startTimeout();
        }
        return handler;
    }

    private ChannelFutureListener writeHandler(final Message.Request request, final ResponseHandler handler) {
        return new ChannelFutureListener(){

            public void operationComplete(ChannelFuture writeFuture) {
                Connection.this.writer.decrementAndGet();
                if (!writeFuture.isSuccess()) {
                    logger.debug("{}, stream {}, Error writing request {}", new Object[]{Connection.this, request.getStreamId(), request});
                    Connection.this.dispatcher.removeHandler(handler, true);
                    final TransportException ce = writeFuture.cause() instanceof ClosedChannelException ? new TransportException(Connection.this.address, "Error writing: Closed channel") : new TransportException(Connection.this.address, "Error writing", writeFuture.cause());
                    final long latency = System.nanoTime() - handler.startTime;
                    ListeningExecutorService executor = ((Connection)Connection.this).factory.manager.executor;
                    if (!executor.isShutdown()) {
                        executor.execute(new Runnable(){

                            @Override
                            public void run() {
                                handler.callback.onException(Connection.this, Connection.this.defunct(ce), latency, handler.retryCount);
                            }
                        });
                    }
                } else {
                    logger.trace("{}, stream {}, request sent successfully", (Object)Connection.this, (Object)request.getStreamId());
                }
            }
        };
    }

    boolean hasOwner() {
        return this.ownerRef.get() != null;
    }

    boolean setOwner(Owner owner) {
        return this.ownerRef.compareAndSet(null, owner);
    }

    void release() {
        Owner owner = this.ownerRef.get();
        if (owner instanceof HostConnectionPool) {
            ((HostConnectionPool)owner).returnConnection(this);
        }
    }

    public boolean isClosed() {
        return this.closeFuture.get() != null;
    }

    public CloseFuture closeAsync() {
        boolean terminated;
        Host host;
        ConnectionCloseFuture future = new ConnectionCloseFuture();
        if (!this.closeFuture.compareAndSet(null, future)) {
            return this.closeFuture.get();
        }
        logger.debug("{} closing connection", (Object)this);
        if (this.signaled.compareAndSet(false, true) && (host = this.factory.manager.metadata.getHost(this.address)) != null) {
            host.convictionPolicy.signalConnectionClosed(this);
        }
        if (!(terminated = this.tryTerminate(false))) {
            long terminateTime = System.currentTimeMillis() + 2L * this.factory.getReadTimeoutMillis();
            this.factory.reaper.register(this, terminateTime);
        }
        return future;
    }

    boolean tryTerminate(boolean force) {
        assert (this.isClosed());
        ConnectionCloseFuture future = this.closeFuture.get();
        if (future.isDone()) {
            logger.debug("{} has already terminated", (Object)this);
            return true;
        }
        if (force || this.dispatcher.pending.isEmpty()) {
            if (force) {
                logger.warn("Forcing termination of {}. This should not happen and is likely a bug, please report.", (Object)this);
            }
            future.force();
            return true;
        }
        logger.debug("Not terminating {}: there are still pending requests", (Object)this);
        return false;
    }

    public String toString() {
        return String.format("Connection[%s, inFlight=%d, closed=%b]", this.name, this.inFlight.get(), this.isClosed());
    }

    private void flush(FlushItem item) {
        Flusher alt;
        EventLoop loop = item.channel.eventLoop();
        Flusher flusher = (Flusher)flusherLookup.get(loop);
        if (flusher == null && (alt = flusherLookup.putIfAbsent(loop, flusher = new Flusher(loop))) != null) {
            flusher = alt;
        }
        flusher.queued.add(item);
        flusher.start();
    }

    static interface Owner {
        public void onConnectionDefunct(Connection var1);
    }

    private static class Initializer
    extends ChannelInitializer<SocketChannel> {
        private static final Message.ProtocolDecoder messageDecoder = new Message.ProtocolDecoder();
        private static final Message.ProtocolEncoder messageEncoderV1 = new Message.ProtocolEncoder(1);
        private static final Message.ProtocolEncoder messageEncoderV2 = new Message.ProtocolEncoder(2);
        private static final Frame.Encoder frameEncoder = new Frame.Encoder();
        private final int protocolVersion;
        private final Connection connection;
        private final FrameCompressor compressor;
        private final SSLOptions sslOptions;
        private final NettyOptions nettyOptions;
        private final ChannelHandler idleStateHandler;

        public Initializer(Connection connection, int protocolVersion, FrameCompressor compressor, SSLOptions sslOptions, int heartBeatIntervalSeconds, NettyOptions nettyOptions) {
            this.connection = connection;
            this.protocolVersion = protocolVersion;
            this.compressor = compressor;
            this.sslOptions = sslOptions;
            this.nettyOptions = nettyOptions;
            this.idleStateHandler = new IdleStateHandler(0, 0, heartBeatIntervalSeconds);
        }

        protected void initChannel(SocketChannel channel) throws Exception {
            ChannelPipeline pipeline = channel.pipeline();
            if (this.sslOptions != null) {
                SSLEngine engine = this.sslOptions.context.createSSLEngine();
                engine.setUseClientMode(true);
                engine.setEnabledCipherSuites(this.sslOptions.cipherSuites);
                SslHandler handler = new SslHandler(engine);
                pipeline.addLast("ssl", (ChannelHandler)handler);
            }
            pipeline.addLast("frameDecoder", (ChannelHandler)new Frame.Decoder());
            pipeline.addLast("frameEncoder", (ChannelHandler)frameEncoder);
            if (this.compressor != null) {
                pipeline.addLast("frameDecompressor", (ChannelHandler)new Frame.Decompressor(this.compressor));
                pipeline.addLast("frameCompressor", (ChannelHandler)new Frame.Compressor(this.compressor));
            }
            pipeline.addLast("messageDecoder", (ChannelHandler)messageDecoder);
            pipeline.addLast("messageEncoder", (ChannelHandler)(this.protocolVersion == 1 ? messageEncoderV1 : messageEncoderV2));
            pipeline.addLast("idleStateHandler", this.idleStateHandler);
            pipeline.addLast("dispatcher", (ChannelHandler)this.connection.dispatcher);
            this.nettyOptions.afterChannelInitialized(channel);
        }
    }

    public static interface DefaultResponseHandler {
        public void handle(Message.Response var1);
    }

    static class ResponseHandler {
        public final Connection connection;
        public final int streamId;
        public final ResponseCallback callback;
        public final int retryCount;
        private final long startTime;
        private volatile Timeout timeout;
        private final AtomicBoolean isCancelled = new AtomicBoolean();

        public ResponseHandler(Connection connection, ResponseCallback callback) throws BusyConnectionException {
            this.connection = connection;
            this.streamId = connection.dispatcher.streamIdHandler.next();
            this.callback = callback;
            this.retryCount = callback.retryCount();
            this.startTime = System.nanoTime();
        }

        void startTimeout() {
            long timeoutMs = this.connection.factory.getReadTimeoutMillis();
            this.timeout = timeoutMs <= 0L ? null : ((Connection)this.connection).factory.timer.newTimeout(this.onTimeoutTask(), timeoutMs, TimeUnit.MILLISECONDS);
        }

        void cancelTimeout() {
            if (this.timeout != null) {
                this.timeout.cancel();
            }
        }

        public boolean cancelHandler() {
            if (!this.isCancelled.compareAndSet(false, true)) {
                return false;
            }
            this.connection.dispatcher.removeHandler(this, false);
            return true;
        }

        private TimerTask onTimeoutTask() {
            return new TimerTask(){

                public void run(Timeout timeout) {
                    if (ResponseHandler.this.callback.onTimeout(ResponseHandler.this.connection, System.nanoTime() - ResponseHandler.this.startTime, ResponseHandler.this.retryCount)) {
                        ResponseHandler.this.cancelHandler();
                    }
                }
            };
        }
    }

    static interface ResponseCallback {
        public Message.Request request();

        public int retryCount();

        public void onSet(Connection var1, Message.Response var2, long var3, int var5);

        public void onException(Connection var1, Exception var2, long var3, int var5);

        public boolean onTimeout(Connection var1, long var2, int var4);
    }

    static class Future
    extends AbstractFuture<Message.Response>
    implements RequestHandler.Callback {
        private final Message.Request request;
        private volatile InetSocketAddress address;

        public Future(Message.Request request) {
            this.request = request;
        }

        @Override
        public void register(RequestHandler handler) {
        }

        @Override
        public Message.Request request() {
            return this.request;
        }

        @Override
        public int retryCount() {
            return 0;
        }

        @Override
        public void onSet(Connection connection, Message.Response response, ExecutionInfo info, Statement statement, long latency) {
            this.onSet(connection, response, latency, 0);
        }

        @Override
        public void onSet(Connection connection, Message.Response response, long latency, int retryCount) {
            this.address = connection.address;
            super.set((Object)response);
        }

        @Override
        public void onException(Connection connection, Exception exception, long latency, int retryCount) {
            if (connection != null) {
                this.address = connection.address;
            }
            super.setException((Throwable)exception);
        }

        @Override
        public boolean onTimeout(Connection connection, long latency, int retryCount) {
            assert (connection != null);
            this.address = connection.address;
            return super.setException((Throwable)new OperationTimedOutException(connection.address));
        }

        public InetSocketAddress getAddress() {
            return this.address;
        }
    }

    private class ConnectionCloseFuture
    extends CloseFuture {
        private ConnectionCloseFuture() {
        }

        @Override
        public ConnectionCloseFuture force() {
            if (Connection.this.channel == null) {
                this.set(null);
                return this;
            }
            Connection.this.dispatcher.errorOutAllHandler(new TransportException(Connection.this.address, "Connection has been closed"));
            ChannelFuture future = Connection.this.channel.close();
            future.addListener((GenericFutureListener)new ChannelFutureListener(){

                public void operationComplete(ChannelFuture future) {
                    Connection.this.factory.allChannels.remove((Object)Connection.this.channel);
                    if (future.cause() != null) {
                        logger.warn("Error closing channel", future.cause());
                        ConnectionCloseFuture.this.setException(future.cause());
                    } else {
                        ConnectionCloseFuture.this.set(null);
                    }
                }
            });
            return this;
        }
    }

    private class ChannelCloseListener
    implements ChannelFutureListener {
        private ChannelCloseListener() {
        }

        public void operationComplete(ChannelFuture future) throws Exception {
            if (!Connection.this.isInitialized || Connection.this.isClosed()) {
                Connection.this.dispatcher.errorOutAllHandler(new TransportException(Connection.this.address, "Channel has been closed"));
                Connection.this.closeAsync().force();
            } else {
                Connection.this.defunct(new TransportException(Connection.this.address, "Channel has been closed"));
            }
        }
    }

    class Dispatcher
    extends SimpleChannelInboundHandler<Message.Response> {
        public final StreamIdGenerator streamIdHandler = new StreamIdGenerator();
        private final ConcurrentMap<Integer, ResponseHandler> pending = new ConcurrentHashMap<Integer, ResponseHandler>();

        Dispatcher() {
        }

        public void add(ResponseHandler handler) {
            ResponseHandler old = this.pending.put(handler.streamId, handler);
            assert (old == null);
        }

        public void removeHandler(ResponseHandler handler, boolean releaseStreamId) {
            boolean removed;
            if (!releaseStreamId) {
                this.streamIdHandler.mark(handler.streamId);
            }
            if (!(removed = this.pending.remove(handler.streamId, handler))) {
                if (!releaseStreamId) {
                    this.streamIdHandler.unmark(handler.streamId);
                }
                return;
            }
            handler.cancelTimeout();
            if (releaseStreamId) {
                this.streamIdHandler.release(handler.streamId);
            }
            if (Connection.this.isClosed()) {
                Connection.this.tryTerminate(false);
            }
        }

        protected void channelRead0(ChannelHandlerContext ctx, Message.Response response) throws Exception {
            int streamId = response.getStreamId();
            if (logger.isTraceEnabled()) {
                logger.trace("{}, stream {}, received: {}", new Object[]{Connection.this, streamId, this.asDebugString(response)});
            }
            if (streamId < 0) {
                ((Connection)Connection.this).factory.defaultHandler.handle(response);
                return;
            }
            ResponseHandler handler = (ResponseHandler)this.pending.remove(streamId);
            this.streamIdHandler.release(streamId);
            if (handler == null) {
                this.streamIdHandler.unmark(streamId);
                if (logger.isDebugEnabled()) {
                    logger.debug("{} Response received on stream {} but no handler set anymore (either the request has timed out or it was closed due to another error). Received message is {}", new Object[]{Connection.this, streamId, this.asDebugString(response)});
                }
                return;
            }
            handler.cancelTimeout();
            handler.callback.onSet(Connection.this, response, System.nanoTime() - handler.startTime, handler.retryCount);
            if (Connection.this.isClosed()) {
                Connection.this.tryTerminate(false);
            }
        }

        public void userEventTriggered(ChannelHandlerContext ctx, Object evt) throws Exception {
            if (!Connection.this.isClosed() && evt instanceof IdleStateEvent && ((IdleStateEvent)evt).state() == IdleState.ALL_IDLE) {
                logger.debug("{} was inactive for {} seconds, sending heartbeat", (Object)Connection.this, (Object)((Connection)Connection.this).factory.configuration.getPoolingOptions().getHeartbeatIntervalSeconds());
                Connection.this.write(HEARTBEAT_CALLBACK);
            }
        }

        private String asDebugString(Object obj) {
            if (obj == null) {
                return "null";
            }
            String msg = obj.toString();
            if (msg.length() < 500) {
                return msg;
            }
            return msg.substring(0, 500) + "... [message of size " + msg.length() + " truncated]";
        }

        public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) throws Exception {
            if (logger.isDebugEnabled()) {
                logger.debug(String.format("%s connection error", Connection.this), cause);
            }
            if (Connection.this.writer.get() > 0) {
                return;
            }
            Connection.this.defunct(new TransportException(Connection.this.address, String.format("Unexpected exception triggered (%s)", cause), cause));
        }

        public void errorOutAllHandler(ConnectionException ce) {
            Iterator iter = this.pending.values().iterator();
            while (iter.hasNext()) {
                ResponseHandler handler = (ResponseHandler)iter.next();
                handler.cancelTimeout();
                handler.callback.onException(Connection.this, ce, System.nanoTime() - handler.startTime, handler.retryCount);
                iter.remove();
            }
        }
    }

    private static class FlushItem {
        final Channel channel;
        final Object request;
        final ChannelFutureListener listener;

        private FlushItem(Channel channel, Object request, ChannelFutureListener listener) {
            this.channel = channel;
            this.request = request;
            this.listener = listener;
        }
    }

    private static final class Flusher
    implements Runnable {
        final WeakReference<EventLoop> eventLoopRef;
        final Queue<FlushItem> queued = new ConcurrentLinkedQueue<FlushItem>();
        final AtomicBoolean running = new AtomicBoolean(false);
        final HashSet<Channel> channels = new HashSet();
        int runsWithNoWork = 0;

        private Flusher(EventLoop eventLoop) {
            this.eventLoopRef = new WeakReference<EventLoop>(eventLoop);
        }

        void start() {
            EventLoop eventLoop;
            if (!this.running.get() && this.running.compareAndSet(false, true) && (eventLoop = (EventLoop)this.eventLoopRef.get()) != null) {
                eventLoop.execute((Runnable)this);
            }
        }

        @Override
        public void run() {
            FlushItem flush;
            boolean doneWork = false;
            while (null != (flush = this.queued.poll())) {
                Channel channel = flush.channel;
                if (!channel.isActive()) continue;
                this.channels.add(channel);
                channel.write(flush.request).addListener((GenericFutureListener)flush.listener);
                doneWork = true;
            }
            for (Channel channel : this.channels) {
                channel.flush();
            }
            this.channels.clear();
            if (doneWork) {
                this.runsWithNoWork = 0;
            } else if (++this.runsWithNoWork > 5) {
                this.running.set(false);
                if (this.queued.isEmpty() || !this.running.compareAndSet(false, true)) {
                    return;
                }
            }
            EventLoop eventLoop = (EventLoop)this.eventLoopRef.get();
            if (eventLoop != null && !eventLoop.isShuttingDown()) {
                eventLoop.schedule((Runnable)this, 10000L, TimeUnit.NANOSECONDS);
            }
        }
    }

    public static class Factory {
        public final Timer timer;
        final EventLoopGroup eventLoopGroup;
        private final Class<? extends Channel> channelClass;
        private final ChannelGroup allChannels = new DefaultChannelGroup((EventExecutor)GlobalEventExecutor.INSTANCE);
        private final ConcurrentMap<Host, AtomicInteger> idGenerators = new ConcurrentHashMap<Host, AtomicInteger>();
        public final DefaultResponseHandler defaultHandler;
        final Cluster.Manager manager;
        final Cluster.ConnectionReaper reaper;
        public final Configuration configuration;
        public final AuthProvider authProvider;
        private volatile boolean isShutdown;
        volatile int protocolVersion;
        private final NettyOptions nettyOptions;

        Factory(Cluster.Manager manager, Configuration configuration) {
            this.defaultHandler = manager;
            this.manager = manager;
            this.reaper = manager.reaper;
            this.configuration = configuration;
            this.authProvider = configuration.getProtocolOptions().getAuthProvider();
            this.protocolVersion = configuration.getProtocolOptions().initialProtocolVersion;
            this.nettyOptions = configuration.getNettyOptions();
            this.eventLoopGroup = this.nettyOptions.eventLoopGroup(manager.threadFactory("nio-worker"));
            this.channelClass = this.nettyOptions.channelClass();
            this.timer = this.nettyOptions.timer(manager.threadFactory("timeouter"));
        }

        public int getPort() {
            return this.configuration.getProtocolOptions().getPort();
        }

        public Connection open(Host host) throws ConnectionException, InterruptedException, UnsupportedProtocolVersionException, ClusterNameMismatchException {
            InetSocketAddress address = host.getSocketAddress();
            if (this.isShutdown) {
                throw new ConnectionException(address, "Connection factory is shut down");
            }
            host.convictionPolicy.signalConnectionsOpening(1);
            Connection connection = new Connection(this.buildConnectionName(host), address, this);
            try {
                connection.initAsync().get();
                return connection;
            }
            catch (ExecutionException e) {
                throw Factory.launderAsyncInitException(e);
            }
        }

        public Connection open(HostConnectionPool pool) throws ConnectionException, InterruptedException, UnsupportedProtocolVersionException, ClusterNameMismatchException {
            pool.host.convictionPolicy.signalConnectionsOpening(1);
            Connection connection = new Connection(this.buildConnectionName(pool.host), pool.host.getSocketAddress(), this, pool);
            try {
                connection.initAsync().get();
                return connection;
            }
            catch (ExecutionException e) {
                throw Factory.launderAsyncInitException(e);
            }
        }

        public List<Connection> newConnections(HostConnectionPool pool, int count) {
            pool.host.convictionPolicy.signalConnectionsOpening(count);
            ArrayList connections = Lists.newArrayListWithCapacity((int)count);
            for (int i = 0; i < count; ++i) {
                connections.add(new Connection(this.buildConnectionName(pool.host), pool.host.getSocketAddress(), this, pool));
            }
            return connections;
        }

        private String buildConnectionName(Host host) {
            return host.getSocketAddress().toString() + '-' + this.getIdGenerator(host).getAndIncrement();
        }

        static RuntimeException launderAsyncInitException(ExecutionException e) throws ConnectionException, InterruptedException, UnsupportedProtocolVersionException, ClusterNameMismatchException {
            Throwable t = e.getCause();
            if (t instanceof ConnectionException) {
                throw (ConnectionException)t;
            }
            if (t instanceof InterruptedException) {
                throw (InterruptedException)t;
            }
            if (t instanceof UnsupportedProtocolVersionException) {
                throw (UnsupportedProtocolVersionException)t;
            }
            if (t instanceof ClusterNameMismatchException) {
                throw (ClusterNameMismatchException)t;
            }
            if (t instanceof DriverException) {
                throw (DriverException)t;
            }
            return new RuntimeException("Unexpected exception during connection initialization", t);
        }

        private AtomicInteger getIdGenerator(Host host) {
            AtomicInteger old;
            AtomicInteger g = (AtomicInteger)this.idGenerators.get(host);
            if (g == null && (old = this.idGenerators.putIfAbsent(host, g = new AtomicInteger(1))) != null) {
                g = old;
            }
            return g;
        }

        public long getReadTimeoutMillis() {
            return this.configuration.getSocketOptions().getReadTimeoutMillis();
        }

        private Bootstrap newBootstrap() {
            Integer sendBufferSize;
            Integer receiveBufferSize;
            Boolean tcpNoDelay;
            Integer soLinger;
            Boolean reuseAddress;
            Bootstrap b = new Bootstrap();
            ((Bootstrap)b.group(this.eventLoopGroup)).channel(this.channelClass);
            SocketOptions options = this.configuration.getSocketOptions();
            b.option(ChannelOption.CONNECT_TIMEOUT_MILLIS, (Object)options.getConnectTimeoutMillis());
            Boolean keepAlive = options.getKeepAlive();
            if (keepAlive != null) {
                b.option(ChannelOption.SO_KEEPALIVE, (Object)keepAlive);
            }
            if ((reuseAddress = options.getReuseAddress()) != null) {
                b.option(ChannelOption.SO_REUSEADDR, (Object)reuseAddress);
            }
            if ((soLinger = options.getSoLinger()) != null) {
                b.option(ChannelOption.SO_LINGER, (Object)soLinger);
            }
            if ((tcpNoDelay = options.getTcpNoDelay()) != null) {
                b.option(ChannelOption.TCP_NODELAY, (Object)tcpNoDelay);
            }
            if ((receiveBufferSize = options.getReceiveBufferSize()) != null) {
                b.option(ChannelOption.SO_RCVBUF, (Object)receiveBufferSize);
            }
            if ((sendBufferSize = options.getSendBufferSize()) != null) {
                b.option(ChannelOption.SO_SNDBUF, (Object)sendBufferSize);
            }
            this.nettyOptions.afterBootstrapInitialized(b);
            return b;
        }

        public void shutdown() {
            this.isShutdown = true;
            this.allChannels.close().awaitUninterruptibly();
            this.nettyOptions.onClusterClose(this.eventLoopGroup);
            this.nettyOptions.onClusterClose(this.timer);
        }
    }

    static enum State {
        OPEN,
        TRASHED,
        RESURRECTING,
        GONE;

    }
}

