package com.css.sdk.cservice.adapter;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.icu.text.UFormat;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.BaseAdapter;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.css.sdk.R;
import com.css.sdk.cservice.activity.CssVideoPlayActivity;
import com.css.sdk.cservice.activity.PreviewActivity;
import com.css.sdk.cservice.base.Constants;
import com.css.sdk.cservice.base.utils.LogUtils;
import com.css.sdk.cservice.data.Chat;
import com.css.sdk.cservice.data.OrderDetail;
import com.css.sdk.cservice.imageloader.ImageLoader;
import com.css.sdk.cservice.userdata.GlobalData;
import com.luck.picture.lib.PictureSelector;
import com.luck.picture.lib.PictureVideoPlayActivity;
import com.luck.picture.lib.config.PictureConfig;
import com.luck.picture.lib.style.PictureParameterStyle;

import java.util.ArrayList;

public class ChatAdapter extends BaseAdapter {

    private Context mContext;
    private OrderDetail detail;
    private LayoutInflater inflater;

    private final int ASK = 0;
    private final int ANSWER = 1;
    private final int HINT = 2;

    public ChatAdapter(Context mContext, OrderDetail detail) {
        this.mContext = mContext;
        this.detail = detail;
        this.inflater = LayoutInflater.from(mContext);
    }


    /**
     * 当重新提交问题时候被调用
     *
     * @param detail
     */
    public void bindData(OrderDetail detail) {
        this.detail = detail;
    }

    @Override
    public int getCount() {
        if (detail == null || detail.chats == null) {
            return 0;
        }

        if (detail.status.equals(Constants.QA_STATUS_WAITING)
                || detail.status.equals(Constants.QA_STATUS_WAITING_APPEND)
                || detail.status.equals(Constants.QA_STATUS_CLOSED)
                || detail.status.equals(Constants.QA_STATUS_CLOSED_TIMEOUT)
                || detail.status.equals(Constants.QA_STATUS_PROCESSING)) {
            return detail.chats.size() + 1;
        } else {
            return detail.chats.size();
        }
    }

    @Override
    public Object getItem(int position) {
        if (detail == null || detail.chats == null) {
            return null;
        }
        if (getCount() > detail.chats.size() && position == getCount() - 1) {
            return null;
        }
        return detail.chats.get(position);
    }

    @Override
    public long getItemId(int position) {
        return position;
    }

    @Override
    public View getView(int position, View convertView, ViewGroup parent) {
        switch (getItemViewType(position)) {
            case HINT:
                return handleHint(convertView);
            case ASK:
                return handleChat(position, convertView, true);
            case ANSWER:
                return handleChat(position, convertView, false);
        }
        return convertView;
    }

    private View handleHint(View convertView) {
        HintViewHolder hintViewHolder = null;
        if (convertView == null) {
            convertView = inflater.inflate(R.layout.css_question_detail_item_hint, null);
            hintViewHolder = new HintViewHolder(convertView);
            convertView.setTag(hintViewHolder);
        } else {
            hintViewHolder = (HintViewHolder) convertView.getTag();
        }
        if (detail.status.equals(Constants.QA_STATUS_WAITING) || detail.status.equals(Constants.QA_STATUS_WAITING_APPEND)) {
            hintViewHolder.hint.setText(R.string.css_string_fqa_status_waiting_or_append);
        } else if (detail.status.equals(Constants.QA_STATUS_PROCESSING)) {
            hintViewHolder.hint.setText(R.string.css_string_fqa_status_processing);
        } else if (detail.status.equals(Constants.QA_STATUS_CLOSED_TIMEOUT)) {
            hintViewHolder.hint.setText(R.string.css_string_fqa_status_close);
        } else if (detail.status.equals(Constants.QA_STATUS_CLOSED)) {
            hintViewHolder.hint.setText(R.string.css_string_fqa_status_close_by_yourself);
        }
        return convertView;
    }

    private View handleChat(int position, View convertView, boolean isAsk) {
        ChatViewHolder viewHolder = null;
        if (convertView == null) {
            if (isAsk) {
                convertView = inflater.inflate(R.layout.css_question_detail_chat_ask, null);
            } else {
                convertView = inflater.inflate(R.layout.css_question_detail_chat_answer, null);
            }

            viewHolder = new ChatViewHolder(convertView);
            convertView.setTag(viewHolder);
        } else {
            viewHolder = (ChatViewHolder) convertView.getTag();
        }

        Chat chat = detail.chats.get(position);
        if ((chat.pics != null && chat.pics.size() > 0) || (chat.videos != null && chat.videos.size() > 0)) {
            viewHolder.picsLl.setVisibility(View.VISIBLE);
            dealMedias(viewHolder, chat);
        } else {
            viewHolder.picsLl.setVisibility(View.GONE);
        }

        if (getItemViewType(position) == ASK) {
            viewHolder.original.setTextColor(mContext.getResources().getColor(R.color.css_color_black_1));
        } else if (getItemViewType(position) == ANSWER) {
            viewHolder.original.setTextColor(mContext.getResources().getColor(R.color.css_color_black_1));
        }
        viewHolder.original.setText(chat.original);
        if (TextUtils.isEmpty(chat.translation)) {
            viewHolder.translationLl.setVisibility(View.GONE);
        } else {
            viewHolder.translationLl.setVisibility(View.VISIBLE);
            viewHolder.translation.setText(chat.translation);

            int width_o = (int) android.text.Layout.getDesiredWidth(viewHolder.original.getText(), viewHolder.original.getPaint());
            int width_t = (int) android.text.Layout.getDesiredWidth(viewHolder.translation.getText(), viewHolder.translation.getPaint());

            ViewGroup.LayoutParams params = viewHolder.line.getLayoutParams();
            params.width = width_o > width_t ? width_o : width_t;
            viewHolder.line.setLayoutParams(params);
        }
        return convertView;
    }

    private void dealMedias(ChatViewHolder holder, final Chat chat) {
        // 计算视频和图片的数量
        int mediaNumber = chat.pics.size() + chat.videos.size();
        setImageVisible(holder, mediaNumber);
        showMedia(holder, chat);
    }

    /**
     * 控制如何显示图片和视频
     *
     * @param holder
     * @param chat
     */
    private void showMedia(ChatViewHolder holder, Chat chat) {
        // 优先选择布置图片
        int picSize = chat.pics.size();
        int videoSize = chat.videos.size();
        if (picSize > 0) {
            for (int i = 0; i < picSize; i++) {
                ImageLoader.getInstance().load(chat.pics.get(i), holder.mediaHolderImages[i]);
                holder.mediaHolderImages[i].setTag(chat.pics.get(i));
                int position = i;
                holder.mediaHolderImages[i].setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        toPreviewPicActivity(chat.pics, position);
                    }
                });
            }
        }
        if (videoSize > 0) {
            for (int i = picSize == 0 ? 0 : picSize; i < picSize + videoSize; i++) {
                holder.mediaHolderImages[i].setImageResource(R.drawable.css_video_holder);
                holder.mediaHolderImages[i].setTag(chat.videos.get(picSize > 0 ? i - picSize : i));
                String path = chat.videos.get(picSize > 0 ? i - picSize : i);
                holder.mediaHolderImages[i].setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        toPreviewVideoActivity(path);
                    }
                });
            }
        }

    }

    /**
     * 控制显示几个
     *
     * @param holder
     * @param mediaNumber
     */
    private void setImageVisible(ChatViewHolder holder, int mediaNumber) {
        holder.pic1.setVisibility(View.VISIBLE);
        holder.pic2.setVisibility(mediaNumber > 1 ? View.VISIBLE : View.GONE);
        holder.pic3.setVisibility(mediaNumber > 2 ? View.VISIBLE : View.GONE);
    }


    private void toPreviewPicActivity(ArrayList<String> images, int position) {
        if (images != null && images.size() > 0) {
            Intent intent = new Intent(mContext, PreviewActivity.class);
            Bundle b = new Bundle();
            b.putStringArrayList(Constants.KEY_ALL_PICS, images);
            b.putInt(Constants.KEY_POSITION, position);
            b.putInt(Constants.KEY_PREVIEW_TYPE, Constants.PREVIEW_TYPE_VIEW);
            intent.putExtras(b);
            mContext.startActivity(intent);
        }
    }

    private void toPreviewVideoActivity(String path) {
        if (TextUtils.isEmpty(path)) {
            LogUtils.w("invalidated data of video", null);
            return;
        }
        Intent intent = new Intent(mContext, CssVideoPlayActivity.class);
        intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        intent.putExtra(CssVideoPlayActivity.VIDEO_PATH, path);
        mContext.startActivity(intent);
    }


    class HintViewHolder {
        TextView hint;

        public HintViewHolder(View view) {
            this.hint = view.findViewById(R.id.css_question_detail_hint);
        }
    }

    class ChatViewHolder {
        TextView original;
        TextView translation;
        LinearLayout translationLl;
        LinearLayout picsLl;
        ImageView pic1;
        ImageView pic2;
        ImageView pic3;
        ImageView[] mediaHolderImages = new ImageView[3];
        View line;

        public ChatViewHolder(View view) {
            original = view.findViewById(R.id.css_chat_original_tv);
            translation = view.findViewById(R.id.css_chat_translation_tv);
            translationLl = view.findViewById(R.id.css_chat_translation_ll);
            picsLl = view.findViewById(R.id.css_chat_pic_ll);
            line = view.findViewById(R.id.css_chat_line);
            pic1 = view.findViewById(R.id.css_chat_pic1);
            pic2 = view.findViewById(R.id.css_chat_pic2);
            pic3 = view.findViewById(R.id.css_chat_pic3);
            mediaHolderImages[0] = pic1;
            mediaHolderImages[1] = pic2;
            mediaHolderImages[2] = pic3;
        }
    }

    @Override
    public int getViewTypeCount() {
        return 3;
    }

    public int getItemViewType(int position) {
        if (getCount() > detail.chats.size() && position == getCount() - 1) {
            return HINT;
        }
        if (detail.chats.get(position).send.equals("ask")) {
            return ASK;
        } else {
            return ANSWER;
        }
    }
}
