package com.css.sdk.cservice.imageloader.core;

import android.annotation.TargetApi;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.media.ThumbnailUtils;
import android.net.Uri;
import android.os.Build;
import android.provider.MediaStore;
import android.webkit.MimeTypeMap;
import android.widget.ImageView;

import com.css.sdk.cservice.utils.ImageUtils;
import com.css.sdk.cservice.utils.IoUtils;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;

public class ImageDecorder {
    public static class ImageDecorderParams {
        String url;
        ImageUtils.ImageSize actualSize;
        ImageUtils.ImageSize expectSize;
        ImageView imageView;
    }

    /**
     * {@value}
     */
    public static final int DEFAULT_HTTP_CONNECT_TIMEOUT = 5 * 1000;
    /**
     * {@value}
     */
    public static final int DEFAULT_HTTP_READ_TIMEOUT = 20 * 1000;

    /**
     * {@value}
     */
    protected static final int BUFFER_SIZE = 32 * 1024; // 32 Kb
    /**
     * {@value}
     */
    protected static final String ALLOWED_URI_CHARS = "@#&=*+-_.,:!?()/~'%";

    protected static final int MAX_REDIRECT_COUNT = 5;

    protected final Context context;
    protected final int connectTimeout;
    protected final int readTimeout;

    public ImageDecorder(Context context) {
        this(context, DEFAULT_HTTP_CONNECT_TIMEOUT, DEFAULT_HTTP_READ_TIMEOUT);
    }

    public ImageDecorder(Context context, int connectTimeout, int readTimeout) {
        this.context = context.getApplicationContext();
        this.connectTimeout = connectTimeout;
        this.readTimeout = readTimeout;
    }

    public Bitmap decodeByStream(InputStream imageStream, ImageDecorderParams params) throws IOException {
        if (imageStream == null) {
            return null;
        }
        //根据imageStream获得实际的宽高
        params.actualSize = ImageUtils.getImageSize(imageStream);
        //重置流，企图压缩
        imageStream = resetStream(imageStream, params.url);
        return decodeBitmap(imageStream, params);
    }

    public Bitmap decode(ImageDecorderParams params) throws IOException {
        String imageUrl = params.url;
        InputStream imageStream = getImageStream(imageUrl);
        return decodeByStream(imageStream, params);

    }

    protected InputStream resetStream(InputStream imageStream, String url) throws IOException {
        if (imageStream.markSupported()) {
            try {
                imageStream.reset();
                return imageStream;
            } catch (IOException ignored) {
            }
        }
        IoUtils.closeSilently(imageStream);
        return getImageStream(url);
    }

    private Bitmap decodeBitmap(InputStream imageStream, ImageDecorderParams params) {
        Bitmap bitmap = null;
        try {
            BitmapFactory.Options options = new BitmapFactory.Options();
            options.inJustDecodeBounds = false;
            options.inPreferredConfig = Bitmap.Config.RGB_565;
            options.inSampleSize = ImageUtils.calculateInSampleSize(params.actualSize, params.expectSize);
            bitmap = BitmapFactory.decodeStream(imageStream, null, options);
        } catch (Exception e) {

        } finally {
            IoUtils.closeSilently(imageStream);
        }
        return bitmap;
    }


    public InputStream getImageStream(String imageUri) throws IOException {
        if (imageUri.startsWith("https") || imageUri.startsWith("http")) {
            return getStreamFromNetwork(imageUri);
        } else {
            return getStreamFromFile(imageUri);
        }
    }

    //Http/Https
    protected InputStream getStreamFromNetwork(String imageUri) throws IOException {
        HttpURLConnection conn = createConnection(imageUri);

        int redirectCount = 0;
        while (conn.getResponseCode() / 100 == 3 && redirectCount < MAX_REDIRECT_COUNT) {
            conn = createConnection(conn.getHeaderField("Location"));
            redirectCount++;
        }

        InputStream imageStream;
        try {
            imageStream = conn.getInputStream();
        } catch (IOException e) {
            IoUtils.readAndCloseStream(conn.getErrorStream());
            throw e;
        }
        if (!shouldBeProcessed(conn)) {
            IoUtils.closeSilently(imageStream);
            throw new IOException("Image request failed with response code " + conn.getResponseCode());
        }
        return new ContentLengthInputStream(new BufferedInputStream(imageStream, BUFFER_SIZE), conn.getContentLength());
    }


    protected HttpURLConnection createConnection(String url) throws IOException {
        String encodedUrl = Uri.encode(url, ALLOWED_URI_CHARS);
        HttpURLConnection conn = (HttpURLConnection) new URL(encodedUrl).openConnection();
        conn.setConnectTimeout(connectTimeout);
        conn.setReadTimeout(readTimeout);
        return conn;
    }

    protected boolean shouldBeProcessed(HttpURLConnection conn) throws IOException {
        return conn.getResponseCode() == 200;
    }


    protected InputStream getStreamFromFile(String imageUri) throws IOException {
        if (isVideoFileUri(imageUri)) {
            return getVideoThumbnailStream(imageUri);
        } else {
            BufferedInputStream imageStream = new BufferedInputStream(new FileInputStream(imageUri), BUFFER_SIZE);
            return new ContentLengthInputStream(imageStream, (int) new File(imageUri).length());
        }
    }

    private boolean isVideoFileUri(String uri) {
        String extension = MimeTypeMap.getFileExtensionFromUrl(uri);
        String mimeType = MimeTypeMap.getSingleton().getMimeTypeFromExtension(extension);
        return mimeType != null && mimeType.startsWith("video/");
    }


    @TargetApi(Build.VERSION_CODES.FROYO)
    private InputStream getVideoThumbnailStream(String filePath) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.FROYO) {
            Bitmap bitmap = ThumbnailUtils
                    .createVideoThumbnail(filePath, MediaStore.Images.Thumbnails.FULL_SCREEN_KIND);
            if (bitmap != null) {
                ByteArrayOutputStream bos = new ByteArrayOutputStream();
                bitmap.compress(Bitmap.CompressFormat.PNG, 0, bos);
                return new ByteArrayInputStream(bos.toByteArray());
            }
        }
        return null;
    }
}
