package com.css.sdk.cservice.refresh;

import android.animation.ObjectAnimator;
import android.content.Context;
import android.os.Handler;
import android.support.v4.widget.ListViewCompat;
import android.util.AttributeSet;
import android.util.TypedValue;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewConfiguration;
import android.view.ViewGroup;
import android.view.animation.DecelerateInterpolator;
import android.widget.ListView;
import android.widget.Scroller;

import com.css.sdk.R;

public class RefreshLayout extends ViewGroup {
    /**
     * 头、尾高度，dp
     */
    public int REFRESH_VIEW_HEIGHT_DP = 50;

    static final int REFRESHING = R.string.css_string_refreshing;
    static final int RELEASE_REFRESH = R.string.css_string_release_to_refresh;
    static final int PULL_DOWN = R.string.css_string_pull_to_refresh;
    static final int REFRESH_FINISH = R.string.css_string_load_completed;

    /**
     * 无操作
     */
    private final int DRAG_ACTION_NULL = 0x00;

    /**
     * 下拉
     */
    private final int DRAG_ACTION_PULL_DOWN = 0x01;

    /**
     * 下拉返回
     */
    private final int DRAG_ACTION_PULL_DOWN_BACK = 0x02;
    /**
     * 拖拽行为动作
     */
    private int dragAction = DRAG_ACTION_NULL;


    private Context mContext;

    Scroller scroller;
    boolean isRefreshing = false;
    /**
     * 刷新高度的偏移量，dp
     */
    private int refreshHeightOffset = 10;
    /**
     * 处于刷新状态时头view的高度
     */
    private int headViewRefreshingHeight;

    /**
     * 头view可允许下拉刷新时划过的距离
     */
    private int headViewAccessRefreshDistance;
    private static final float DECELERATE_INTERPOLATION_FACTOR = 2f;
    private DecelerateInterpolator mDecelerateInterpolator;

    private int mTouchSlop;

    public RefreshLayout(Context context) {
        super(context);
        scroller = new Scroller(context);
    }

    public RefreshLayout(Context context, AttributeSet attrs) {
        super(context, attrs);
        mContext = context;
        mTouchSlop = ViewConfiguration.get(context).getScaledTouchSlop();
        REFRESH_VIEW_HEIGHT_DP = dp2px(mContext, REFRESH_VIEW_HEIGHT_DP);
        refreshHeightOffset = dp2px(mContext, refreshHeightOffset);
        headViewRefreshingHeight = REFRESH_VIEW_HEIGHT_DP;
        headViewAccessRefreshDistance = headViewRefreshingHeight + refreshHeightOffset;
        mDecelerateInterpolator = new DecelerateInterpolator(DECELERATE_INTERPOLATION_FACTOR);
        scroller = new Scroller(context,mDecelerateInterpolator);
        addHead();
    }

    private View headView;
    private ListView listview;

    private void addHead() {
        HeadManager headManager = new HeadManager(mContext, this);
        addView(headManager.getHeadView());
    }

    public RefreshLayout(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
    }

    //onMeasure函数，若我们的控件都是固定值或match_parent时，系统提供了默认的实现方式，如果有wrap_content的情况，我们必须要重写该函数
    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec);
        measureChildren(widthMeasureSpec, heightMeasureSpec);//即对子view进行测量，如果没有这一句，我们的子View不会显示。
    }

    @Override
    protected void onLayout(boolean b, int i, int i1, int i2, int i3) {
        if (getChildCount() < 2) {
            return;
        }
        //开始布局，headerview不显示，下拉时才显示
        headView = getChildAt(0);
        headView.layout(0, -headView.getMeasuredHeight(), headView.getMeasuredWidth(), 0);
        listview = (ListView) getChildAt(1);
        listview.layout(0, 0, listview.getMeasuredWidth(), listview.getMeasuredHeight());
    }


    private float initialDownY;
    private float initialDownX;

    //重点，onInterceptTouchEvent负责拦截触摸事件
    @Override
    public boolean onInterceptTouchEvent(MotionEvent ev) {
        switch (ev.getAction()) {
            case MotionEvent.ACTION_DOWN:
                initialDownY = ev.getRawY();
                break;
            case MotionEvent.ACTION_MOVE:
                float dy = ev.getRawY() - initialDownY;
                initialDownY = ev.getRawY();
                //判断当前内容View是否还能继续向上滑动，利用ViewCompat的canScrollVertically判断view是否还能在竖直方向滑动，对应还有canScrollHorizontal
                boolean a = canChildScrollUp();
                if (dy > mTouchSlop && !a) {
                    return true;
                }
                break;
            case MotionEvent.ACTION_UP:
        }
        return super.onInterceptTouchEvent(ev);
    }

    private boolean canChildScrollUp() {
        return ListViewCompat.canScrollList(listview, -1);
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        switch (event.getAction()) {
            case MotionEvent.ACTION_DOWN:
                pullDownDistance = 0;
                initialDownY = event.getY();
            case MotionEvent.ACTION_MOVE:
                int dy = (int) (event.getRawY() - initialDownY);
                if (!isRefreshing) {
                    scrollBy(0, -dy / 2);
                    if (dy > 0) {
                        onPullDown(dy);
                    } else {
                        onPullDownBack(dy);
                    }
                }
                initialDownY = event.getRawY();
                break;
            case MotionEvent.ACTION_UP:
                release();
                break;
            default:
                break;
        }
        return super.onTouchEvent(event);
    }

    /**
     * 手指释放
     */
    private void release() {
        if (dragAction == DRAG_ACTION_PULL_DOWN || dragAction == DRAG_ACTION_PULL_DOWN_BACK) {
            //如果拖拽的距离大于可刷新距离
            if (pullDownDistance >= headViewAccessRefreshDistance) {
                isRefreshing = true;
                if (onRefreshListener != null) {
                    onRefreshListener.refresh();
                }
                if (onChangeViewTip != null) {
                    onChangeViewTip.changeHeadTips(REFRESHING);
                }
                adjustRefreshPosition();
            } else {
                if (!isRefreshing) {
                    scroller.startScroll(0, getScrollY(), 0, -getScrollY());
                    cancelRefresh();
                    invalidate();
                }
            }
        }
    }

    /**
     * 刷新后恢复原始位置所用时间
     */
    private final int ANIMATOR_TIME = 300;


    private int getPullDownDistance() {
        return pullDownDistance;
    }

    private void setPullDownDistance(int pullDownDistance) {
        this.pullDownDistance = pullDownDistance;
        if (onChangeViewHeight != null) {
            onChangeViewHeight.changeHeadViewHeight(pullDownDistance);
        }
    }

    /**
     * 调整刷新位置
     */
    private void adjustRefreshPosition() {
        ObjectAnimator objectAnimator = ObjectAnimator.ofInt(this, "pullDownDistance", pullDownDistance, headViewRefreshingHeight);
        objectAnimator.setDuration(ANIMATOR_TIME);
        objectAnimator.start();
    }

    private void cancelRefresh() {
        ObjectAnimator objectAnimator = ObjectAnimator.ofInt(this, "pullDownDistance", pullDownDistance, 0);
        objectAnimator.setDuration(ANIMATOR_TIME);
        objectAnimator.start();
    }

    private int pullDownDistance = 0;

    /**
     * 下拉刷新
     */
    private void onPullDown(int dy) {
        dragAction = DRAG_ACTION_PULL_DOWN;
        pullDownDistance += Math.abs(dy) / 2;
        changeTipsRefresh();
        if (onChangeViewHeight != null) {
            onChangeViewHeight.changeHeadViewHeight(pullDownDistance);
        }
    }

    /**
     * 下拉刷新返回
     */
    private void onPullDownBack(int dy) {
        dragAction = DRAG_ACTION_PULL_DOWN_BACK;
        pullDownDistance -= Math.abs(dy) / 2;
        changeTipsRefresh();
        if (onChangeViewHeight != null) {
            onChangeViewHeight.changeHeadViewHeight(pullDownDistance);
        }
    }

    private void changeTipsRefresh() {
        if (pullDownDistance >= headViewAccessRefreshDistance) {
            if (onChangeViewTip != null) {
                onChangeViewTip.changeHeadTips(RELEASE_REFRESH);
            }
        } else {
            if (onChangeViewTip != null) {
                onChangeViewTip.changeHeadTips(PULL_DOWN);
            }
        }
    }

    @Override
    public void computeScroll() {
        if (scroller.computeScrollOffset()) {
            scrollTo(scroller.getCurrX(), scroller.getCurrY());
            invalidate();
        }
    }

    //必须重写该方法。其实就是不做处理即可，因为在recyclerview中，如果recyclerview监听到自己在滑动时，会调用该父View的该方法来禁用父view拦截事件的功能，这样我们就无法监听了，因此在这里我们不做处理，写为空。
    @Override
    public void requestDisallowInterceptTouchEvent(boolean disallowIntercept) {

    }

    /**
     * 停止刷新
     */
    private void stopRefresh() {
        ObjectAnimator objectAnimator = ObjectAnimator.ofInt(this, "pullDownDistance", pullDownDistance, 0);
        objectAnimator.setDuration(ANIMATOR_TIME);
        objectAnimator.start();
    }

    Handler handler = new Handler() {
        public void handleMessage(android.os.Message msg) {
            if (msg.what == 1) {
                stopRefresh();
                scroller.startScroll(0, getScrollY(), 0, -getScrollY());
                invalidate();
            }
        }

        ;
    };

    //更新成功时调用的方法。
    public void finishRefresh() {
        isRefreshing = false;
        if (onChangeViewTip != null) {
            onChangeViewTip.changeHeadTips(REFRESH_FINISH);
        }
        new Thread(new Runnable() {
            @Override
            public void run() {
                handler.sendEmptyMessageDelayed(1, 500);
            }
        }).start();
    }

    private Swipe.OnChangeViewHeight onChangeViewHeight;

    protected void setOnChangeViewHeight(Swipe.OnChangeViewHeight onChangeViewHeight) {
        this.onChangeViewHeight = onChangeViewHeight;
    }

    private Swipe.OnChangeViewTip onChangeViewTip;

    protected void setOnChangeViewTip(Swipe.OnChangeViewTip onChangeViewTip) {
        this.onChangeViewTip = onChangeViewTip;
    }

    private Swipe.OnRefreshListener onRefreshListener;

    public void setOnRefreshListener(Swipe.OnRefreshListener onRefreshListener) {
        this.onRefreshListener = onRefreshListener;
    }

    /**
     * dp转px
     */
    private int dp2px(Context context, float dpValue) {
        return (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, dpValue, context.getResources().getDisplayMetrics());
    }
}
