package com.css.sdk.cservice.imageloader.core;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.Handler;
import android.os.Message;
import android.widget.ImageView;

import com.css.sdk.cservice.imageloader.disk.LruDiskCache;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.util.concurrent.locks.ReentrantLock;

public class LoadImageTask implements Runnable{

    public static final int MSG_LOAD_SUCCESS = 0x110;
    public static final int MSG_LOAD_FAIL = MSG_LOAD_SUCCESS + 1;

    protected Handler mUiHandler;
    protected ImageRequest request;
    private ImageDecorder mImageDecorder;
    private LruDiskCache mDisCache;

    public LoadImageTask(Context context, ImageRequest request, Handler uiHandler, LruDiskCache diskCache) {
        this.mUiHandler = uiHandler;
        this.request = request;
        mImageDecorder = new ImageDecorder(context);
        this.mDisCache = diskCache;
    }

    @Override
    public void run() {
        Bitmap bitmap = null;
        File diskCache = mDisCache.get(request.getUrl());
        ReentrantLock loadFromUriLock = request.loadFromUriLock;
        loadFromUriLock.lock();
        String imageUrl = request.getUrl();
        InputStream imageStream;
        try {
            if (diskCache != null && diskCache.exists()) {
                bitmap = mImageDecorder.decodeByStream(new FileInputStream(diskCache), buildDecodeParams(request));
            } else {
                imageStream = mImageDecorder.getImageStream(imageUrl);
                //存储到硬盘缓存
                if(imageUrl.startsWith("http")||imageUrl.startsWith("https")){
                    saveToDisk(request.getUrl(), imageStream);
                }
                //网络加载
                bitmap = mImageDecorder.decodeByStream(imageStream, buildDecodeParams(request));
            }

            if (bitmap != null) {
                request.setBitmap(bitmap);
                sendSuccessMsg(request);
            } else {
                sendErrorMsg(request);
            }

        } catch (Exception e) {
            e.printStackTrace();
            sendErrorMsg(request);
        }finally {
            loadFromUriLock.unlock();
        }
    }

    private void saveToDisk(String url, InputStream imageStream) throws Exception{
        BitmapFactory.Options options = new BitmapFactory.Options();
        options.inSampleSize=1;
        Bitmap bm = BitmapFactory.decodeStream(imageStream,null, options);
        mDisCache.save(url,bm);
    }

    protected void sendSuccessMsg(ImageRequest request)
    {
        Message msg = Message.obtain(null, MSG_LOAD_SUCCESS);
        msg.obj = request;
        mUiHandler.sendMessage(msg);
    }

    protected void sendErrorMsg(ImageRequest request)
    {
        Message msg = Message.obtain(null, MSG_LOAD_FAIL);
        msg.obj = request;
        mUiHandler.sendMessage(msg);
    }

    protected ImageDecorder.ImageDecorderParams buildDecodeParams(
            ImageRequest request)
    {
        ImageDecorder.ImageDecorderParams params = new ImageDecorder.ImageDecorderParams();
        params.imageView = (ImageView) request.getTarget();
        params.expectSize = request.getExpectSize();
        params.url = request.getUrl();
        return params;
    }
}
