/*
 * Decompiled with CFR 0.152.
 */
package com.couchbase.client.core.cnc;

import com.couchbase.client.core.annotation.Stability;
import com.couchbase.client.core.cnc.EventBus;
import com.couchbase.client.core.cnc.events.tracing.OrphanRecordDroppedEvent;
import com.couchbase.client.core.cnc.events.tracing.OrphanReporterFailureDetectedEvent;
import com.couchbase.client.core.cnc.events.tracing.OrphansRecordedEvent;
import com.couchbase.client.core.deps.org.jctools.queues.MpscUnboundedArrayQueue;
import com.couchbase.client.core.env.OrphanReporterConfig;
import com.couchbase.client.core.logging.RedactableArgument;
import com.couchbase.client.core.msg.Request;
import com.couchbase.client.core.msg.kv.KeyValueRequest;
import com.couchbase.client.core.msg.view.ViewRequest;
import com.couchbase.client.core.service.ServiceType;
import com.couchbase.client.core.util.HostAndPort;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.Queue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import reactor.core.publisher.Mono;

@Stability.Internal
public class OrphanReporter {
    private static final AtomicInteger ORPHAN_REPORTER_ID = new AtomicInteger();
    private final AtomicBoolean running = new AtomicBoolean(false);
    private final Thread worker;
    private final Queue<Request<?>> orphanQueue;
    private final long emitIntervalNanos;
    private final int sampleSize;
    private final EventBus eventBus;

    public OrphanReporter(EventBus eventBus, OrphanReporterConfig config) {
        this.eventBus = eventBus;
        this.orphanQueue = new MpscUnboundedArrayQueue(config.queueLength());
        this.emitIntervalNanos = config.emitInterval().toNanos();
        this.sampleSize = config.sampleSize();
        this.worker = new Thread(new Worker());
        this.worker.setDaemon(true);
    }

    public Mono<Void> start() {
        return Mono.defer(() -> {
            if (this.running.compareAndSet(false, true)) {
                this.worker.start();
            }
            return Mono.empty();
        });
    }

    public Mono<Void> stop(Duration timeout) {
        return Mono.defer(() -> {
            if (this.running.compareAndSet(true, false)) {
                this.worker.interrupt();
            }
            return Mono.empty();
        });
    }

    public void report(Request<?> request) {
        if (!this.orphanQueue.offer(request)) {
            this.eventBus.publish(new OrphanRecordDroppedEvent(request.getClass()));
        }
    }

    private class Worker
    implements Runnable {
        private final long workerSleepMs = Long.parseLong(System.getProperty("com.couchbase.orphanReporterSleep", "100"));
        private final Comparator<Request<?>> THRESHOLD_COMPARATOR = Comparator.comparingLong(o -> o.context().logicalRequestLatency());
        private long lastThresholdLog;
        private boolean hasThresholdWritten;
        private final Queue<Request<?>> kvOrphans = new PriorityQueue(this.THRESHOLD_COMPARATOR);
        private final Queue<Request<?>> queryOrphans = new PriorityQueue(this.THRESHOLD_COMPARATOR);
        private final Queue<Request<?>> viewOrphans = new PriorityQueue(this.THRESHOLD_COMPARATOR);
        private final Queue<Request<?>> searchOrphans = new PriorityQueue(this.THRESHOLD_COMPARATOR);
        private final Queue<Request<?>> analyticsOrphans = new PriorityQueue(this.THRESHOLD_COMPARATOR);
        private long kvOrphanCount = 0L;
        private long queryOrphanCount = 0L;
        private long viewOrphanCount = 0L;
        private long searchOrphanCount = 0L;
        private long analyticsOrphanCount = 0L;

        private Worker() {
        }

        @Override
        public void run() {
            Thread.currentThread().setName("cb-orphan-" + ORPHAN_REPORTER_ID.incrementAndGet());
            while (OrphanReporter.this.running.get()) {
                try {
                    this.handleOrphanQueue();
                    Thread.sleep(this.workerSleepMs);
                }
                catch (InterruptedException ex) {
                    if (OrphanReporter.this.running.get()) continue;
                    return;
                }
                catch (Exception ex) {
                    OrphanReporter.this.eventBus.publish(new OrphanReporterFailureDetectedEvent(ex));
                }
            }
        }

        private void handleOrphanQueue() {
            long now = System.nanoTime();
            if (now - this.lastThresholdLog > OrphanReporter.this.emitIntervalNanos) {
                this.prepareAndLogOrphans();
                this.lastThresholdLog = now;
            }
            Request request;
            while ((request = (Request)OrphanReporter.this.orphanQueue.poll()) != null) {
                ServiceType serviceType = request.serviceType();
                if (serviceType == ServiceType.KV) {
                    this.updateSet(this.kvOrphans, request);
                    ++this.kvOrphanCount;
                    continue;
                }
                if (serviceType == ServiceType.QUERY) {
                    this.updateSet(this.queryOrphans, request);
                    ++this.queryOrphanCount;
                    continue;
                }
                if (serviceType == ServiceType.VIEWS) {
                    this.updateSet(this.viewOrphans, request);
                    ++this.viewOrphanCount;
                    continue;
                }
                if (serviceType == ServiceType.SEARCH) {
                    this.updateSet(this.searchOrphans, request);
                    ++this.searchOrphanCount;
                    continue;
                }
                if (serviceType != ServiceType.ANALYTICS) continue;
                this.updateSet(this.analyticsOrphans, request);
                ++this.analyticsOrphanCount;
            }
            return;
        }

        private void updateSet(Queue<Request<?>> set, Request<?> request) {
            set.add(request);
            while (set.size() > OrphanReporter.this.sampleSize) {
                set.remove();
            }
            this.hasThresholdWritten = true;
        }

        private void prepareAndLogOrphans() {
            if (!this.hasThresholdWritten) {
                return;
            }
            this.hasThresholdWritten = false;
            ArrayList<Map<String, Object>> output = new ArrayList<Map<String, Object>>();
            if (!this.kvOrphans.isEmpty()) {
                output.add(this.convertOrphanMetadata(this.kvOrphans, this.kvOrphanCount, "kv"));
                this.kvOrphans.clear();
                this.kvOrphanCount = 0L;
            }
            if (!this.queryOrphans.isEmpty()) {
                output.add(this.convertOrphanMetadata(this.queryOrphans, this.queryOrphanCount, "query"));
                this.queryOrphans.clear();
                this.queryOrphanCount = 0L;
            }
            if (!this.viewOrphans.isEmpty()) {
                output.add(this.convertOrphanMetadata(this.viewOrphans, this.viewOrphanCount, "view"));
                this.viewOrphans.clear();
                this.viewOrphanCount = 0L;
            }
            if (!this.searchOrphans.isEmpty()) {
                output.add(this.convertOrphanMetadata(this.searchOrphans, this.searchOrphanCount, "search"));
                this.searchOrphans.clear();
                this.searchOrphanCount = 0L;
            }
            if (!this.analyticsOrphans.isEmpty()) {
                output.add(this.convertOrphanMetadata(this.analyticsOrphans, this.analyticsOrphanCount, "analytics"));
                this.analyticsOrphans.clear();
                this.analyticsOrphanCount = 0L;
            }
            this.logOrphans(output);
        }

        private Map<String, Object> convertOrphanMetadata(Queue<Request<?>> requests, long count, String serviceType) {
            HashMap<String, Object> output = new HashMap<String, Object>();
            ArrayList top = new ArrayList();
            for (Request request : requests) {
                HashMap<String, Object> fieldMap = new HashMap<String, Object>();
                if (request != null) {
                    long serverDuration;
                    String name = request.getClass().getSimpleName().replace("Request", "").toLowerCase();
                    fieldMap.put("s", name);
                    String operationId = request.operationId();
                    if (operationId != null) {
                        fieldMap.put("i", operationId);
                    }
                    if (request instanceof KeyValueRequest) {
                        fieldMap.put("b", ((KeyValueRequest)request).bucket());
                    } else if (request instanceof ViewRequest) {
                        fieldMap.put("b", ((ViewRequest)request).bucket());
                    }
                    String localId = request.context().lastChannelId();
                    if (localId != null) {
                        fieldMap.put("c", RedactableArgument.redactSystem(localId));
                    }
                    HostAndPort local = request.context().lastDispatchedFrom();
                    HostAndPort peer = request.context().lastDispatchedTo();
                    if (local != null) {
                        fieldMap.put("l", RedactableArgument.redactSystem(local.toString()));
                    }
                    if (peer != null) {
                        fieldMap.put("r", RedactableArgument.redactSystem(peer.toString()));
                    }
                    if ((serverDuration = request.context().serverLatency()) > 0L) {
                        fieldMap.put("d", serverDuration);
                    }
                    long timeout = request.timeout().toMillis();
                    fieldMap.put("t", timeout);
                }
                top.add(fieldMap);
            }
            output.put("service", serviceType);
            output.put("count", count);
            output.put("top", top);
            return output;
        }

        void logOrphans(List<Map<String, Object>> toLog) {
            OrphanReporter.this.eventBus.publish(new OrphansRecordedEvent(Duration.ofNanos(OrphanReporter.this.emitIntervalNanos), toLog));
        }
    }
}

