/*
 * Decompiled with CFR 0.152.
 */
package com.clickhouse.client.api.data_formats.internal;

import com.clickhouse.client.api.ClientException;
import com.clickhouse.data.ClickHouseColumn;
import com.clickhouse.data.ClickHouseDataType;
import com.clickhouse.data.ClickHouseEnum;
import com.clickhouse.data.value.ClickHouseBitmap;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Array;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.Period;
import java.time.ZonedDateTime;
import java.time.temporal.TemporalAmount;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import java.util.UUID;
import org.slf4j.Logger;
import org.slf4j.helpers.NOPLogger;

public class BinaryStreamReader {
    public static final Class<?> NO_TYPE_HINT = null;
    private final InputStream input;
    private final Logger log;
    private final TimeZone timeZone;
    private final ByteBufferAllocator bufferAllocator;
    private final boolean jsonAsString;
    private final Class<?> arrayDefaultTypeHint;
    private byte[] int16Buff = new byte[2];
    private byte[] int32Buff = new byte[4];
    private byte[] int64Buff = new byte[8];
    public static final int INT16_SIZE = 2;
    public static final int INT32_SIZE = 4;
    public static final int INT64_SIZE = 8;
    public static final int INT128_SIZE = 16;
    public static final int INT256_SIZE = 32;
    private static final byte[] B1 = new byte[8];
    public static final int[] BASES = new int[]{1, 10, 100, 1000, 10000, 100000, 1000000, 10000000, 100000000, 1000000000};
    private final byte[] STRING_BUFF = new byte[1024];
    private static final Set<Byte> DECIMAL_TAGS = Collections.unmodifiableSet(new HashSet<Byte>(Arrays.asList(ClickHouseDataType.Decimal.getBinTag(), ClickHouseDataType.Decimal32.getBinTag(), ClickHouseDataType.Decimal64.getBinTag(), ClickHouseDataType.Decimal128.getBinTag(), ClickHouseDataType.Decimal256.getBinTag())));
    private static final ClickHouseColumn JSON_PLACEHOLDER_COL = (ClickHouseColumn)ClickHouseColumn.parse((String)"v Dynamic").get(0);

    BinaryStreamReader(InputStream input, TimeZone timeZone, Logger log, ByteBufferAllocator bufferAllocator, boolean jsonAsString, Map<ClickHouseDataType, Class<?>> typeHintMapping) {
        this.log = log == null ? NOPLogger.NOP_LOGGER : log;
        this.timeZone = timeZone;
        this.input = input;
        this.bufferAllocator = bufferAllocator;
        this.jsonAsString = jsonAsString;
        this.arrayDefaultTypeHint = typeHintMapping == null || typeHintMapping.isEmpty() ? NO_TYPE_HINT : typeHintMapping.get(ClickHouseDataType.Array);
    }

    public <T> T readValue(ClickHouseColumn column) throws IOException {
        return this.readValue(column, null);
    }

    public <T> T readValue(ClickHouseColumn column, Class<?> typeHint) throws IOException {
        int isNull;
        if (column.isNullable() && (isNull = BinaryStreamReader.readByteOrEOF(this.input)) == 1) {
            return null;
        }
        ClickHouseColumn actualColumn = column.getDataType() == ClickHouseDataType.Dynamic ? this.readDynamicData() : column;
        ClickHouseDataType dataType = actualColumn.getDataType();
        int precision = actualColumn.getPrecision();
        int scale = actualColumn.getScale();
        TimeZone timezone = actualColumn.getTimeZoneOrDefault(this.timeZone);
        try {
            switch (dataType) {
                case FixedString: {
                    byte[] bytes = precision > this.STRING_BUFF.length ? new byte[precision] : this.STRING_BUFF;
                    BinaryStreamReader.readNBytes(this.input, bytes, 0, precision);
                    return (T)new String(bytes, 0, precision, StandardCharsets.UTF_8);
                }
                case String: {
                    return (T)this.readString();
                }
                case Int8: {
                    return (T)Byte.valueOf(this.readByte());
                }
                case UInt8: {
                    return (T)Short.valueOf(this.readUnsignedByte());
                }
                case Int16: {
                    return (T)Short.valueOf(this.readShortLE());
                }
                case UInt16: {
                    return (T)Integer.valueOf(this.readUnsignedShortLE());
                }
                case Int32: {
                    return (T)Integer.valueOf(this.readIntLE());
                }
                case UInt32: {
                    return (T)Long.valueOf(this.readUnsignedIntLE());
                }
                case Int64: {
                    return (T)Long.valueOf(this.readLongLE());
                }
                case UInt64: {
                    return (T)this.readBigIntegerLE(8, true);
                }
                case Int128: {
                    return (T)this.readBigIntegerLE(16, false);
                }
                case UInt128: {
                    return (T)this.readBigIntegerLE(16, true);
                }
                case Int256: {
                    return (T)this.readBigIntegerLE(32, false);
                }
                case UInt256: {
                    return (T)this.readBigIntegerLE(32, true);
                }
                case Decimal: {
                    return (T)this.readDecimal(precision, scale);
                }
                case Decimal32: {
                    return (T)this.readDecimal(ClickHouseDataType.Decimal32.getMaxPrecision(), scale);
                }
                case Decimal64: {
                    return (T)this.readDecimal(ClickHouseDataType.Decimal64.getMaxPrecision(), scale);
                }
                case Decimal128: {
                    return (T)this.readDecimal(ClickHouseDataType.Decimal128.getMaxPrecision(), scale);
                }
                case Decimal256: {
                    return (T)this.readDecimal(ClickHouseDataType.Decimal256.getMaxPrecision(), scale);
                }
                case Float32: {
                    return (T)Float.valueOf(this.readFloatLE());
                }
                case Float64: {
                    return (T)Double.valueOf(this.readDoubleLE());
                }
                case Bool: {
                    return (T)Boolean.valueOf(BinaryStreamReader.readByteOrEOF(this.input) == 1);
                }
                case Enum8: {
                    byte enum8Val = (byte)this.readUnsignedByte();
                    String name = actualColumn.getEnumConstants().nameNullable((int)enum8Val);
                    return (T)new EnumValue(name == null ? "<unknown>" : name, enum8Val);
                }
                case Enum16: {
                    short enum16Val = (short)this.readUnsignedShortLE();
                    String name = actualColumn.getEnumConstants().nameNullable((int)enum16Val);
                    return (T)new EnumValue(name == null ? "<unknown>" : name, enum16Val);
                }
                case Date: {
                    return BinaryStreamReader.convertDateTime(this.readDate(timezone), typeHint);
                }
                case Date32: {
                    return BinaryStreamReader.convertDateTime(this.readDate32(timezone), typeHint);
                }
                case DateTime: {
                    return BinaryStreamReader.convertDateTime(this.readDateTime32(timezone), typeHint);
                }
                case DateTime32: {
                    return BinaryStreamReader.convertDateTime(this.readDateTime32(timezone), typeHint);
                }
                case DateTime64: {
                    return BinaryStreamReader.convertDateTime(this.readDateTime64(scale, timezone), typeHint);
                }
                case Time: {
                    return (T)Integer.valueOf(this.readIntLE());
                }
                case Time64: {
                    return (T)Long.valueOf(this.readLongLE());
                }
                case IntervalYear: 
                case IntervalQuarter: 
                case IntervalMonth: 
                case IntervalWeek: 
                case IntervalDay: 
                case IntervalHour: 
                case IntervalMinute: 
                case IntervalSecond: 
                case IntervalMicrosecond: 
                case IntervalMillisecond: 
                case IntervalNanosecond: {
                    return (T)this.readIntervalValue(dataType, this.input);
                }
                case IPv4: {
                    return (T)Inet4Address.getByAddress(this.readNBytesLE(this.input, 4));
                }
                case IPv6: {
                    return (T)Inet6Address.getByAddress(BinaryStreamReader.readNBytes(this.input, 16));
                }
                case UUID: {
                    return (T)new UUID(this.readLongLE(), this.readLongLE());
                }
                case Point: {
                    return (T)this.readGeoPoint();
                }
                case Polygon: {
                    return (T)this.readGeoPolygon();
                }
                case MultiPolygon: {
                    return (T)this.readGeoMultiPolygon();
                }
                case MultiLineString: {
                    return (T)this.readGeoPolygon();
                }
                case Ring: {
                    return (T)this.readGeoRing();
                }
                case LineString: {
                    return (T)this.readGeoRing();
                }
                case JSON: {
                    if (this.jsonAsString) {
                        return (T)BinaryStreamReader.readString(this.input);
                    }
                    return (T)this.readJsonData(this.input);
                }
                case Array: {
                    if (typeHint == null) {
                        typeHint = this.arrayDefaultTypeHint;
                    }
                    return BinaryStreamReader.convertArray(this.readArray(actualColumn), typeHint);
                }
                case Map: {
                    return (T)this.readMap(actualColumn);
                }
                case Tuple: {
                    return (T)this.readTuple(actualColumn);
                }
                case Nothing: {
                    return null;
                }
                case SimpleAggregateFunction: {
                    return this.readValue((ClickHouseColumn)column.getNestedColumns().get(0));
                }
                case AggregateFunction: {
                    return (T)this.readBitmap(actualColumn);
                }
                case Variant: {
                    return (T)this.readVariant(actualColumn);
                }
                case Dynamic: {
                    return this.readValue(actualColumn, typeHint);
                }
                case Nested: {
                    return BinaryStreamReader.convertArray(this.readNested(actualColumn), typeHint);
                }
            }
            throw new IllegalArgumentException("Unsupported data type: " + actualColumn.getDataType());
        }
        catch (EOFException e) {
            throw e;
        }
        catch (Exception e) {
            this.log.debug("Failed to read value for column {}, {}", (Object)column.getColumnName(), (Object)e.getLocalizedMessage());
            throw new ClientException("Failed to read value for column " + column.getColumnName(), e);
        }
    }

    private TemporalAmount readIntervalValue(ClickHouseDataType dataType, InputStream input) throws IOException {
        BigInteger v = this.readBigIntegerLE(8, true);
        switch (dataType) {
            case IntervalYear: {
                return Period.ofYears(v.intValue());
            }
            case IntervalQuarter: {
                return Period.ofMonths(3 * v.intValue());
            }
            case IntervalMonth: {
                return Period.ofMonths(v.intValue());
            }
            case IntervalWeek: {
                return Period.ofWeeks(v.intValue());
            }
            case IntervalDay: {
                return Period.ofDays(v.intValue());
            }
            case IntervalHour: {
                return Duration.ofHours(v.longValue());
            }
            case IntervalMinute: {
                return Duration.ofMinutes(v.longValue());
            }
            case IntervalSecond: {
                return Duration.ofSeconds(v.longValue());
            }
            case IntervalMicrosecond: {
                return Duration.ofNanos(v.longValue() * 1000L);
            }
            case IntervalMillisecond: {
                return Duration.ofMillis(v.longValue());
            }
            case IntervalNanosecond: {
                return Duration.ofNanos(v.longValue());
            }
        }
        throw new ClientException("Unsupported interval type: " + dataType);
    }

    private static <T> T convertDateTime(ZonedDateTime value, Class<?> typeHint) {
        if (typeHint == null) {
            return (T)value;
        }
        if (LocalDateTime.class.isAssignableFrom(typeHint)) {
            return (T)value.toLocalDateTime();
        }
        if (LocalDate.class.isAssignableFrom(typeHint)) {
            return (T)value.toLocalDate();
        }
        return (T)value;
    }

    private static <T> T convertArray(ArrayValue value, Class<?> typeHint) {
        if (typeHint == null) {
            return (T)value;
        }
        if (typeHint == Object.class) {
            return (T)value.asList();
        }
        if (List.class.isAssignableFrom(typeHint)) {
            return (T)value.asList();
        }
        if (typeHint.isArray()) {
            return (T)value.array;
        }
        return (T)value;
    }

    public short readShortLE() throws IOException {
        return BinaryStreamReader.readShortLE(this.input, this.int16Buff);
    }

    public static short readShortLE(InputStream input, byte[] buff) throws IOException {
        BinaryStreamReader.readNBytes(input, buff, 0, 2);
        return (short)(buff[0] & 0xFF | (buff[1] & 0xFF) << 8);
    }

    public int readIntLE() throws IOException {
        return BinaryStreamReader.readIntLE(this.input, this.int32Buff);
    }

    public static int readIntLE(InputStream input, byte[] buff) throws IOException {
        BinaryStreamReader.readNBytes(input, buff, 0, 4);
        return buff[0] & 0xFF | (buff[1] & 0xFF) << 8 | (buff[2] & 0xFF) << 16 | (buff[3] & 0xFF) << 24;
    }

    public long readLongLE() throws IOException {
        return BinaryStreamReader.readLongLE(this.input, this.int64Buff);
    }

    public static long readLongLE(InputStream input, byte[] buff) throws IOException {
        BinaryStreamReader.readNBytes(input, buff, 0, 8);
        return 0xFFL & (long)buff[0] | (0xFFL & (long)buff[1]) << 8 | (0xFFL & (long)buff[2]) << 16 | (0xFFL & (long)buff[3]) << 24 | (0xFFL & (long)buff[4]) << 32 | (0xFFL & (long)buff[5]) << 40 | (0xFFL & (long)buff[6]) << 48 | (0xFFL & (long)buff[7]) << 56;
    }

    public byte readByte() throws IOException {
        return (byte)BinaryStreamReader.readByteOrEOF(this.input);
    }

    public short readUnsignedByte() throws IOException {
        return (short)(BinaryStreamReader.readByteOrEOF(this.input) & 0xFF);
    }

    public int readUnsignedShortLE() throws IOException {
        return BinaryStreamReader.readUnsignedShortLE(this.input, this.int16Buff);
    }

    public static int readUnsignedShortLE(InputStream input, byte[] buff) throws IOException {
        return BinaryStreamReader.readShortLE(input, buff) & 0xFFFF;
    }

    public long readUnsignedIntLE() throws IOException {
        return (long)this.readIntLE() & 0xFFFFFFFFL;
    }

    public static long readUnsignedIntLE(InputStream input, byte[] buff) throws IOException {
        return (long)BinaryStreamReader.readIntLE(input, buff) & 0xFFFFFFFFL;
    }

    public BigInteger readBigIntegerLE(int len, boolean unsigned) throws IOException {
        return BinaryStreamReader.readBigIntegerLE(this.input, this.bufferAllocator.allocate(len), len, unsigned);
    }

    public static BigInteger readBigIntegerLE(InputStream input, byte[] buff, int len, boolean unsigned) throws IOException {
        byte[] bytes = BinaryStreamReader.readNBytesLE(input, buff, 0, len);
        return unsigned ? new BigInteger(1, bytes) : new BigInteger(bytes);
    }

    public float readFloatLE() throws IOException {
        return Float.intBitsToFloat(this.readIntLE());
    }

    public double readDoubleLE() throws IOException {
        return Double.longBitsToDouble(this.readLongLE());
    }

    public BigDecimal readDecimal(int precision, int scale) throws IOException {
        if (precision <= ClickHouseDataType.Decimal32.getMaxScale()) {
            return BigDecimal.valueOf(this.readIntLE(), scale);
        }
        BigDecimal v = precision <= ClickHouseDataType.Decimal64.getMaxScale() ? BigDecimal.valueOf(this.readLongLE(), scale) : (precision <= ClickHouseDataType.Decimal128.getMaxScale() ? new BigDecimal(this.readBigIntegerLE(16, false), scale) : new BigDecimal(this.readBigIntegerLE(32, false), scale));
        return v;
    }

    public static byte[] readNBytes(InputStream inputStream, int len) throws IOException {
        byte[] bytes = new byte[len];
        return BinaryStreamReader.readNBytes(inputStream, bytes, 0, len);
    }

    public static byte[] readNBytes(InputStream inputStream, byte[] buffer, int offset, int len) throws IOException {
        int r;
        for (int total = 0; total < len; total += r) {
            r = inputStream.read(buffer, offset + total, len - total);
            if (r != -1) continue;
            throw new EOFException("End of stream reached before reading all data");
        }
        return buffer;
    }

    private byte[] readNBytesLE(InputStream input, int len) throws IOException {
        return BinaryStreamReader.readNBytesLE(input, this.bufferAllocator.allocate(len), 0, len);
    }

    public static byte[] readNBytesLE(InputStream input, byte[] buffer, int offset, int len) throws IOException {
        byte[] bytes = BinaryStreamReader.readNBytes(input, buffer, 0, len);
        int s = 0;
        for (int i = len - 1; s < i; ++s, --i) {
            byte b = bytes[s];
            bytes[s] = bytes[i];
            bytes[i] = b;
        }
        return bytes;
    }

    public ArrayValue readArray(ClickHouseColumn column) throws IOException {
        ArrayValue array;
        int len = BinaryStreamReader.readVarInt(this.input);
        if (len == 0) {
            return new ArrayValue(Object.class, 0);
        }
        ClickHouseColumn itemTypeColumn = (ClickHouseColumn)column.getNestedColumns().get(0);
        if (column.getArrayNestedLevel() == 1) {
            array = this.readArrayItem(itemTypeColumn, len);
        } else {
            array = new ArrayValue(ArrayValue.class, len);
            for (int i = 0; i < len; ++i) {
                array.set(i, this.readArray(itemTypeColumn));
            }
        }
        return array;
    }

    public ArrayValue readArrayItem(ClickHouseColumn itemTypeColumn, int len) throws IOException {
        ArrayValue array;
        if (itemTypeColumn.isNullable()) {
            array = new ArrayValue(Object.class, len);
            for (int i = 0; i < len; ++i) {
                array.set(i, this.readValue(itemTypeColumn));
            }
        } else {
            Object firstValue = this.readValue(itemTypeColumn);
            Class<Object> itemClass = firstValue.getClass();
            if (firstValue instanceof Byte) {
                itemClass = Byte.TYPE;
            } else if (firstValue instanceof Character) {
                itemClass = Character.TYPE;
            } else if (firstValue instanceof Short) {
                itemClass = Short.TYPE;
            } else if (firstValue instanceof Integer) {
                itemClass = Integer.TYPE;
            } else if (firstValue instanceof Long) {
                itemClass = Long.TYPE;
            } else if (firstValue instanceof Boolean) {
                itemClass = Boolean.TYPE;
            }
            array = new ArrayValue(itemClass, len);
            array.set(0, firstValue);
            for (int i = 1; i < len; ++i) {
                array.set(i, this.readValue(itemTypeColumn));
            }
        }
        return array;
    }

    public void skipValue(ClickHouseColumn column) throws IOException {
        this.readValue(column, null);
    }

    public Map<?, ?> readMap(ClickHouseColumn column) throws IOException {
        int len = BinaryStreamReader.readVarInt(this.input);
        if (len == 0) {
            return Collections.emptyMap();
        }
        ClickHouseColumn keyType = column.getKeyInfo();
        ClickHouseColumn valueType = column.getValueInfo();
        LinkedHashMap map = new LinkedHashMap(len);
        for (int i = 0; i < len; ++i) {
            Object key = this.readValue(keyType);
            Object value = this.readValue(valueType);
            map.put(key, value);
        }
        return map;
    }

    public Object[] readTuple(ClickHouseColumn column) throws IOException {
        int len = column.getNestedColumns().size();
        Object[] tuple = new Object[len];
        for (int i = 0; i < len; ++i) {
            tuple[i] = this.readValue((ClickHouseColumn)column.getNestedColumns().get(i));
        }
        return tuple;
    }

    public ArrayValue readNested(ClickHouseColumn column) throws IOException {
        int len = BinaryStreamReader.readVarInt(this.input);
        if (len == 0) {
            return new ArrayValue(Object[].class, 0);
        }
        ArrayValue array = new ArrayValue(Object[].class, len);
        for (int i = 0; i < len; ++i) {
            int tupleLen = column.getNestedColumns().size();
            Object[] tuple = new Object[tupleLen];
            for (int j = 0; j < tupleLen; ++j) {
                tuple[j] = this.readValue((ClickHouseColumn)column.getNestedColumns().get(j));
            }
            array.set(i, tuple);
        }
        return array;
    }

    public Object readVariant(ClickHouseColumn column) throws IOException {
        byte ordNum = this.readByte();
        return this.readValue((ClickHouseColumn)column.getNestedColumns().get(ordNum));
    }

    public double[] readGeoPoint() throws IOException {
        return new double[]{this.readDoubleLE(), this.readDoubleLE()};
    }

    public double[][] readGeoRing() throws IOException {
        int count = BinaryStreamReader.readVarInt(this.input);
        double[][] value = new double[count][2];
        for (int i = 0; i < count; ++i) {
            value[i] = this.readGeoPoint();
        }
        return value;
    }

    public double[][][] readGeoPolygon() throws IOException {
        int count = BinaryStreamReader.readVarInt(this.input);
        double[][][] value = new double[count][][];
        for (int i = 0; i < count; ++i) {
            value[i] = this.readGeoRing();
        }
        return value;
    }

    private double[][][][] readGeoMultiPolygon() throws IOException {
        int count = BinaryStreamReader.readVarInt(this.input);
        double[][][][] value = new double[count][][][];
        for (int i = 0; i < count; ++i) {
            value[i] = this.readGeoPolygon();
        }
        return value;
    }

    public static int readVarInt(InputStream input) throws IOException {
        int value = 0;
        for (int i = 0; i < 10; ++i) {
            byte b = (byte)BinaryStreamReader.readByteOrEOF(input);
            value |= (b & 0x7F) << 7 * i;
            if ((b & 0x80) == 0) break;
        }
        return value;
    }

    private ZonedDateTime readDate(TimeZone tz) throws IOException {
        return BinaryStreamReader.readDate(this.input, this.bufferAllocator.allocate(2), tz);
    }

    public static ZonedDateTime readDate(InputStream input, byte[] buff, TimeZone tz) throws IOException {
        LocalDate d = LocalDate.ofEpochDay(BinaryStreamReader.readUnsignedShortLE(input, buff));
        return d.atStartOfDay(tz.toZoneId()).withZoneSameInstant(tz.toZoneId());
    }

    public ZonedDateTime readDate32(TimeZone tz) throws IOException {
        return BinaryStreamReader.readDate32(this.input, this.bufferAllocator.allocate(4), tz);
    }

    public static ZonedDateTime readDate32(InputStream input, byte[] buff, TimeZone tz) throws IOException {
        LocalDate d = LocalDate.ofEpochDay(BinaryStreamReader.readIntLE(input, buff));
        return d.atStartOfDay(tz.toZoneId()).withZoneSameInstant(tz.toZoneId());
    }

    private ZonedDateTime readDateTime32(TimeZone tz) throws IOException {
        return BinaryStreamReader.readDateTime32(this.input, this.bufferAllocator.allocate(4), tz);
    }

    public static ZonedDateTime readDateTime32(InputStream input, byte[] buff, TimeZone tz) throws IOException {
        long time = BinaryStreamReader.readUnsignedIntLE(input, buff);
        return Instant.ofEpochSecond(Math.max(time, 0L)).atZone(tz.toZoneId());
    }

    public ZonedDateTime readDateTime64(int scale, TimeZone tz) throws IOException {
        return BinaryStreamReader.readDateTime64(this.input, this.bufferAllocator.allocate(8), scale, tz);
    }

    public static ZonedDateTime readDateTime64(InputStream input, byte[] buff, int scale, TimeZone tz) throws IOException {
        long value = BinaryStreamReader.readLongLE(input, buff);
        int nanoSeconds = 0;
        if (scale > 0) {
            int factor = BASES[scale];
            nanoSeconds = (int)(value % (long)factor);
            value /= (long)factor;
            if (nanoSeconds < 0) {
                nanoSeconds += factor;
                --value;
            }
            if ((long)nanoSeconds > 0L) {
                nanoSeconds *= BASES[9 - scale];
            }
        }
        return Instant.ofEpochSecond(value, nanoSeconds).atZone(tz.toZoneId());
    }

    public String readString() throws IOException {
        int len = BinaryStreamReader.readVarInt(this.input);
        if (len == 0) {
            return "";
        }
        byte[] dest = len > this.STRING_BUFF.length ? new byte[len] : this.STRING_BUFF;
        BinaryStreamReader.readNBytes(this.input, dest, 0, len);
        return new String(dest, 0, len, StandardCharsets.UTF_8);
    }

    public static String readString(InputStream input) throws IOException {
        int len = BinaryStreamReader.readVarInt(input);
        if (len == 0) {
            return "";
        }
        return new String(BinaryStreamReader.readNBytes(input, len), StandardCharsets.UTF_8);
    }

    public static int readByteOrEOF(InputStream input) throws IOException {
        int b = input.read();
        if (b < 0) {
            throw new EOFException("End of stream reached before reading all data");
        }
        return b;
    }

    public static boolean isReadToPrimitive(ClickHouseDataType dataType) {
        switch (dataType) {
            case Int8: 
            case UInt8: 
            case Int16: 
            case UInt16: 
            case Int32: 
            case UInt32: 
            case Int64: 
            case Float32: 
            case Float64: 
            case Bool: 
            case Enum8: 
            case Enum16: {
                return true;
            }
        }
        return false;
    }

    private ClickHouseBitmap readBitmap(ClickHouseColumn column) throws IOException {
        return ClickHouseBitmap.deserialize((InputStream)this.input, (ClickHouseDataType)((ClickHouseColumn)column.getNestedColumns().get(0)).getDataType());
    }

    private ClickHouseColumn readDynamicData() throws IOException {
        byte tag = this.readByte();
        if (tag == 34) {
            byte intervalKind = this.readByte();
            ClickHouseDataType type = (ClickHouseDataType)ClickHouseDataType.intervalKind2Type.get(intervalKind);
            if (type == null) {
                throw new ClientException("Unsupported interval kind: " + intervalKind);
            }
            return ClickHouseColumn.of((String)"v", (ClickHouseDataType)type, (boolean)false, (int)0, (int)0);
        }
        if (tag == ClickHouseDataType.DateTime32.getBinTag()) {
            byte scale = this.readByte();
            return ClickHouseColumn.of((String)"v", (String)("DateTime32(" + scale + ")"));
        }
        if (tag == ClickHouseDataType.DateTime64.getBinTag() - 1) {
            byte scale = this.readByte();
            return ClickHouseColumn.of((String)"v", (String)("DateTime64(" + scale + ")"));
        }
        if (tag == ClickHouseDataType.DateTime64.getBinTag()) {
            byte scale = this.readByte();
            String timezone = BinaryStreamReader.readString(this.input);
            return ClickHouseColumn.of((String)"v", (String)("DateTime64(" + scale + (timezone.isEmpty() ? "" : ", " + timezone) + ")"));
        }
        if (tag == 44) {
            String typeName = BinaryStreamReader.readString(this.input);
            return ClickHouseColumn.of((String)"v", (String)typeName);
        }
        if (DECIMAL_TAGS.contains(tag)) {
            byte precision = this.readByte();
            byte scale = this.readByte();
            return ClickHouseColumn.of((String)"v", (ClickHouseDataType)((ClickHouseDataType)ClickHouseDataType.binTag2Type.get(tag)), (boolean)false, (int)precision, (int)scale);
        }
        if (tag == ClickHouseDataType.Array.getBinTag()) {
            ClickHouseColumn elementColumn = this.readDynamicData();
            return ClickHouseColumn.of((String)"v", (String)("Array(" + elementColumn.getOriginalTypeName() + ")"));
        }
        if (tag == ClickHouseDataType.Map.getBinTag()) {
            ClickHouseColumn keyInfo = this.readDynamicData();
            ClickHouseColumn valueInfo = this.readDynamicData();
            return ClickHouseColumn.of((String)"v", (String)("Map(" + keyInfo.getOriginalTypeName() + "," + valueInfo.getOriginalTypeName() + ")"));
        }
        if (tag == ClickHouseDataType.Enum8.getBinTag() || tag == ClickHouseDataType.Enum16.getBinTag()) {
            int constants = BinaryStreamReader.readVarInt(this.input);
            int[] values = new int[constants];
            String[] names = new String[constants];
            ClickHouseDataType enumType = constants > 127 ? ClickHouseDataType.Enum16 : ClickHouseDataType.Enum8;
            for (int i = 0; i < constants; ++i) {
                names[i] = BinaryStreamReader.readString(this.input);
                values[i] = enumType == ClickHouseDataType.Enum8 ? (int)this.readUnsignedByte() : this.readUnsignedShortLE();
            }
            return new ClickHouseColumn(enumType, "v", enumType.name(), false, false, Collections.emptyList(), Collections.emptyList(), new ClickHouseEnum(names, values));
        }
        if (tag == 35) {
            ClickHouseColumn column = this.readDynamicData();
            return ClickHouseColumn.of((String)"v", (String)("Nullable(" + column.getOriginalTypeName() + ")"));
        }
        ClickHouseDataType type = (ClickHouseDataType)ClickHouseDataType.binTag2Type.get(tag);
        if (type == null) {
            throw new ClientException("Unsupported data type with tag " + tag);
        }
        return ClickHouseColumn.of((String)"v", (ClickHouseDataType)type, (boolean)false, (int)0, (int)0);
    }

    private Map<String, Object> readJsonData(InputStream input) throws IOException {
        int numOfPaths = BinaryStreamReader.readVarInt(input);
        if (numOfPaths == 0) {
            return Collections.emptyMap();
        }
        HashMap<String, Object> obj = new HashMap<String, Object>();
        for (int i = 0; i < numOfPaths; ++i) {
            String path = BinaryStreamReader.readString(input);
            Object value = this.readValue(JSON_PLACEHOLDER_COL);
            obj.put(path, value);
        }
        return obj;
    }

    public static interface ByteBufferAllocator {
        public byte[] allocate(int var1);
    }

    public static class EnumValue
    extends Number {
        public final String name;
        public final int value;

        public EnumValue(String name, int value) {
            this.name = name;
            this.value = value;
        }

        @Override
        public int intValue() {
            return this.value;
        }

        @Override
        public long longValue() {
            return this.value;
        }

        @Override
        public float floatValue() {
            return this.value;
        }

        @Override
        public double doubleValue() {
            return this.value;
        }

        public String toString() {
            return this.name;
        }
    }

    public static class ArrayValue {
        final int length;
        final Class<?> itemType;
        final Object array;
        int nextPos = 0;
        private List<?> list = null;

        ArrayValue(Class<?> itemType, int length) {
            this.itemType = itemType;
            this.length = length;
            try {
                this.array = itemType.isArray() ? Array.newInstance(Object[].class, length) : Array.newInstance(itemType, length);
            }
            catch (Exception e) {
                throw new IllegalArgumentException("Failed to create array of type: " + itemType, e);
            }
        }

        public int length() {
            return this.length;
        }

        public Object get(int index) {
            return Array.get(this.array, index);
        }

        public void set(int index, Object value) {
            try {
                Array.set(this.array, index, value);
            }
            catch (IllegalArgumentException e) {
                throw new IllegalArgumentException("Failed to set value at index: " + index + " value " + value + " of class " + value.getClass().getName(), e);
            }
        }

        public boolean append(Object value) {
            this.set(this.nextPos++, value);
            return this.nextPos == this.length;
        }

        public synchronized <T> List<T> asList() {
            if (this.list == null) {
                ArrayList list = new ArrayList(this.length);
                for (int i = 0; i < this.length; ++i) {
                    Object item = this.get(i);
                    if (item instanceof ArrayValue) {
                        list.add(((ArrayValue)item).asList());
                        continue;
                    }
                    list.add(item);
                }
                this.list = list;
            }
            return this.list;
        }
    }

    public static class CachingByteBufferAllocator
    implements ByteBufferAllocator {
        private static final int MAX_PREALLOCATED_SIZE = 32;
        private final byte[][] preallocated = new byte[33][];

        public CachingByteBufferAllocator() {
            for (int i = 0; i < this.preallocated.length; ++i) {
                this.preallocated[i] = new byte[i];
            }
        }

        @Override
        public byte[] allocate(int size) {
            if (size < this.preallocated.length) {
                return this.preallocated[size];
            }
            return new byte[size];
        }
    }

    public static class DefaultByteBufferAllocator
    implements ByteBufferAllocator {
        @Override
        public byte[] allocate(int size) {
            return new byte[size];
        }
    }
}

