package com.clevertap.android.sdk.bitmap

import android.graphics.BitmapFactory
import com.clevertap.android.sdk.Logger
import com.clevertap.android.sdk.Utils
import com.clevertap.android.sdk.network.DownloadedBitmap
import com.clevertap.android.sdk.network.DownloadedBitmap.Status.DOWNLOAD_FAILED
import com.clevertap.android.sdk.network.DownloadedBitmapFactory
import java.io.ByteArrayOutputStream
import java.io.InputStream
import java.net.HttpURLConnection

open class BitmapInputStreamDecoder(
    val saveBytes: Boolean = false,
    val saveBitmap : Boolean = true,
    val logger: Logger? = null
) : IBitmapInputStreamReader {

    override fun readInputStream(
        inputStream: InputStream,
        connection: HttpURLConnection,
        downloadStartTimeInMilliseconds: Long
    ): DownloadedBitmap {
        logger?.verbose("reading bitmap input stream in BitmapInputStreamDecoder....")

        val bufferForHttpInputStream = ByteArray(16384)
        val finalDataFromHttpInputStream = ByteArrayOutputStream()

        var totalBytesRead = 0
        var bytesRead: Int

        // Read data from input stream
        while (inputStream.read(bufferForHttpInputStream).also { bytesRead = it } != -1) {
            totalBytesRead += bytesRead
            finalDataFromHttpInputStream.write(bufferForHttpInputStream, 0, bytesRead)
            logger?.verbose("Downloaded $totalBytesRead bytes")
        }
        logger?.verbose("Total download size for bitmap = $totalBytesRead")

        val dataReadFromStreamInByteArray = finalDataFromHttpInputStream.toByteArray()

        val fileLength = connection.contentLength
        if (fileLength != -1 && fileLength != totalBytesRead) {
            val reason = "Incomplete Download"
            logger?.debug("File not loaded completely not going forward. URL was: ${connection.url}, Reason: $reason")
            return DownloadedBitmapFactory.nullBitmapWithStatus(DOWNLOAD_FAILED, reason)
        }

        val downloadedBitmap: DownloadedBitmap
        if (saveBitmap) {
            // Decode the bitmap from decompressed data
            val bitmap = BitmapFactory.decodeByteArray(
                dataReadFromStreamInByteArray,
                0,
                dataReadFromStreamInByteArray.size
            )

            downloadedBitmap = if (bitmap != null) {
                DownloadedBitmapFactory.successBitmap(
                    bitmap = bitmap,
                    downloadTime = Utils.getNowInMillis() - downloadStartTimeInMilliseconds,
                    data = if (saveBytes) {
                        dataReadFromStreamInByteArray
                    } else {
                        null
                    }
                )
            } else {
                val reason = "DecodeByteArray error"
                logger?.debug(reason)
                DownloadedBitmapFactory.nullBitmapWithStatus(DOWNLOAD_FAILED, reason)
            }
        } else {
            downloadedBitmap = DownloadedBitmapFactory.successBytes(
                downloadTime = Utils.getNowInMillis() - downloadStartTimeInMilliseconds,
                data = dataReadFromStreamInByteArray
            )
        }
        return downloadedBitmap
    }
}