/*
 * Decompiled with CFR 0.152.
 */
package com.cedarsoftware.util;

import com.cedarsoftware.util.AdjustableGZIPOutputStream;
import com.cedarsoftware.util.ByteUtilities;
import com.cedarsoftware.util.Convention;
import com.cedarsoftware.util.FastByteArrayOutputStream;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.File;
import java.io.Flushable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URLConnection;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.Arrays;
import java.util.Objects;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.DeflaterOutputStream;
import java.util.zip.GZIPInputStream;
import java.util.zip.Inflater;
import java.util.zip.InflaterInputStream;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;

public final class IOUtilities {
    private static final int TRANSFER_BUFFER = 32768;
    private static final int DEFAULT_CONNECT_TIMEOUT = 5000;
    private static final int DEFAULT_READ_TIMEOUT = 30000;
    private static final boolean DEBUG = Boolean.parseBoolean(System.getProperty("io.debug", "false"));
    private static final Logger LOG = Logger.getLogger(IOUtilities.class.getName());

    private static void debug(String msg, Exception e) {
        if (DEBUG) {
            if (e == null) {
                LOG.fine(msg);
            } else {
                LOG.log(Level.FINE, msg, e);
            }
        }
    }

    private IOUtilities() {
    }

    public static InputStream getInputStream(URLConnection c) throws IOException {
        Convention.throwIfNull(c, "URLConnection cannot be null");
        IOUtilities.optimizeConnection(c);
        String enc = c.getContentEncoding();
        InputStream is = c.getInputStream();
        if (enc != null) {
            if ("gzip".equalsIgnoreCase(enc) || "x-gzip".equalsIgnoreCase(enc)) {
                is = new GZIPInputStream(is, 32768);
            } else if ("deflate".equalsIgnoreCase(enc)) {
                is = new InflaterInputStream(is, new Inflater(), 32768);
            }
        }
        return new BufferedInputStream(is, 32768);
    }

    private static void optimizeConnection(URLConnection c) {
        if (c instanceof HttpURLConnection) {
            HttpURLConnection http = (HttpURLConnection)c;
            http.setInstanceFollowRedirects(true);
            http.setUseCaches(false);
            int connectTimeout = 5000;
            int readTimeout = 30000;
            try {
                connectTimeout = Integer.parseInt(System.getProperty("io.connect.timeout", String.valueOf(5000)));
                readTimeout = Integer.parseInt(System.getProperty("io.read.timeout", String.valueOf(30000)));
            }
            catch (NumberFormatException e) {
                IOUtilities.debug("Invalid timeout settings", e);
            }
            http.setConnectTimeout(connectTimeout);
            http.setReadTimeout(readTimeout);
            c.setRequestProperty("Accept-Encoding", "gzip, x-gzip, deflate");
        }
    }

    public static void transfer(File f, URLConnection c, TransferCallback cb) throws IOException {
        Convention.throwIfNull(f, "File cannot be null");
        Convention.throwIfNull(c, "URLConnection cannot be null");
        try (BufferedInputStream in = new BufferedInputStream(Files.newInputStream(f.toPath(), new OpenOption[0]));
             BufferedOutputStream out = new BufferedOutputStream(c.getOutputStream());){
            IOUtilities.transfer((InputStream)in, out, cb);
        }
    }

    public static void transfer(URLConnection c, File f, TransferCallback cb) throws IOException {
        Convention.throwIfNull(c, "URLConnection cannot be null");
        Convention.throwIfNull(f, "File cannot be null");
        try (InputStream in = IOUtilities.getInputStream(c);){
            IOUtilities.transfer(in, f, cb);
        }
    }

    public static void transfer(InputStream s, File f, TransferCallback cb) throws IOException {
        Convention.throwIfNull(s, "InputStream cannot be null");
        Convention.throwIfNull(f, "File cannot be null");
        try (BufferedOutputStream out = new BufferedOutputStream(Files.newOutputStream(f.toPath(), new OpenOption[0]));){
            IOUtilities.transfer(s, out, cb);
        }
    }

    public static void transfer(InputStream in, OutputStream out, TransferCallback cb) throws IOException {
        int count;
        Convention.throwIfNull(in, "InputStream cannot be null");
        Convention.throwIfNull(out, "OutputStream cannot be null");
        byte[] buffer = new byte[32768];
        while ((count = in.read(buffer)) != -1) {
            out.write(buffer, 0, count);
            if (cb == null) continue;
            cb.bytesTransferred(buffer, count);
            if (!cb.isCancelled()) continue;
            break;
        }
    }

    public static void transfer(InputStream in, byte[] bytes) throws IOException {
        Convention.throwIfNull(in, "InputStream cannot be null");
        Convention.throwIfNull(bytes, "byte array cannot be null");
        new DataInputStream(in).readFully(bytes);
    }

    public static void transfer(InputStream in, OutputStream out) throws IOException {
        int count;
        Convention.throwIfNull(in, "InputStream cannot be null");
        Convention.throwIfNull(out, "OutputStream cannot be null");
        byte[] buffer = new byte[32768];
        while ((count = in.read(buffer)) != -1) {
            out.write(buffer, 0, count);
        }
        out.flush();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void transfer(File file, OutputStream out) throws IOException {
        Convention.throwIfNull(file, "File cannot be null");
        Convention.throwIfNull(out, "OutputStream cannot be null");
        try (BufferedInputStream in = new BufferedInputStream(Files.newInputStream(file.toPath(), new OpenOption[0]), 32768);){
            IOUtilities.transfer((InputStream)in, out);
        }
        finally {
            IOUtilities.flush(out);
        }
    }

    public static void close(XMLStreamReader reader) {
        if (reader != null) {
            try {
                reader.close();
            }
            catch (XMLStreamException e) {
                IOUtilities.debug("Failed to close XMLStreamReader", e);
            }
        }
    }

    public static void close(XMLStreamWriter writer) {
        if (writer != null) {
            try {
                writer.close();
            }
            catch (XMLStreamException e) {
                IOUtilities.debug("Failed to close XMLStreamWriter", e);
            }
        }
    }

    public static void close(Closeable c) {
        if (c != null) {
            try {
                c.close();
            }
            catch (IOException e) {
                IOUtilities.debug("Failed to close Closeable", e);
            }
        }
    }

    public static void flush(Flushable f) {
        if (f != null) {
            try {
                f.flush();
            }
            catch (IOException e) {
                IOUtilities.debug("Failed to flush", e);
            }
        }
    }

    public static void flush(XMLStreamWriter writer) {
        if (writer != null) {
            try {
                writer.flush();
            }
            catch (XMLStreamException e) {
                IOUtilities.debug("Failed to flush XMLStreamWriter", e);
            }
        }
    }

    public static byte[] inputStreamToBytes(InputStream in) throws IOException {
        return IOUtilities.inputStreamToBytes(in, Integer.MAX_VALUE);
    }

    public static byte[] inputStreamToBytes(InputStream in, int maxSize) throws IOException {
        Convention.throwIfNull(in, "Inputstream cannot be null");
        if (maxSize <= 0) {
            throw new IllegalArgumentException("maxSize must be > 0");
        }
        try (FastByteArrayOutputStream out = new FastByteArrayOutputStream(16384);){
            int count;
            byte[] buffer = new byte[32768];
            int total = 0;
            while ((count = in.read(buffer)) != -1) {
                if ((total += count) > maxSize) {
                    throw new IOException("Stream exceeds maximum allowed size: " + maxSize);
                }
                out.write(buffer, 0, count);
            }
            byte[] byArray = out.toByteArray();
            return byArray;
        }
    }

    public static void transfer(URLConnection c, byte[] bytes) throws IOException {
        Convention.throwIfNull(c, "URLConnection cannot be null");
        Convention.throwIfNull(bytes, "byte array cannot be null");
        try (BufferedOutputStream out = new BufferedOutputStream(c.getOutputStream());){
            ((OutputStream)out).write(bytes);
        }
    }

    public static void compressBytes(ByteArrayOutputStream original, ByteArrayOutputStream compressed) throws IOException {
        Convention.throwIfNull(original, "Original ByteArrayOutputStream cannot be null");
        Convention.throwIfNull(compressed, "Compressed ByteArrayOutputStream cannot be null");
        try (AdjustableGZIPOutputStream gzipStream = new AdjustableGZIPOutputStream((OutputStream)compressed, 1);){
            original.writeTo(gzipStream);
            gzipStream.flush();
        }
    }

    public static void compressBytes(FastByteArrayOutputStream original, FastByteArrayOutputStream compressed) throws IOException {
        Convention.throwIfNull(original, "Original FastByteArrayOutputStream cannot be null");
        Convention.throwIfNull(compressed, "Compressed FastByteArrayOutputStream cannot be null");
        try (AdjustableGZIPOutputStream gzipStream = new AdjustableGZIPOutputStream((OutputStream)compressed, 1);){
            ((DeflaterOutputStream)gzipStream).write(original.toByteArray(), 0, original.size());
            gzipStream.flush();
        }
    }

    public static byte[] compressBytes(byte[] bytes) {
        return IOUtilities.compressBytes(bytes, 0, bytes.length);
    }

    public static byte[] compressBytes(byte[] bytes, int offset, int len) {
        byte[] byArray;
        Convention.throwIfNull(bytes, "Byte array cannot be null");
        FastByteArrayOutputStream byteStream = new FastByteArrayOutputStream();
        try {
            try (AdjustableGZIPOutputStream gzipStream = new AdjustableGZIPOutputStream((OutputStream)byteStream, 1);){
                ((DeflaterOutputStream)gzipStream).write(bytes, offset, len);
                gzipStream.flush();
            }
            byArray = Arrays.copyOf(byteStream.toByteArray(), byteStream.size());
        }
        catch (Throwable throwable) {
            try {
                try {
                    byteStream.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (Exception e) {
                throw new RuntimeException("Error compressing bytes.", e);
            }
        }
        byteStream.close();
        return byArray;
    }

    public static byte[] uncompressBytes(byte[] bytes) {
        return IOUtilities.uncompressBytes(bytes, 0, bytes.length);
    }

    /*
     * Enabled aggressive exception aggregation
     */
    public static byte[] uncompressBytes(byte[] bytes, int offset, int len) {
        Objects.requireNonNull(bytes, "Byte array cannot be null");
        if (ByteUtilities.isGzipped(bytes, offset)) {
            try (ByteArrayInputStream byteStream = new ByteArrayInputStream(bytes, offset, len);){
                byte[] byArray;
                try (GZIPInputStream gzipStream = new GZIPInputStream((InputStream)byteStream, 32768);){
                    byArray = IOUtilities.inputStreamToBytes(gzipStream);
                }
                return byArray;
            }
            catch (IOException e) {
                throw new RuntimeException("Error uncompressing bytes", e);
            }
        }
        return Arrays.copyOfRange(bytes, offset, offset + len);
    }

    @FunctionalInterface
    public static interface TransferCallback {
        public void bytesTransferred(byte[] var1, int var2);

        default public boolean isCancelled() {
            return false;
        }
    }
}

