/*
 * Decompiled with CFR 0.152.
 */
package com.aventstack.chaintest.http;

import com.aventstack.chaintest.conf.ConfigurationManager;
import com.aventstack.chaintest.domain.ChainTestEntity;
import com.aventstack.chaintest.domain.Test;
import com.aventstack.chaintest.http.ChainTestApiClient;
import com.aventstack.chaintest.http.HttpMethod;
import com.aventstack.chaintest.http.WrappedResponseAsync;
import java.io.IOException;
import java.net.ConnectException;
import java.net.http.HttpResponse;
import java.net.http.HttpTimeoutException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HttpRetryHandler {
    private static final Logger log = LoggerFactory.getLogger(HttpRetryHandler.class);
    public static final int MAX_RETRY_ATTEMPTS = 3;
    public static final long RETRY_INTERVAL = 2000L;
    private final ChainTestApiClient _client;
    private final int _maxRetryAttempts;
    private final long _retryIntervalMs;
    private final boolean _throwAfterMaxRetryAttempts;

    public HttpRetryHandler(ChainTestApiClient client, Map<String, String> config) {
        this._client = client;
        this._maxRetryAttempts = ConfigurationManager.parseConfig(config.get("chaintest.generator.chainlp.client.max-retries"), 3);
        log.debug("Creating HttpRetryHandler instance for {} retry attempts", (Object)this._maxRetryAttempts);
        this._retryIntervalMs = ConfigurationManager.parseConfig(config.get("chaintest.generator.chainlp.client.retry-interval-ms"), 2000L);
        this._throwAfterMaxRetryAttempts = Boolean.parseBoolean(config.get("chaintest.generator.chainlp.client.throw-after-retry-attempts-exceeded"));
    }

    public HttpRetryHandler(ChainTestApiClient client, int maxRetryAttempts, long retryIntervalMs, boolean throwAfterMaxRetryAttempts) {
        this._client = client;
        this._maxRetryAttempts = maxRetryAttempts;
        this._retryIntervalMs = retryIntervalMs;
        this._throwAfterMaxRetryAttempts = throwAfterMaxRetryAttempts;
    }

    public <T extends ChainTestEntity> HttpResponse<T> trySend(T entity, Class<T> clazz, HttpMethod httpMethod, int maxRetryAttempts) throws IOException, InterruptedException {
        for (int i = 0; i <= maxRetryAttempts; ++i) {
            if (i > 0) {
                log.trace("Retry: {}", (Object)i);
            }
            try {
                HttpResponse<T> response = this._client.send(entity, clazz, httpMethod);
                log.debug("Create API returned responseCode: {}", (Object)response.statusCode());
                if (200 == response.statusCode()) {
                    return response;
                }
                if (400 <= response.statusCode() && 499 >= response.statusCode() && 409 != response.statusCode()) {
                    log.error("Failed to save entity {} due to a client-side error, received response code : {}", (Object)clazz.getSimpleName(), (Object)response.statusCode());
                    return response;
                }
            }
            catch (IOException | InterruptedException e) {
                this.handleException(e, i, maxRetryAttempts);
            }
            Thread.sleep(this._retryIntervalMs);
        }
        return null;
    }

    private void handleException(Exception e, int attempt, int maxRetryAttempts) throws IOException, InterruptedException {
        if (e instanceof ConnectException) {
            log.debug("Failed to connect to the ChainTest service", (Throwable)e);
        } else if (e instanceof HttpTimeoutException) {
            log.debug("Timed out while waiting for ChainTest service response", (Throwable)e);
        } else {
            log.debug("An exception occurred while sending entity", (Throwable)e);
        }
        if (attempt == maxRetryAttempts && this._throwAfterMaxRetryAttempts) {
            if (e instanceof IOException) {
                throw (IOException)e;
            }
            if (e instanceof InterruptedException) {
                throw (InterruptedException)e;
            }
        }
    }

    public <T extends ChainTestEntity> HttpResponse<T> trySend(T entity, Class<T> clazz, HttpMethod httpMethod) throws IOException, InterruptedException {
        return this.trySend(entity, clazz, httpMethod, this._maxRetryAttempts);
    }

    public synchronized Map<String, WrappedResponseAsync<Test>> sendWithRetries(Map<String, WrappedResponseAsync<Test>> collection) {
        if (collection.isEmpty()) {
            return collection;
        }
        log.debug("Received collection of size {}. Handler will {}", (Object)collection.size(), this._maxRetryAttempts > 0 ? "retry for " + this._maxRetryAttempts + " attempts on errors" : "will not retry on errors");
        int size = collection.size();
        int retryAttempts = 0;
        ConcurrentHashMap<String, WrappedResponseAsync<Test>> failures = new ConcurrentHashMap<String, WrappedResponseAsync<Test>>(collection);
        while (!failures.isEmpty() && retryAttempts++ <= this._maxRetryAttempts) {
            this.trySendAsyncCollection(failures);
            if (failures.isEmpty() || retryAttempts > this._maxRetryAttempts) continue;
            try {
                this.wait(this._retryIntervalMs);
                log.debug("Retrying {} of {} times", (Object)retryAttempts, (Object)this._maxRetryAttempts);
            }
            catch (InterruptedException ignored) {
                log.debug("Interrupted while waiting for retry interval");
            }
        }
        this.handleFailures(failures, size);
        return failures;
    }

    private void handleFailures(Map<String, WrappedResponseAsync<Test>> failures, int size) {
        if (!failures.isEmpty()) {
            String message = String.format("Failed to transfer %d of %d tests. Make sure the ChainTest API is UP, ensure client-side logging is enabled or investigate API logs to find the underlying cause.", failures.size(), size);
            log.error(message);
            if (this._throwAfterMaxRetryAttempts) {
                throw new IllegalStateException(message);
            }
        }
    }

    private synchronized void trySendAsyncCollection(Map<String, WrappedResponseAsync<Test>> collection) {
        collection.forEach((k, v) -> v.setError(null));
        for (Map.Entry<String, WrappedResponseAsync<Test>> entry : collection.entrySet()) {
            boolean completed = entry.getValue().getResponseFuture().isDone();
            if (!completed) {
                try {
                    entry.getValue().getResponseFuture().join();
                }
                catch (Exception ignored) {
                    log.debug("Failed to join response future");
                }
            }
            HttpResponse response = entry.getValue().getResponse();
            long startedMillis = System.currentTimeMillis();
            while (null == response && System.currentTimeMillis() - startedMillis < 5000L) {
                try {
                    this.wait(100L);
                    response = entry.getValue().getResponse();
                }
                catch (InterruptedException ignored) {
                    log.debug("Interrupted while waiting for response");
                }
            }
            if (null == response) continue;
            if (200 == response.statusCode() || 409 == response.statusCode()) {
                collection.entrySet().removeIf(x -> ((String)x.getKey()).equals(entry.getKey()));
                continue;
            }
            if (400 > response.statusCode() || 499 < response.statusCode() || 409 == response.statusCode()) continue;
            log.error("Failed to persist entity {} due to a client-side error, received response code : {}", (Object)Test.class.getSimpleName(), (Object)response.statusCode());
            return;
        }
        for (Map.Entry<String, WrappedResponseAsync<Test>> entry : collection.entrySet()) {
            try {
                entry.getValue().setResponseFuture(this._client.sendAsync((Test)entry.getValue().getEntity(), Test.class));
            }
            catch (IOException ignore) {
                log.debug("Failed to send entity, will retry");
            }
        }
    }
}

