package com.atlassian.utils.process;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * A wrapper class containing all of the settings which should be applied to an {@link ExternalProcess} when it is
 * created by the {@link ExternalProcessFactory}.
 * <p/>
 * This class is <i>intentionally mutable</i>. Instances of it are bound to the {@link ExternalProcessBuilder}, and
 * they have the same thread-safety considerations it does--neither class is reentrant.
 *
 * @since 1.5
 */
public class ExternalProcessSettings {

    private final Map<String, String> environment;
    private final List<ProcessMonitor> monitors;

    private List<String> command;
    private long executionTimeout;
    private long idleTimeout;
    private ProcessHandler processHandler;
    private boolean suppressSpecialWindowsBehaviour;
    private boolean useWindowsEncodingWorkaround;
    private File workingDirectory;

    public ExternalProcessSettings() {
        environment = new HashMap<String, String>();
        monitors = new ArrayList<ProcessMonitor>();
    }

    public Map<String, String> getEnvironment() {
        return environment;
    }

    public List<ProcessMonitor> getMonitors() {
        return monitors;
    }

    public List<String> getCommand() {
        return command;
    }

    public long getExecutionTimeout() {
        return executionTimeout;
    }

    public long getIdleTimeout() {
        return idleTimeout;
    }

    public ProcessHandler getProcessHandler() {
        return processHandler;
    }

    public File getWorkingDirectory() {
        return workingDirectory;
    }

    public boolean hasEnvironment() {
        return !environment.isEmpty();
    }

    public boolean hasExecutionTimeout() {
        return executionTimeout > 0L;
    }

    public boolean hasIdleTimeout() {
        return idleTimeout > 0L;
    }

    public boolean isSuppressSpecialWindowsBehaviour() {
        return suppressSpecialWindowsBehaviour;
    }

    public boolean isUseWindowsEncodingWorkaround() {
        return useWindowsEncodingWorkaround;
    }

    public void setCommand(List<String> command) {
        this.command = command;
    }

    public void setExecutionTimeout(long executionTimeout) {
        this.executionTimeout = executionTimeout;
    }

    public void setIdleTimeout(long idleTimeout) {
        this.idleTimeout = idleTimeout;
    }

    public void setProcessHandler(ProcessHandler processHandler) {
        this.processHandler = processHandler;
    }

    public void setSuppressSpecialWindowsBehaviour(boolean suppressSpecialWindowsBehaviour) {
        this.suppressSpecialWindowsBehaviour = suppressSpecialWindowsBehaviour;
    }

    public void setUseWindowsEncodingWorkaround(boolean useWindowsEncodingWorkaround) {
        this.useWindowsEncodingWorkaround = useWindowsEncodingWorkaround;
    }

    public void setWorkingDirectory(File workingDirectory) {
        this.workingDirectory = workingDirectory;
    }
    
    public void validate() {
        if (command == null || command.isEmpty()) {
            throw new IllegalStateException("A command is required");
        }
        if (processHandler == null) {
            throw new IllegalStateException("A ProcessHandler is required");
        }
    }
}
