package com.atlassian.utils.process;

import org.apache.log4j.Logger;
import org.apache.log4j.Priority;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;


/**
 * Utility class to simplify the building of an ExternalProcess instance
 *
 */
public class ExternalProcessBuilder {
    private ProcessHandler handler;
    private InputHandler input;
    private OutputHandler output;
    private OutputHandler error;
    private final List<ProcessMonitor> monitors = new ArrayList<ProcessMonitor>();
    private List<String> command;
    private final Map<String, String> environment = new HashMap<String, String>();
    private File workingDir;
    private long timeout;
    private boolean disableWindowsEncodingWorkaround;

    public ExternalProcessBuilder handlers(InputHandler input, OutputHandler output, OutputHandler error) {
        this.input = input;
        this.output = output;
        this.error = error;
        return this;
    }

    public ExternalProcessBuilder handler(ProcessHandler handler) {
        this.handler = handler;
        return this;
    }

    public ExternalProcessBuilder handlers(OutputHandler output, OutputHandler error) {
        return this.handlers(null, output, error);
    }

    public ExternalProcessBuilder handlers(OutputHandler output) {
        return this.handlers(null, output, null);
    }

    public ExternalProcessBuilder handlers(InputHandler input, OutputHandler output) {
        return this.handlers(input, output, null);
    }

    public ExternalProcessBuilder command(List<String> command, File workingDir, long timeout) {
        this.command = command;
        this.workingDir = workingDir;
        this.timeout = timeout;
        return this;
    }

    public ExternalProcessBuilder command(List<String> command, File workingDir) {
        this.command = command;
        this.workingDir = workingDir;
        return this;
    }

    public ExternalProcessBuilder command(List<String> command) {
        this.command = command;
        return this;
    }

    public ExternalProcessBuilder timeout(long timeout) {
        this.timeout = timeout;
        return this;
    }

    public ExternalProcessBuilder log(Logger logger, Priority priority) {
        addMonitor(new LoggingProcessMonitor(logger, priority));
        return this;
    }

    public ExternalProcessBuilder log(Logger logger, Priority priority, StringObfuscator obfuscator) {
        addMonitor(new LoggingProcessMonitor(logger, priority, obfuscator));
        return this;
    }

    public ExternalProcessBuilder addMonitor(ProcessMonitor... monitors) {
        this.monitors.addAll(Arrays.asList(monitors));
        return this;
    }

    public ExternalProcessBuilder env(String variable, String value) {
        if (value != null) {
            this.environment.put(variable, value);
        } else {
            this.environment.put(variable, "");
        }
        return this;
    }
    
    public ExternalProcessBuilder env(Map<String, String> environment) {
        this.environment.putAll(environment);
        for (Map.Entry<String, String> entry : environment.entrySet()) {
            env(entry.getKey(), entry.getValue());
        }
        return this;
    }

    public ExternalProcessBuilder useWindowsEncodingWorkaround()
    {
        this.disableWindowsEncodingWorkaround = true;
        return this;
    }
    
    public ExternalProcess build() {
        ProcessHandler h = this.handler;
        if (this.handler == null) {
            // no processHandler defined, create a pluggableprocesshandler
            PluggableProcessHandler plugHandler = new PluggableProcessHandler();
            plugHandler.setInputHandler(this.input);
            plugHandler.setOutputHandler(this.output);
            if (this.error != null) {
                plugHandler.setErrorHandler(this.error);
            } else {
                plugHandler.setErrorHandler(new StringOutputHandler());
            }

            h = plugHandler;
        }

        ExternalProcess process = new ExternalProcess(new LinkedList<String>(command), h);
        if (timeout > 0L) {
            process.setTimeout(timeout);
        }
        process.setWorkingDir(workingDir);
        for (ProcessMonitor monitor: monitors) {
            if (monitor != null) {
                process.addMonitor(monitor);
            }
        }
        if (!environment.isEmpty()) {
            process.setEnvironment(new HashMap<String, String>(environment));
        }

        if (disableWindowsEncodingWorkaround)
        {
            process.setUseWindowsEncodingWorkaround(true);
        }

        return process;
    }
}
