package com.atlassian.user.search.query;

import com.atlassian.user.*;
import com.atlassian.user.configuration.Configuration;
import com.atlassian.user.configuration.ConfigurationException;
import com.atlassian.user.configuration.util.InitializationCheck;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.DefaultSearchResult;
import com.atlassian.user.search.SearchResult;
import com.atlassian.user.search.query.match.*;
import com.atlassian.user.search.page.DefaultPager;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.search.page.PagerUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.*;

/**
 * This {@link EntityQueryParser} is not efficient, as it handles all queries by loading
 * user and group {@link Entity} objects into memory.
 */
public class DefaultEntityQueryParser implements EntityQueryParser
{
    private static final Logger log = Logger.getLogger(DefaultEntityQueryParser.class);
    private final QueryValidator queryValidator = new QueryValidator();

    protected UserManager userManager;
    protected GroupManager groupManager;
    protected RepositoryIdentifier repository;
    protected Method entityNameMethod;
    protected Method emailMethod;
    protected Method fullnameMethod;
    private static final Class<User> userClass = User.class;

    public DefaultEntityQueryParser(RepositoryIdentifier repo, UserManager userManager, GroupManager groupManager)
    {
        try
        {
            entityNameMethod = userClass.getMethod("getName");
            emailMethod = userClass.getMethod("getEmail");
            fullnameMethod = userClass.getMethod("getFullName");
        }
        catch (NoSuchMethodException e)
        {
            log.error(e.getMessage());
        }

        this.userManager = userManager;
        this.groupManager = groupManager;
        this.repository = repo;
    }

    public void init(HashMap args) throws ConfigurationException
    {
        this.userManager = (UserManager) args.get(Configuration.USERMANAGER);
        this.groupManager = (GroupManager) args.get(Configuration.GROUPMANAGER);
        this.repository = (RepositoryIdentifier) args.get(Configuration.REPOSITORY);

        InitializationCheck.validateArgs(args, new String[]{Configuration.USERMANAGER,
            Configuration.GROUPMANAGER,
            Configuration.REPOSITORY}, this);

        try
        {
            entityNameMethod = userClass.getMethod("getName");
            emailMethod = userClass.getMethod("getEmail");
            fullnameMethod = userClass.getMethod("getFullName");
        }
        catch (NoSuchMethodException e)
        {
            log.error(e.getMessage());
        }
    }

    protected <T extends Entity> Pager<T> parseQuery(Method userMethod, TermQuery<T> q, Pager<T> data) throws IllegalAccessException, InvocationTargetException
    {
        String searchTerm = StringUtils.defaultString(q.getTerm()).toLowerCase(); // to allow case insensitive search
        if (searchTerm.indexOf(TermQuery.WILDCARD) >= 0)
            return data;

        Matcher matcher;
        if (q.isMatchingSubstring())
        {
            if (q.getMatchingRule().equals(TermQuery.SUBSTRING_STARTS_WITH))
            {
                matcher = new StartsWithIgnoreCaseMatcher();
            }
            else if (q.getMatchingRule().equals(TermQuery.SUBSTRING_ENDS_WITH))
            {
                matcher = new EndsWithIgnoreCaseMatcher();
            }
            else
            {
                matcher = new ContainsIgnoreCaseMatcher();
            }
        }
        else
        {
            matcher = new EqualsIgnoreCaseMatcher();
        }

        List<T> matches = new ArrayList<T>();

        for (T entity : data)
        {
            String userInfo = (String) userMethod.invoke(entity);

            if (matcher.matches(userInfo, searchTerm))
                matches.add(entity);
        }
        return new DefaultPager<T>(matches);
    }

    public <T extends Entity> Pager<T> find(Query<T> query) throws EntityException
    {
        if (query instanceof TermQuery)
        {
            try
            {
                if (query instanceof UserNameTermQuery)
                    return parseQuery(entityNameMethod, (TermQuery) query, userManager.getUsers());
                else if (query instanceof GroupNameTermQuery)
                    return parseQuery(entityNameMethod, (TermQuery) query, groupManager.getGroups());
                else if (query instanceof EmailTermQuery)
                    return parseQuery(emailMethod, (TermQuery) query, userManager.getUsers());
                else if (query instanceof FullNameTermQuery)
                    return parseQuery(fullnameMethod, (TermQuery) query, userManager.getUsers());
            }
            catch (IllegalAccessException e)
            {
                throw new EntityException(e);
            }
            catch (InvocationTargetException e)
            {
                throw new EntityException(e);
            }
        }
        else if (query instanceof BooleanQuery)
        {
            return evaluateBoolean((BooleanQuery<T>) query);
        }

        return null;
    }

    private <T extends Entity> Pager<T> evaluateBoolean(BooleanQuery<T> query)
    {
        List<Query<T>> queries = query.getQueries();
        Pager<T> allResults = null;

        boolean anding = query.isAND();

        for (Query<T> nextQuery : queries)
        {
            List<T> initialResult;

            try
            {
                if (allResults == null)
                {
                    allResults = find(nextQuery);
                }
                else if (nextQuery instanceof BooleanQuery)
                {
                    if (anding)
                    {
                        Pager<T> resultsToAnd = evaluateBoolean((BooleanQuery<T>) nextQuery);
                        List<T> allResultsList = PagerUtils.toList(allResults);

                        List<T> resultsToAndList = new ArrayList<T>(PagerUtils.toList(resultsToAnd));
                        resultsToAndList.retainAll(allResultsList);
                        allResults = new DefaultPager<T>(resultsToAndList);
                    }
                    else
                    {
                        Pager<T> resultsToOr = evaluateBoolean((BooleanQuery<T>) nextQuery);
                        List<T> resultsToOrList = new ArrayList<T>(PagerUtils.toList(resultsToOr));
                        List<T> intersection = findIntersection(PagerUtils.toList(allResults), resultsToOrList);
                        allResults = new DefaultPager<T>(intersection);
                    }
                }
                else if (anding)
                {

                    if (nextQuery instanceof UserNameTermQuery)
                    {
                        initialResult = PagerUtils.toList(parseQuery(entityNameMethod, (TermQuery<T>) nextQuery,
                            allResults));
                        initialResult.addAll(PagerUtils.toList(allResults));
                        allResults = new DefaultPager<T>(initialResult);
                    }
                    else if (nextQuery instanceof GroupNameTermQuery)
                    {
                        initialResult = PagerUtils.toList(parseQuery(entityNameMethod, (TermQuery<T>) nextQuery,
                            allResults));
                        initialResult.addAll(PagerUtils.toList(allResults));
                        allResults = new DefaultPager<T>(initialResult);
                    }
                    else if (nextQuery instanceof EmailTermQuery)
                    {
                        initialResult = PagerUtils.toList(parseQuery(emailMethod, (TermQuery<T>) nextQuery, allResults));
                        initialResult.addAll(PagerUtils.toList(allResults));
                        allResults = new DefaultPager<T>(initialResult);
                    }
                    else if (nextQuery instanceof FullNameTermQuery)
                    {
                        initialResult = PagerUtils.toList(parseQuery(fullnameMethod, (TermQuery<T>) nextQuery,
                            allResults));
                        initialResult.addAll(PagerUtils.toList(allResults));
                        allResults = new DefaultPager<T>(initialResult);
                    }
                }
                else // OR
                {
                    initialResult = PagerUtils.toList(find(nextQuery));
                    List<T> intersection = findIntersection(PagerUtils.toList(allResults), initialResult);
                    allResults = new DefaultPager<T>(intersection);
                }
            }
            catch (Exception e)
            {
                log.error(e.getClass().getName() + " - " + e.getMessage());
            }
        }

        return allResults;
    }

    private <T> List<T> findIntersection(List<? extends T> list1, List<? extends T> list2)
    {
        List<T> result = new ArrayList<T>(list1);

        list2.removeAll(list1);
        result.addAll(list2);

        return result;
    }

    public SearchResult<User> findUsers(Query<User> query) throws EntityException
    {
        queryValidator.assertValid(query);
        Pager<User> pager = find(query);
        return new DefaultSearchResult<User>(pager, repository.getKey());
    }

    public SearchResult<Group> findGroups(Query<Group> query) throws EntityException
    {
        queryValidator.assertValid(query);
        Pager<Group> pager = find(query);
        return new DefaultSearchResult<Group>(pager, repository.getKey());
    }

    public SearchResult<User> findUsers(Query<User> query, QueryContext context) throws EntityException
    {
        if (!context.contains(repository))
            return null;

        return findUsers(query);
    }

    public SearchResult<Group> findGroups(Query<Group> query, QueryContext context) throws EntityException
    {
        if (!context.contains(repository))
            return null;

        return findGroups(query);
    }
}