package com.atlassian.user.impl.osuser;

import com.atlassian.user.*;
import com.atlassian.user.Entity;
import com.atlassian.user.User;
import com.atlassian.user.Group;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.EntityNameAlphaComparator;
import com.atlassian.user.search.page.DefaultPager;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.util.Assert;
import com.opensymphony.user.*;
import com.opensymphony.user.provider.AccessProvider;
import org.apache.log4j.Logger;

import java.text.Collator;
import java.util.*;

/**
 * An adaptor class for {@link com.opensymphony.user.provider.AccessProvider} and some of the higher level operations of
 * {@link com.opensymphony.user.UserManager}
 * <p/>
 * The rule is to use the credentialsProvider and/or profileProvider (for propertySets) for most things. Store()
 * operations must be called on the entity itself.
 */
public class OSUGroupManager extends OSUEntityManager implements GroupManager
{
    protected final Logger log = Logger.getLogger(this.getClass());

    private final OSUAccessor osuserAccessor;
    private final GenericAccessProviderWrapper accessProvider;

    public OSUGroupManager(RepositoryIdentifier repository, OSUAccessor accessor)
    {
        super(repository);
        this.osuserAccessor = accessor;
        this.accessProvider = new GenericAccessProviderWrapper(accessor.getAccessProvider());
    }

    public Pager<Group> getGroups()
    {
        SortedSet<Group> atlassianGroups = getGroupsFromAccessProvider();
        return new DefaultPager<Group>(atlassianGroups);
    }

    private SortedSet<Group> getGroupsFromAccessProvider()
    {
        SortedSet<Group> atlassianGroups = new TreeSet<Group>(new EntityNameAlphaComparator());
        for (Object o : accessProvider.list())
        {
            String groupName = (String) o;
            Group atlassianGroup = getGroup(groupName);

            if (atlassianGroup != null)
                atlassianGroups.add(atlassianGroup);
        }
        return atlassianGroups;
    }

    public Group createGroup(String groupName) throws EntityException
    {
        Group group = null;

        if (accessProvider.handles(groupName))
            throw new com.atlassian.user.impl.DuplicateEntityException(
                "Group named [" + groupName + "] already exists in accessProvider ["
                    + accessProvider.toString());

        /**
         * Whenever a load(entityName) is called on a UserProvider the result
         * is, in fact, a reconstructed object, as below.
         *
         * This should obviously be cached.
         */
        if (accessProvider.create(groupName))
            group = new OSUGroup(new com.opensymphony.user.Group(groupName, osuserAccessor));

        return group;
    }

    public void removeGroup(Group group) throws EntityException, IllegalArgumentException
    {
        if (group == null)
            throw new IllegalArgumentException("Group is null.");
        else if (!(group instanceof OSUGroup))
            throw new IllegalArgumentException("User is not a OSUGroup [" + group.getClass().getName());

        Group groupToRemove = getGroup(group.getName());

        List<String> users = accessProvider.listUsersInGroup(groupToRemove.getName());
        users = new ArrayList<String>(users);

        for (String username : users)
        {
            accessProvider.removeFromGroup(username, groupToRemove.getName());
        }

        accessProvider.remove(group.getName());
    }

    public void addMembership(Group group, User user)
    {
        if (group == null || getGroup(group.getName()) == null)
            throw new IllegalArgumentException(
                "Cannot add membership for unknown group: [" + (group == null ? "null" : group.getName()) + "]");

        accessProvider.addToGroup(user.getName(), group.getName());
    }

    public boolean hasMembership(Group group, User user)
    {
        /**
         * if the group is not an OpenSymphony wrapper, this manager will not have membership information
         * for the user.
         */
        if (!(group instanceof OSUGroup))
            return false;

        return accessProvider.inGroup(user.getName(), group.getName());
    }

    public void removeMembership(Group group, User user)
    {
        if (group == null || getGroup(group.getName()) == null)
            throw new IllegalArgumentException("Can't remove membership for unknown group: [" +
                (group == null ? "null" : group.getName()) + "]");
        accessProvider.removeFromGroup(user.getName(), group.getName());
    }

    public boolean isReadOnly(Group group) throws EntityException
    {
        return !(accessProvider.handles(group.getName()));
    }

    public boolean supportsExternalMembership() throws EntityException
    {
        return false;
    }

    public Pager<String> getMemberNames(Group group) throws EntityException
    {
        if (!(group instanceof OSUGroup))
            return DefaultPager.emptyPager();

        List<String> memberNames = new ArrayList<String>(accessProvider.listUsersInGroup(group.getName()));
        memberNames.removeAll(Arrays.asList(new Object[]{ null })); // remove all nulls
        Collections.sort(memberNames, Collator.getInstance());

        return new DefaultPager<String>(memberNames);
    }

    public Pager<String> getLocalMemberNames(Group group) throws EntityException
    {
        if (!(group instanceof OSUGroup))
            return DefaultPager.emptyPager();

        List<String> memberNames = new ArrayList<String>(accessProvider.listUsersInGroup(group.getName()));
        Collections.sort(memberNames, Collator.getInstance());
        return new DefaultPager<String>(memberNames);
    }

    public Pager<String> getExternalMemberNames(Group group) throws EntityException
    {
        throw new UnsupportedOperationException("External membership is not supported.");
    }

    public void saveGroup(Group group) throws EntityException
    {
        if (!(accessProvider.handles(group.getName())))
            return;

        com.opensymphony.user.Group g = new com.opensymphony.user.Group(group.getName(), osuserAccessor);

        try
        {
            g.store();
        }
        catch (ImmutableException e)
        {
            throw new RepositoryException(e);
        }
    }

    public Group getGroup(String groupName)
    {
        if (!accessProvider.handles(groupName))
            return null;

        com.opensymphony.user.Group osgroup = new com.opensymphony.user.Group(groupName, osuserAccessor);
        return new OSUGroup(osgroup);
    }

    public Pager<Group> getGroups(User user) throws RepositoryException
    {
        Assert.notNull(user, "User must not be null");
        if (!osuserAccessor.getCredentialsProvider().handles(user.getName()))
            return DefaultPager.emptyPager();

        final SortedSet<Group> groups = new TreeSet<Group>(new EntityNameAlphaComparator());

        Collection<String> groupNames = accessProvider.listGroupsContainingUser(user.getName());
        for (String groupName : groupNames)
        {
            groups.add(getGroup(groupName));
        }

        return new DefaultPager<Group>(groups);
    }

    public List<Group> getWritableGroups()
    {
        return new ArrayList<Group>(getGroupsFromAccessProvider());
    }

    /**
     * @return the {@link com.atlassian.user.repository.RepositoryIdentifier} which is managed by this instance.
     */
    public RepositoryIdentifier getIdentifier()
    {
        return repository;
    }

    public RepositoryIdentifier getRepository(Entity entity) throws EntityException
    {
        if (getGroup(entity.getName()) != null)
            return repository;

        return null;
    }

    /**
     * @return mutability is hard-coded within each {@link com.opensymphony.user.provider.UserProvider} then applied to
     *         the entity produced (here a {@link com.opensymphony.user.Group}).
     *         <p/>
     *         All providers in the base OSUUser package are hardcoded to be mutable. This implementation returns the
     *         mutability value of the first group returned by {@link com.opensymphony.user.provider.UserProvider#list()}.
     *         <p/>
     *         If there are no groups it returns true.
     *         <p/>
     *         Thus, if you want immutable groups you should override this method or ensure that there is at least one
     *         entity which is handled by the provider. All groups should have the same mutability value.
     */
    public boolean isCreative()
    {
        List groupNames = accessProvider.list();
        if (groupNames.isEmpty())
            return true;

        String groupName = (String) groupNames.get(0);
        return new com.opensymphony.user.Group(groupName, osuserAccessor).isMutable();
    }

    @SuppressWarnings("unchecked")
    private static class GenericAccessProviderWrapper
    {
        private final AccessProvider provider;

        public GenericAccessProviderWrapper(AccessProvider provider)
        {
            this.provider = provider;
        }

        public boolean addToGroup(String s, String s1)
        {
            return provider.addToGroup(s, s1);
        }

        public boolean inGroup(String s, String s1)
        {
            return provider.inGroup(s, s1);
        }

        public List<String> listGroupsContainingUser(String s)
        {
            return provider.listGroupsContainingUser(s);
        }

        public List<String> listUsersInGroup(String s)
        {
            return provider.listUsersInGroup(s);
        }

        public boolean removeFromGroup(String s, String s1)
        {
            return provider.removeFromGroup(s, s1);
        }

        public boolean create(String s)
        {
            return provider.create(s);
        }

        public void flushCaches()
        {
            provider.flushCaches();
        }

        public boolean handles(String s)
        {
            return provider.handles(s);
        }

        public boolean init(Properties properties)
        {
            return provider.init(properties);
        }

        public List list()
        {
            return provider.list();
        }

        public boolean load(String s, com.opensymphony.user.Entity.Accessor accessor)
        {
            return provider.load(s, accessor);
        }

        public boolean remove(String s)
        {
            return provider.remove(s);
        }

        public boolean store(String s, com.opensymphony.user.Entity.Accessor accessor)
        {
            return provider.store(s, accessor);
        }
    }
}