package com.atlassian.user.configuration.xml;

import org.dom4j.Node;
import org.dom4j.DocumentException;
import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

import java.util.*;
import java.io.IOException;
import java.io.FileNotFoundException;
import java.net.URL;

import com.atlassian.user.util.ClassLoaderUtils;
import com.atlassian.user.configuration.Configuration;

/**
 * Loads the default configuration for the available atlassian-user repositories. These defaults define
 * fall-through values for any configuration option not specified in the proper atlassian-user config.
 *
 * <p>The parser will search the classpath roots for all files matching the defaults config file
 * name (<code>atlassian-user-defaults.xml</code>), and load the default configuration values from them.
 * If defaults are configured for the same repository type in multiple defaults files, the behaviour is
 * undefined.
 *
 * <p>For an example of the file format of the defaults file, just look at the <code>atlassian-user-defaults.xml</code>
 * included in atlassian-user.
 */
public class XMLDefaultsParser
{
    private static final Logger log = Logger.getLogger(XMLDefaultsParser.class);

    public static final String DEFAULTS_FILE_NAME = "atlassian-user-defaults.xml";

    private final List<Node> defaultsBaseNodes = new ArrayList<Node>();

    public XMLDefaultsParser() throws IOException, DocumentException
    {
        this(DEFAULTS_FILE_NAME);
    }

    public XMLDefaultsParser(String defaultsFileName) throws IOException, DocumentException
    {
        this(new String[] { defaultsFileName });
    }

    private void initialiseDefaultsNodesForFile(String defaultsFileName)
            throws IOException, DocumentException
    {
        Enumeration defaultsFileUrls = ClassLoaderUtils.getResources(defaultsFileName, this.getClass());

        while (defaultsFileUrls.hasMoreElements())
        {
            URL url = (URL) defaultsFileUrls.nextElement();
            Node defaultsNode = findBaseNodeInFile(url);
            if (defaultsNode != null)
                defaultsBaseNodes.add(defaultsNode);
            else
                log.error("Unable to find valid atlassian-user defaults data in file: " + url);
        }
    }

    public XMLDefaultsParser(String[] defaultsFileNames) throws DocumentException, IOException
    {
        for (String defaultsFileName : defaultsFileNames)
            initialiseDefaultsNodesForFile(defaultsFileName);

        if (defaultsBaseNodes.isEmpty())
            throw new FileNotFoundException("No valid user defaults files found in classpath with name: " + StringUtils.join(defaultsFileNames, ", "));
    }

    public Map<String, String> getDefaultClassesConfigForKey(String key) throws DocumentException, IOException
    {
        Map<String, String> defaults = new HashMap<String, String>();

        for (Node node : defaultsBaseNodes)
        {
            Node defaultsNode = node.selectSingleNode(key);

            if (defaultsNode != null)
                defaults.putAll(XMLConfigUtil.parseRepositoryElementForClassNames((Element) defaultsNode));
        }

        return defaults;
    }

    private Node findBaseNodeInFile(URL url) throws DocumentException
    {
        SAXReader reader = new SAXReader();
        Document doc = reader.read(url);

        return doc.selectSingleNode("//" + Configuration.DEFAULT);
    }

    public Map<String, String> getDefaultParameterConfigForKey(String key)
    {
        Map<String, String> defaults = new HashMap<String, String>();

        for (Node node : defaultsBaseNodes)
        {
            Node defaultsNode = node.selectSingleNode(key);

            if (defaultsNode != null)
            {
                defaults.putAll(XMLConfigUtil.parseRepositoryElementForStringData((Element) defaultsNode));
                break;
            }
        }

        return defaults;
    }
}
