package com.atlassian.user.util;

import org.apache.log4j.Logger;

import java.io.*;

/*
 * Atlassian Source Code Template.
 * User: anton
 * Date: 15/08/2003
 * Time: 11:38:41
 * CVS Revision: $Revision: 2774 $
 * Last CVS Commit: $Date: 2006-01-12 22:12:04 -0600 (Thu, 12 Jan 2006) $
 * Author of last CVS Commit: $Author: nfaiz $
 */

public class FileUtils
{
    private static final int DEFAULT_BUFFER_SIZE = 1024 * 4;
    private static final Logger log = Logger.getLogger(FileUtils.class);

    /**
     * Copy file from source to destination. The directories up to <code>destination</code> will be
     * created if they don't already exist. <code>destination</code> will be overwritten if it
     * already exists.
     *
     * @param source      An existing non-directory <code>File</code> to copy bytes from.
     * @param destination A non-directory <code>File</code> to write bytes to (possibly
     *                    overwriting).
     * @throws java.io.IOException if <code>source</code> does not exist, <code>destination</code> cannot be
     *                             written to, or an IO error occurs during copying.
     */
    public static void copyFile(final File source, final File destination)
            throws IOException
    {
        //check source exists
        if (!source.exists())
        {
            final String message = "File " + source + " does not exist";
            throw new IOException(message);
        }

        //does destinations directory exist ?
        if (destination.getParentFile() != null &&
                !destination.getParentFile().exists())
        {
            destination.getParentFile().mkdirs();
        }

        //make sure we can write to destination
        if (destination.exists() && !destination.canWrite())
        {
            final String message = "Unable to open file " +
                    destination + " for writing.";
            throw new IOException(message);
        }

        final FileInputStream input = new FileInputStream(source);
        final FileOutputStream output = new FileOutputStream(destination);
        copy(input, output);
        shutdownStream(input);
        shutdownStream(output);

        if (source.length() != destination.length())
        {
            final String message = "Failed to copy full contents from " + source +
                    " to " + destination;
            throw new IOException(message);
        }
    }

    /**
     * Copy bytes from an <code>InputStream</code> to an <code>OutputStream</code>.
     *
     * @param input  the <code>InputStream</code> to read from
     * @param output the <code>OutputStream</code> to write to
     * @return the number of bytes copied
     * @throws IOException In case of an I/O problem
     */
    public static int copy(final InputStream input, final OutputStream output)
            throws IOException
    {
        return copy(input, output, DEFAULT_BUFFER_SIZE);
    }

    /**
     * Copy bytes from an <code>InputStream</code> to an <code>OutputStream</code>.
     *
     * @param input      the <code>InputStream</code> to read from
     * @param output     the <code>OutputStream</code> to write to
     * @param bufferSize Size of internal buffer to use.
     * @return the number of bytes copied
     * @throws IOException In case of an I/O problem
     */
    public static int copy(final InputStream input,
                           final OutputStream output,
                           final int bufferSize)
            throws IOException
    {
        final byte[] buffer = new byte[bufferSize];
        int count = 0;
        int n = 0;
        while (-1 != (n = input.read(buffer)))
        {
            output.write(buffer, 0, n);
            count += n;
        }
        return count;
    }

    /**
     * Unconditionally close an <code>OutputStream</code>.
     * Equivalent to {@link OutputStream#close()}, except any exceptions will be ignored.
     *
     * @param output A (possibly null) OutputStream
     */
    public static void shutdownStream(final OutputStream output)
    {
        if (output == null)
        {
            return;
        }

        try
        {
            output.close();
        }
        catch (final IOException ioe)
        {
        }
    }

    /**
     * Unconditionally close an <code>InputStream</code>.
     * Equivalent to {@link InputStream#close()}, except any exceptions will be ignored.
     *
     * @param input A (possibly null) InputStream
     */
    public static void shutdownStream(final InputStream input)
    {
        if (input == null)
        {
            return;
        }

        try
        {
            input.close();
        }
        catch (final IOException ioe)
        {
        }
    }


    /**
     * safely performs a recursive delete on a directory
     *
     * @param dir
     * @return
     */
    public static boolean deleteDir(File dir)
    {
        if (dir == null)
        {
            return false;
        }

        // to see if this directory is actually a symbolic link to a directory,
        // we want to get its canonical path - that is, we follow the link to
        // the file it's actually linked to
        File candir;
        try
        {
            candir = dir.getCanonicalFile();
        }
        catch (IOException e)
        {
            return false;
        }

        // a symbolic link has a different canonical path than its actual path,
        // unless it's a link to itself
        if (!candir.equals(dir.getAbsoluteFile()))
        {
            // this file is a symbolic link, and there's no reason for us to
            // follow it, because then we might be deleting something outside of
            // the directory we were told to delete
            return false;
        }

        // now we go through all of the files and subdirectories in the
        // directory and delete them one by one
        File[] files = candir.listFiles();
        if (files != null)
        {
            for (int i = 0; i < files.length; i++)
            {
                File file = files[i];

                // in case this directory is actually a symbolic link, or it's
                // empty, we want to try to delete the link before we try
                // anything
                boolean deleted = !file.delete();
                if (deleted)
                {
                    // deleting the file failed, so maybe it's a non-empty
                    // directory
                    if (file.isDirectory()) deleteDir(file);

                    // otherwise, there's nothing else we can do
                }
            }
        }

        // now that we tried to clear the directory out, we can try to delete it
        // again
        return dir.delete();
    }

    public static void deleteFilesBeginningWith(String directory, String prefix)
    {
        File dir = new File(directory);

        if (!dir.isDirectory())
            throw new IllegalArgumentException("directory arg. is not a dir. [" + directory + "]");

        File[] files = dir.listFiles();

        for (int i = 0; i < files.length; i++)
        {
            File file = files[i];
            if (file.getName().startsWith(prefix))
            {
                if (!file.delete())
                    throw new RuntimeException("Could not delete " + file.getName());
            }
        }
    }

    /**
     * creates a temp file with a given filename from an input stream in the classpath or a file in the filesystem
     *
     * @param fileName
     * @return
     * @throws IOException
     */
    public static File copyIntoTemporaryFile(String source, String fileName) throws IOException, Exception
    {
        File temp = File.createTempFile(fileName, null);
        temp.deleteOnExit();

        // Write content to the tempfile
        BufferedWriter out = new BufferedWriter(new FileWriter(temp));
        out.write(source);
        out.close();

        return temp;
    }

    public static String getInputStreamTextContent(InputStream is)
    {
        if (is == null)
        {
            return null;
        }

        String result = null;

        try
        {
            ByteArrayOutputStream baos = new ByteArrayOutputStream(is.available());

            pump(is, baos);

            result = new String(baos.toByteArray());

            is.close();
        }
        catch (IOException e)
        {
            log.error("IOException reading stream: " + e, e);
        }

        return result;
    }


    private static void pump(InputStream is, OutputStream os) throws IOException
    {
        byte[] buffer = new byte[DEFAULT_BUFFER_SIZE];
        int lengthRead;

        while ((lengthRead = is.read(buffer)) >= 0)
        {
            os.write(buffer, 0, lengthRead);
        }
    }
}
