package com.atlassian.user.security.password;

/**
 * Represents a user's password. Allows the same methods to take both encrypted and unencrypted credentials.
 *
 * @since 2.2
 */
public final class Credential
{
    /**
     * The null or empty credential, which is encrypted and has an invalid hash so it can never be authenticated against.
     */
    public static final Credential NONE = Credential.encrypted("x");

    private final boolean encrypted;
    private final String value;

    public static Credential encrypted(String hash)
    {
        return new Credential(true, hash);
    }

    public static Credential unencrypted(String password)
    {
        return new Credential(false, password);
    }

    private Credential(boolean encrypted, String value)
    {
        if (value == null) throw new IllegalArgumentException("Credential value cannot be null. Use Credential.NONE instead.");
        this.encrypted = encrypted;
        this.value = value;
    }

    /**
     * Returns {@code true} if the string returned by {@link #getValue()} is an encrypted hash of the password,
     * rather than the plaintext password itself.
     */
    public boolean isEncrypted()
    {
        return encrypted;
    }

    /**
     * Returns the password, either in plain text if {@link #isEncrypted()} returns false, or as a SHA1-512
     * hash if {@link #isEncrypted()} returns true.
     */
    public String getValue()
    {
        return value;
    }

    @Override
    public boolean equals(Object o)
    {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Credential that = (Credential) o;
        return encrypted == that.encrypted && value.equals(that.value);
    }

    @Override
    public int hashCode()
    {
        return 31 * (encrypted ? 1 : 0) + value.hashCode();
    }
}
