package com.atlassian.user.impl.hibernate;

import com.atlassian.user.*;
import com.atlassian.user.impl.DuplicateEntityException;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.impl.hibernate.repository.HibernateRepository;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.page.DefaultPager;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.security.password.Credential;
import com.atlassian.user.security.password.PasswordEncryptor;
import com.atlassian.user.util.Assert;
import net.sf.hibernate.HibernateException;
import net.sf.hibernate.Query;
import net.sf.hibernate.Session;
import org.springframework.dao.DataAccessException;
import org.springframework.orm.hibernate.HibernateCallback;
import org.springframework.orm.hibernate.SessionFactoryUtils;
import org.springframework.orm.hibernate.support.HibernateDaoSupport;

import java.util.List;
import java.util.Set;

public class HibernateUserManager extends HibernateDaoSupport implements UserManager
{
    private static final String USERNAME_FIELD = "username";
    public static final String ENTITYID_FIELD = "entityid";

    private final RepositoryIdentifier identifier;
    private final PasswordEncryptor passwordEncryptor;

    public HibernateUserManager(RepositoryIdentifier identifier, HibernateRepository repository, PasswordEncryptor passwordEncryptor)
    {
        this.identifier = identifier;
        this.passwordEncryptor = passwordEncryptor;
        setSessionFactory(repository.getSessionFactory());
    }

    public Pager<User> getUsers() throws EntityException
    {
        try
        {
            return new DefaultPager<User>(getUsersFromHibernate());
        }
        catch (DataAccessException e)
        {
            throw new RepositoryException(e);
        }
    }

    public Pager<String> getUserNames() throws EntityException
    {
        try
        {
            return new DefaultPager<String>(getUsernamesFromHibernate());
        }
        catch (DataAccessException e)
        {
            throw new RepositoryException(e);
        }
    }

    /**
     * @return an {@link User} if one could be found, otherwise null.
     * @throws EntityException representing the exception which prohibited looking for or retrieving the user.
     */
    public User getUser(final String username) throws EntityException
    {
        return internalGetUser(username);
    }

    public User createUser(String username) throws EntityException
    {
        validateNewUserName(username);

        User user = new DefaultHibernateUser(username);
        getHibernateTemplate().save(user);

        return user;
    }

    public User createUser(User userTemplate, Credential credential) throws EntityException
    {
        validateNewUserName(userTemplate.getName());

        DefaultHibernateUser user = new DefaultHibernateUser(userTemplate.getName());
        user.setFullName(userTemplate.getFullName());
        user.setEmail(userTemplate.getEmail());
        user.setPassword(passwordEncryptor.getEncryptedValue(credential));
        getHibernateTemplate().save(user);

        return user;
    }

    private void validateNewUserName(String name) throws EntityException
    {
        if (name == null)
            throw new IllegalArgumentException("Username cannot be null.");

        User existingUser = getUser(name);
        if (existingUser != null)
            throw new DuplicateEntityException("User with name [" + name + "] already exists in this repository (" + identifier.getName() + ")");
    }

    /**
     * Encrypts the plain password, sets it on the user, and saves the user.
     */
    public void alterPassword(User user, String password) throws EntityException
    {
        DefaultHibernateUser foundUser = internalGetUser(user.getName());
        if (foundUser == null)
            throw new EntityException("This repository [" + identifier.getName() + "] does not handle user [" + user.getName() + "]");

        String encryptedPassword = passwordEncryptor.encrypt(password);
        foundUser.setPassword(encryptedPassword);
        getHibernateTemplate().saveOrUpdate(foundUser);
    }

    public void saveUser(User user) throws EntityException
    {
        Assert.notNull(user, "User must not be null");
        DefaultHibernateUser persistedUser = internalGetUser(user.getName());
        if (persistedUser == null)
                throw new EntityException("This repository [" + identifier + "] does not handle user [" + user.getName() + "]");

        persistedUser.setFullName(user.getFullName());
        persistedUser.setEmail(user.getEmail());
        getHibernateTemplate().saveOrUpdate(persistedUser);
    }

    /**
     * Removes the specified group, if it is present.
     *
     * @throws EntityException if an exception which prohibited removal
     */
    public void removeUser(User user) throws EntityException
    {
        final DefaultHibernateUser foundUser = internalGetUser(user.getName());
        if (foundUser == null)
            throw new IllegalArgumentException("User can not be found in this user manager: [" + user + "]");

        List<DefaultHibernateGroup> groups = getGroupsForLocalUser(foundUser);
        if (groups != null)
        {
            foundUser.setGroups(null);

            for (DefaultHibernateGroup group : groups)
            {
                Set members = group.getLocalMembers();

                if (members != null)
                    members.remove(foundUser);

                getHibernateTemplate().saveOrUpdate(group);
            }
        }

        getHibernateTemplate().delete(foundUser);
    }

    @SuppressWarnings({"unchecked"})
    private DefaultHibernateUser internalGetUser(final String username) throws RepositoryException
    {
        Assert.notNull(username, "User must not be null");
        List<DefaultHibernateUser> result;
        try
        {
            result = getHibernateTemplate().executeFind(new HibernateCallback()
            {
                public Object doInHibernate(Session session) throws HibernateException
                {
                    Query query = session.getNamedQuery("atluser.user_find");
                    SessionFactoryUtils.applyTransactionTimeout(query, getSessionFactory());
                    query.setCacheable(true);
                    query.setParameter(USERNAME_FIELD, username);
                    return query.list();
                }
            });
        }
        catch (DataAccessException e)
        {
            throw new RepositoryException(e);
        }

        return result.isEmpty() ? null : result.get(0);
    }

    public boolean isReadOnly(User user) throws EntityException
    {
        return false;
    }

    /**
     * @return a {@link com.atlassian.user.security.password.PasswordEncryptor} which handles the encrypytion of passwords for users managed by this
     *         object.
     * @throws UnsupportedOperationException - for {@link com.atlassian.user.UserManager} objects which do not create {@link com.atlassian.user.User} objects.
     */
    @SuppressWarnings({"UnusedDeclaration"})
    public PasswordEncryptor getPasswordEncryptor(User user) throws EntityException
    {
        return passwordEncryptor;
    }


    /**
     * @return the {@link com.atlassian.user.repository.RepositoryIdentifier} which is managed by this instance.
     */
    public RepositoryIdentifier getIdentifier()
    {
        return identifier;
    }

    public RepositoryIdentifier getRepository(Entity entity) throws EntityException
    {
        return getUser(entity.getName()) == null ? null : identifier;
    }

    /**
     * Used to detemine whether an entity can be added (eg, can call {@link com.atlassian.user.UserManager#createUser(String)} or {@link com.atlassian.user.GroupManager#createGroup(String)}
     */
    public boolean isCreative()
    {
        return true;
    }

    @SuppressWarnings({"unchecked"})
    private List<User> getUsersFromHibernate()
    {
        List<User> result;
        result = getHibernateTemplate().executeFind(new HibernateCallback()
        {
            public Object doInHibernate(Session session) throws HibernateException
            {
                Query queryObject = session.getNamedQuery("atluser.user_findAll");
                SessionFactoryUtils.applyTransactionTimeout(queryObject, getSessionFactory());

                return queryObject.list();
            }
        });
        return result;
    }

    @SuppressWarnings("unchecked")
    private List<String> getUsernamesFromHibernate()
    {
        List result;
        result = getHibernateTemplate().executeFind(new HibernateCallback()
        {
            public Object doInHibernate(Session session) throws HibernateException
            {
                Query queryObject = session.getNamedQuery("atluser.user_findAllUserNames");
                SessionFactoryUtils.applyTransactionTimeout(queryObject, getSessionFactory());

                return queryObject.list();
            }
        });
        return result;
    }

    @SuppressWarnings("unchecked")
    private List<DefaultHibernateGroup> getGroupsForLocalUser(final DefaultHibernateUser user) throws RepositoryException
    {
        Assert.notNull(user, "User must not be null");

        try
        {
            return getHibernateTemplate().executeFind(new HibernateCallback()
            {
                public Object doInHibernate(Session session) throws HibernateException
                {
                    Query queryObject = session.getNamedQuery("atluser.group_getGroupsForUser");
                    SessionFactoryUtils.applyTransactionTimeout(queryObject, getSessionFactory());
                    queryObject.setLong(ENTITYID_FIELD, user.getId());
                    queryObject.setCacheable(true);
                    return queryObject.list();
                }
            });
        }
        catch (DataAccessException e)
        {
            throw new RepositoryException(e);
        }
    }
}
