package com.atlassian.user.impl.hibernate;

import com.atlassian.user.*;
import com.atlassian.user.impl.DuplicateEntityException;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.impl.hibernate.repository.HibernateRepository;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.page.DefaultPager;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.search.page.PagerFactory;
import com.atlassian.user.util.Assert;
import net.sf.hibernate.HibernateException;
import net.sf.hibernate.ObjectDeletedException;
import net.sf.hibernate.Query;
import net.sf.hibernate.Session;
import org.springframework.dao.DataAccessException;
import org.springframework.orm.hibernate.HibernateCallback;
import org.springframework.orm.hibernate.HibernateTemplate;
import org.springframework.orm.hibernate.SessionFactoryUtils;
import org.springframework.orm.hibernate.support.HibernateDaoSupport;

import java.util.*;

/**
 * A HibernateGroupManager which handles membership for local and external entities.
 */
public class HibernateGroupManager extends HibernateDaoSupport implements GroupManager
{
    public static final String GROUPNAME_FIELD = "groupname";
    public static final String GROUPID_FIELD = "groupid";
    public static final String ENTITYID_FIELD = "entityid";
    public static final String EXTERNAL_ENTITY_NAME_FIELD = "externalEntityName";

    private final RepositoryIdentifier identifier;
    protected final HibernateRepository repository;
    protected final UserManager userManager;
    protected final ExternalEntityDAO externalEntityDao;

    private static final boolean USE_EXPERIMENTAL_MAPPINGS = Boolean.getBoolean("com.atlassian.user.experimentalMapping");

    public HibernateGroupManager(RepositoryIdentifier identifier, HibernateRepository repository, UserManager userManager, ExternalEntityDAO externalEntityDao)
    {
        this.identifier = identifier;
        this.repository = repository;
        this.userManager = userManager;
        setSessionFactory(repository.getSessionFactory());
        this.externalEntityDao = externalEntityDao;
    }

    public Pager<Group> getGroups() throws EntityException
    {
        List<Group> result;

        try
        {
            result = getGroupsFromHibernate();
        }
        catch (DataAccessException e)
        {
            throw new RepositoryException(e);
        }

        if (result == null)
            return DefaultPager.emptyPager();

        return new DefaultPager<Group>(result);
    }

    public List<Group> getWritableGroups()
    {
        return getGroupsFromHibernate();
    }

    public Pager<Group> getGroups(User user) throws EntityException
    {
        Collection<Group> groups = getAllGroupsForUser(user);
        return new DefaultPager<Group>(groups);
    }

    protected Collection<Group> getAllGroupsForUser(User user) throws RepositoryException
    {
        Assert.notNull(user, "User must not be null");

        if (isUserExternal(user))
            return getGroupsForExternalEntity(getCorrespondingExternalEntity(user));

        return getGroupsForLocalUser(user);
    }

    /**
     * for the time being, lets define an internal user as a hibernate user. All other impl's of User are external users.
     */
    protected boolean isUserExternal(User user)
    {
        return !(user instanceof DefaultHibernateUser);
    }

    private List<Group> getGroupsForLocalUser(User user) throws RepositoryException
    {
        Assert.notNull(user, "User must not be null");
        Assert.isInstanceOf(DefaultHibernateUser.class, user);

        if (isUserExternal(user))
            return Collections.emptyList();

        return getLocalUserGroupsFromHibernate((DefaultHibernateUser) user);
    }

    private List<Group> getGroupsForExternalEntity(final ExternalEntity externalEntity) throws RepositoryException
    {
        if (externalEntity == null)
            throw new IllegalArgumentException("Input (externalEntity) is null.");

        return getExternalUserGroupsFromHibernate(externalEntity);
    }

    /**
     * @return a {@link Pager} instance which can hold {@link User} <b>and</b> {@link ExternalEntity} objects.
     */
    public Pager<String> getMemberNames(Group group) throws EntityException
    {
        if (group == null)
            throw new IllegalArgumentException("Group cannot be null.");
        if (!isHandledGroup(group))
            throw new IllegalArgumentException("Group passed to HibernateGroupManager must be of type 'DefaultHibernateGroup'");

        // We don't try to merge the result returned by getExternalMemberNames(group) and getLocalMemberNames(group)
        // because it is not possible with hql(not implemented) or native sql through hibernate (bug in hibernate)
        // When iterating through returned result client will get ordered internal members and then ordered external members.
        // To order everything together we can use SQL union but HQL does not support it yet
        // http://opensource.atlassian.com/projects/hibernate/browse/HHH-1050
        return PagerFactory.getPager(getExternalMemberNames(group), getLocalMemberNames(group));
    }

    protected void validateGroup(Group group)
    {
        if (group == null)
            throw new IllegalArgumentException("Input (group) is null.");
    }

    public Pager<String> getLocalMemberNames(Group group) throws EntityException
    {
        validateGroup(group);
        return new DefaultPager<String>(getLocalMemberNamesFromHibernate((DefaultHibernateGroup) group));
    }

    public Pager<User> getLocalMembers(Group group) throws RepositoryException
    {
        if (group == null)
            throw new IllegalArgumentException("Input (group) is null.");
        else if (!isHandledGroup(group)) //nothing here for a non-Hibernate group
            return DefaultPager.emptyPager();

        DefaultHibernateGroup defGroup = (DefaultHibernateGroup) group;
        return new DefaultPager<User>(new ArrayList<User>(defGroup.getLocalMembers()));
    }

    private boolean isHandledGroup(Group group)
    {
        return (group instanceof DefaultHibernateGroup);
    }

    public Pager<String> getExternalMemberNames(Group group) throws EntityException
    {
        if (group == null)
            throw new IllegalArgumentException("Input (group) is null.");
        else if (!isHandledGroup(group))
            return DefaultPager.emptyPager();

        final DefaultHibernateGroup defGroup = (DefaultHibernateGroup) group;

        return new DefaultPager<String>(getExternalMemberNamesFromHibernate(defGroup));
    }

    public DefaultHibernateGroup getGroup(Group group) throws EntityException
    {
        //for permormance try to fetch a group by id if it is hibernate group
        if (group instanceof DefaultHibernateGroup)
        {
           // let's make sure we load this group in Session
            try
            {
                return (DefaultHibernateGroup) getSession().get(DefaultHibernateGroup.class, ((DefaultHibernateGroup) group).getId());
            }
            catch (ObjectDeletedException e)
            {
                return null;
            }
            catch (HibernateException e)
            {
                throw new EntityException(e);
            }
        }
        else
        {
            return getGroup(group.getName());
        }

    }


    public DefaultHibernateGroup getGroup(final String groupname) throws EntityException
    {
        if (groupname == null)
            throw new IllegalArgumentException("Input (groupname) is null.");

        List result;
        DefaultHibernateGroup foundGroup = null;

        try
        {
            result = getHibernateTemplate().executeFind(new HibernateCallback()
            {
                public Object doInHibernate(Session session) throws HibernateException
                {
                    Query queryObject = session.getNamedQuery("atluser.group_find");
                    SessionFactoryUtils.applyTransactionTimeout(queryObject, getSessionFactory());

                    if (groupname != null)
                        queryObject.setParameter(GROUPNAME_FIELD, groupname);

                    return queryObject.list();
                }
            });
        }
        catch (DataAccessException e)
        {
            throw new RepositoryException(e);
        }

        try
        {
            foundGroup = (DefaultHibernateGroup) result.get(0);
        }
        catch (Exception e)
        {
            return foundGroup;
        }

        return foundGroup;
    }

    public Group createGroup(String groupname) throws EntityException
    {
        if (groupname == null)
            throw new IllegalArgumentException("Input (groupname) is null.");

        Group group = getGroup(groupname);

        if (group == null)
        {
            group = new DefaultHibernateGroup(groupname);
            getHibernateTemplate().save(group);
        }
        else
            throw new DuplicateEntityException("Group [" + groupname + "] already exists in this repository (" +
                    identifier.getName() + ")");

        return group;
    }

    /**
     * Removes the specified group, if it is present.
     *
     * @throws com.atlassian.user.EntityException
     *          - representing the exception which prohibited removal
     */
    public void removeGroup(Group group) throws EntityException
    {
        Group groupInSession = getGroupInSession(group);

        DefaultHibernateGroup dGroup = (DefaultHibernateGroup) groupInSession;
        dGroup.setExternalMembers(null);
        dGroup.setLocalMembers(null);

        getHibernateTemplate().delete(groupInSession);
    }

    public void addMembership(Group group, User user) throws EntityException
    {
        validateGroupAndUser(group, user);

        DefaultHibernateGroup dGroup = getGroupInSession(group);

        if (isUserExternal(user))
        {
            addExternalUserMembership(user, dGroup);
        }
        else
        {
            addLocalUserMembership(user, dGroup);
        }

        getHibernateTemplate().saveOrUpdate(dGroup);
    }

    private void addLocalUserMembership(User user, DefaultHibernateGroup dGroup)
    {
        if (USE_EXPERIMENTAL_MAPPINGS)
        {
            DefaultHibernateUser huser = (DefaultHibernateUser) user;
            huser.getGroups().add(dGroup);
            getHibernateTemplate().saveOrUpdate(user);
        }
        else
        {
            if (dGroup.getLocalMembers() == null)
                dGroup.setLocalMembers(new HashSet<User>());

            dGroup.getLocalMembers().add(user);
        }
    }

    private void addExternalUserMembership(User user, DefaultHibernateGroup dGroup)
            throws RepositoryException
    {
        if (dGroup.getExternalMembers() == null)
            dGroup.setExternalMembers(new HashSet<ExternalEntity>());

        dGroup.getExternalMembers().add(getCorrespondingExternalEntity(user));
    }

    protected ExternalEntity getCorrespondingExternalEntity(final User user) throws RepositoryException
    {
        if (user == null)
            throw new IllegalArgumentException("Input (user) is null.");

        ExternalEntity result = externalEntityDao.getExternalEntity(user.getName());

        if (result == null)
            return externalEntityDao.createExternalEntity(user.getName());
        else
            return result;
    }

    public boolean hasMembership(Group group, User user) throws EntityException
    {
        if (group==null || getGroup(group) == null)
            return false;

        validateGroupAndUser(group, user);

        final DefaultHibernateGroup defGroup = getGroupInSession(group);

        if (isUserExternal(user))
        {
            return hasExternalMembership(defGroup, user);
        }
        else
        {
            return hasLocalMembership(defGroup, (DefaultHibernateUser) user);
        }
    }

    protected void validateGroupAndUser(Group group, User user) throws EntityException
    {
        if (group == null)
            throw new IllegalArgumentException("Can't add membership for null group");

        if (getGroup(group) == null)
            throw new IllegalArgumentException("Group unknown: [" + group + "] in [" + identifier.getKey() + "]");

        if (user == null)
            throw new IllegalArgumentException("User unknown: [" + user + "] in [" + identifier.getKey() + "]");

        if (!isHandledGroup(group))
            throw new IllegalArgumentException("Group is not a Hibernate entity [" + group.getClass().getName());
    }

    protected boolean hasExternalMembership(DefaultHibernateGroup defGroup, User user) throws EntityException
    {
        ExternalEntity entity = getCorrespondingExternalEntity(user);

        return defGroup.getExternalMembers().contains(entity);
    }

    protected boolean hasLocalMembership(DefaultHibernateGroup defGroup, DefaultHibernateUser defUser) throws EntityException
    {
        Collection usersGroups = getAllGroupsForUser(defUser);

        return (usersGroups != null && usersGroups.contains(defGroup));
    }

    public void removeMembership(Group group, User user) throws EntityException
    {
        validateGroupAndUser(group, user);

        Group groupInSession = getGroupInSession(group);

        DefaultHibernateGroup hibernateGroup = (DefaultHibernateGroup) groupInSession;

        HibernateTemplate hibernateTemplate = getHibernateTemplate();
        if (isUserExternal(user))
        {
            ExternalEntity extUser = getCorrespondingExternalEntity(user);

            // remove user from group
            hibernateGroup.getExternalMembers().remove(extUser);

        }
        else
        {
            if(USE_EXPERIMENTAL_MAPPINGS) {
                DefaultHibernateUser huser = (DefaultHibernateUser)user;
                huser.getGroups().remove(groupInSession);
                hibernateTemplate.saveOrUpdate(huser);
            }

            else {
            // remove user from group
            hibernateGroup.getLocalMembers().remove(user);
            hibernateTemplate.saveOrUpdate(groupInSession);
            }
        }

        hibernateTemplate.flush();
    }

    public boolean isReadOnly(Group group) throws EntityException
    {
        return (getGroup(group) == null);
    }

    public boolean supportsExternalMembership() throws EntityException
    {
        return true;
    }

    /**
     * @return the {@link com.atlassian.user.repository.RepositoryIdentifier} which is managed by this instance.
     */
    public RepositoryIdentifier getIdentifier()
    {
        return identifier;
    }

    public RepositoryIdentifier getRepository(Entity entity) throws EntityException
    {
        if (getGroup(entity.getName()) != null)
            return identifier;

        return null;
    }

    /**
     * Used to detemine whether an entity can be added (eg, can call {@link com.atlassian.user.UserManager#createUser(String)} or
     * {@link com.atlassian.user.GroupManager#createGroup(String)}
     */
    public boolean isCreative()
    {
        return true;
    }

    /**
     * Used to get a group from Hibernate (in session) from the "cached" version.
     * Ensures that we do not have un-attached group objects.
     */
    private DefaultHibernateGroup getGroupInSession(Group group)
            throws EntityException
    {
        if (group == null)
            throw new IllegalArgumentException("Input (group) is null.");
        else if (!isHandledGroup(group))
            throw new IllegalArgumentException("Group is not a Hibernate entity [" + group.getClass().getName());

        return getGroup(group);
    }

    @SuppressWarnings("unchecked")
    private List<Group> getGroupsFromHibernate()
    {
        return getHibernateTemplate().executeFind(new HibernateCallback()
        {
            public Object doInHibernate(Session session) throws HibernateException
            {
                Query queryObject = session.getNamedQuery("atluser.group_findAll");
                SessionFactoryUtils.applyTransactionTimeout(queryObject, getSessionFactory());
                return queryObject.list();
            }
        });
    }


    @SuppressWarnings("unchecked")
    private List<Group> getLocalUserGroupsFromHibernate(final DefaultHibernateUser defUser) throws RepositoryException
    {
        try
        {
            return getHibernateTemplate().executeFind(new HibernateCallback()
            {
                public Object doInHibernate(Session session) throws HibernateException
                {
                    Query queryObject = session.getNamedQuery("atluser.group_getGroupsForUser");
                    SessionFactoryUtils.applyTransactionTimeout(queryObject, getSessionFactory());
                    queryObject.setLong(ENTITYID_FIELD, defUser.getId());

                    return queryObject.list();
                }
            });
        }
        catch (DataAccessException e)
        {
            throw new RepositoryException(e);
        }
    }

    @SuppressWarnings("unchecked")
    private List<Group> getExternalUserGroupsFromHibernate(final ExternalEntity externalEntity) throws RepositoryException
    {
        try
        {
            return getHibernateTemplate().executeFind(new HibernateCallback()
            {
                public Object doInHibernate(Session session) throws HibernateException
                {
                    Query queryObject = session.getNamedQuery("atluser.group_getGroupsForExternalEntity");
                    SessionFactoryUtils.applyTransactionTimeout(queryObject, getSessionFactory());

                    if (externalEntity != null)
                        queryObject.setLong(ENTITYID_FIELD, externalEntity.getId());

                    return queryObject.list();
                }
            });
        }
        catch (DataAccessException e)
        {
            throw new RepositoryException(e);
        }
    }

    @SuppressWarnings("unchecked")
    private List<String> getLocalMemberNamesFromHibernate(final DefaultHibernateGroup defGroup) throws RepositoryException
    {
        try
        {
            return getHibernateTemplate().executeFind(new HibernateCallback()
            {
                public Object doInHibernate(Session session) throws HibernateException
                {
                    Query queryObject = session.getNamedQuery("atluser.group_getLocalMemberNames");
                    SessionFactoryUtils.applyTransactionTimeout(queryObject, getSessionFactory());

                    if (defGroup != null)
                        queryObject.setLong(GROUPID_FIELD, defGroup.getId());

                    return queryObject.list();
                }
            });
        }
        catch (DataAccessException e)
        {
            throw new RepositoryException(e);
        }
    }

    @SuppressWarnings("unchecked")
    private List<String> getExternalMemberNamesFromHibernate(final DefaultHibernateGroup defGroup)
            throws RepositoryException
    {
        List result;
        try
        {
            result = getHibernateTemplate().executeFind(new HibernateCallback()
            {
                public Object doInHibernate(Session session) throws HibernateException
                {
                    Query queryObject = session.getNamedQuery("atluser.group_getExternalMemberNames");
                    SessionFactoryUtils.applyTransactionTimeout(queryObject, getSessionFactory());

                    if (defGroup != null)
                        queryObject.setLong(GROUPID_FIELD, defGroup.getId());

                    return queryObject.list();
                }
            });
        }
        catch (DataAccessException e)
        {
            throw new RepositoryException(e);
        }
        return result;
    }
}
