package com.atlassian.user.configuration.xml;

import org.dom4j.*;
import org.dom4j.io.SAXReader;
import org.apache.log4j.Logger;

import java.util.*;
import java.io.InputStream;
import java.io.IOException;

import com.atlassian.user.configuration.*;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.repository.DefaultRepositoryIdentifier;

/**
 * Parses repository definitions and configuration out of an <code>atlassian-user.xml</code> file, with fall-back
 * to the defaults loaded by the <code>XMLDefaultsParser</code>.
 *
 * <p>TODO: Document the file format.
 *
 * @see XMLDefaultsParser
 */
public class XMLConfigurationParser
{
    private final static Logger log = Logger.getLogger(XMLConfigurationParser.class);

    private final XMLDefaultsParser defaultsParser;

    /**
     * List of ({@link RepositoryIdentifier}s in the order of delegation.
     */
    private List<RepositoryIdentifier> repositoryIdentifiers = new ArrayList<RepositoryIdentifier>();

    private Map<RepositoryIdentifier, RepositoryConfiguration> repositoryConfigurations = new HashMap<RepositoryIdentifier, RepositoryConfiguration>();


    public XMLConfigurationParser() throws ConfigurationException
    {
        this(XMLDefaultsParser.DEFAULTS_FILE_NAME);
    }

    public XMLConfigurationParser(String defaultsFileName) throws ConfigurationException
    {
        try
        {
            defaultsParser = new XMLDefaultsParser(defaultsFileName);
        }
        catch (RuntimeException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigurationException("Unable to load atlassian-user configuration parser: " + e.getMessage(), e);
        }
    }

    public void parse(InputStream docIS) throws ConfigurationException
    {
        try
        {
            if (docIS == null)
                throw new ConfigurationException("Null inputstream: cannot locate atlassian-user.xml");

            SAXReader reader = new SAXReader();
            Document doc;
            try
            {
                doc = reader.read(docIS);
            }
            catch (DocumentException e)
            {
                throw new ConfigurationException(e);
            }

            Node delegationNode = doc.selectSingleNode("//" + Configuration.DELEGATION);
            Node repositoriesNode = doc.selectSingleNode("//" + Configuration.REPOSITORIES);

            parseRepositories(repositoriesNode);

            if (delegationNode != null)
                parseDelegation(delegationNode);
        }
        catch (RuntimeException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigurationException("Unable to load atlassian-user configuration: " + e.getMessage(), e);
        }
    }

    protected void parseRepositories(Node repositoriesNode) throws ConfigurationException, DocumentException, IOException
    {
        List repositoryElements = repositoriesNode.selectNodes("*");

        if (repositoryElements.isEmpty())
            throw new ConfigurationException("Nothing to init. There are no repositories specified.");

        for (Object repositoryElement1 : repositoryElements)
        {
            Element repositoryElement = (Element) repositoryElement1;

            // Elements in atlassian-user.xml have the name of their type, for example 'ldap', 'osuser' and so on.
            String repositoryType = repositoryElement.getName();

            Map<String, String> defaultComponentClassNames = defaultsParser.getDefaultClassesConfigForKey(repositoryType);
            Map<String, String> defaultComponents = defaultsParser.getDefaultParameterConfigForKey(repositoryType);

            //now we allow the config. to override the defaults
            RepositoryIdentifier identifier = parseRepositoryIdentifier(repositoryElement);
            if (repositoryIdentifiers.contains(identifier))
            {
                throw new ConfigurationException("Repository keys must be unique. Please check that you have not " +
                        "used the key '" + identifier.getKey() + "' more than once in your atlassian-user.xml file.");
            }

            Map<String, String> componentClassNames = XMLConfigUtil.parseRepositoryElementForClassNames(repositoryElement);
            Map<String, String> components = XMLConfigUtil.parseRepositoryElementForStringData(repositoryElement);

            /**
             * Override the default class names (from atlassian-user-defaults.xml) with any values found in
             * the config. XML file.
             */
            for (Object o : defaultComponentClassNames.keySet())
            {
                String componentClassName = (String) o;
                if (!componentClassNames.containsKey(componentClassName))
                {
                    componentClassNames.put(componentClassName, defaultComponentClassNames.get(componentClassName));
                }
            }

            for (String componentName : defaultComponents.keySet())
            {
                if (!components.containsKey(componentName))
                {
                    components.put(componentName, defaultComponents.get(componentName));
                }
            }

            RepositoryProcessor processor = instantiateProcessor(componentClassNames);

            RepositoryConfiguration configuration =
                    new DefaultRepositoryConfiguration(identifier, processor, components, componentClassNames);

            if (isCachingEnabled(repositoryElement))
                configuration.setCacheConfiguration(parseCacheConfiguration());

            repositoryIdentifiers.add(identifier);
            repositoryConfigurations.put(identifier, configuration);
        }
    }

    private RepositoryIdentifier parseRepositoryIdentifier(Element repositoryElement)
    {
        String key = repositoryElement.attributeValue("key");
        if (key == null) throw new RuntimeException("Cannot specify repository without a key");
        String name = repositoryElement.attributeValue("name", "Unnamed repository");
        return new DefaultRepositoryIdentifier(key, name);
    }

    private CacheConfiguration parseCacheConfiguration() throws DocumentException, IOException
    {
        Map classNames = defaultsParser.getDefaultClassesConfigForKey(Configuration.CACHE);
        return new DefaultCacheConfiguration(classNames);
    }

    private boolean isCachingEnabled(Element repositoryElement)
    {
        String cache = repositoryElement.attributeValue(Configuration.CACHE);
        return cache != null && cache.equalsIgnoreCase("true");
    }

    /**
     * Reorders {@link #repositoryIdentifiers} according to the &lt;delegation&gt; element in atlassian-user.xml.
     */
    private void parseDelegation(Node delegationNode)
    {
        List<RepositoryIdentifier> delegationOrder = new LinkedList<RepositoryIdentifier>();

        for (Object o : delegationNode.selectNodes(Configuration.KEY))
        {
            String delegationKey = ((Element) o).getText();
            for (RepositoryIdentifier identifier : repositoryIdentifiers)
            {
                if (delegationKey.equals(identifier.getKey()))
                {
                    delegationOrder.add(identifier);
                    break;
                }
            }
        }
        repositoryIdentifiers = delegationOrder;
    }

    private RepositoryProcessor instantiateProcessor(Map processorInfo)
    {
        String processorClassName = (String) processorInfo.get(Configuration.PROCESSOR);
        RepositoryProcessor processor = null;

        try
        {
            processor = (RepositoryProcessor) Class.forName(processorClassName).newInstance();
        }
        catch (Exception e)
        {
            log.error("Could not instantiate processor: " + e.getMessage());
        }

        return processor;
    }

    /**
     * @return an unmodifiable list of {@link RepositoryConfiguration}s in order of delegation.
     */
    public List<RepositoryConfiguration> getRepositoryConfigurations()
    {
        List<RepositoryConfiguration> result = new LinkedList<RepositoryConfiguration>();

        for (RepositoryIdentifier identifier : repositoryIdentifiers)
            result.add(repositoryConfigurations.get(identifier));
        
        return Collections.unmodifiableList(result);
    }
}
