package com.atlassian.upm;

import com.google.common.collect.ImmutableList;

import static com.atlassian.upm.api.util.Option.option;
import static java.lang.Boolean.getBoolean;

/**
 * Copied in part from {@link com.atlassian.upm.Sys} because this is used in the compatibility SPI
 * which cannot depend on the new version of UPM which would contain the required method.
 */
public abstract class SysCommon
{
    public static final String UPM_ON_DEMAND = "atlassian.upm.on.demand";
    public static final String UPM_OD_PVA_BLACKLIST = "atlassian.upm.on.demand.pva.blacklist";
    
    /**
     * Plugin key/bundle ID of the Atlassian Connect bundled plugin.  This is used by
     * {@code RemotePluginLicenseServiceServiceFactory} to prevent other plugins from using the
     * remote licensing API, and by {@code PluginInstallationService} to ensure that we won't use
     * any install handler components published by other plugins.
     */
    public static final String ATLASSIAN_CONNECT_PLUGIN_KEY = "com.atlassian.plugins.atlassian-connect-plugin";
    
    /**
     * The presence of this bundle attribute indicates that an installed plugin is an Atlassian
     * Connect add-on.
     */
    public static final String ATLASSIAN_CONNECT_BUNDLE_ATTRIBUTE = "Remote-Plugin";
    
    /**
     * Plugin key/bundle ID of the old (beta) Atlassian Connect/Remote Apps plugin.  We still
     * allow this plugin to access the remote plugin license service for now to support current
     * beta developers, but it will be unsupported at some point in the future.
     */
    public static final String OLD_ATLASSIAN_CONNECT_PLUGIN_KEY = "com.atlassian.labs.remoteapps-plugin";
    
    private static final Iterable<String> DEFAULT_BLACKLIST_VALUE = ImmutableList.of("jira-timesheet-plugin",
                                                                                     "com.balsamiq.jira.plugins.mockups",
                                                                                     "com.balsamiq.confluence.plugins.mockups",
                                                                                     "org.swift.confluence.table");

    public static boolean isOnDemand()
    {
        return getBoolean(UPM_ON_DEMAND);
    }

    /**
     * UPM-3696 OnDemand wants to be able to update the PvA blacklist via sysprop (decoupling it from a UPM version update)
     */
    public static Iterable<String> getOnDemandPaidViaAtlassianBlacklist()
    {
        if (!isOnDemand())
        {
            return ImmutableList.of();
        }

        return getPluginKeysFromSysProp(UPM_OD_PVA_BLACKLIST, DEFAULT_BLACKLIST_VALUE);
    }

    public static Iterable<String> getPluginKeysFromSysProp(String propKey, Iterable<String> defaultValue)
    {
        for (String pluginKeys : option(System.getProperty(propKey)))
        {
            ImmutableList.Builder<String> keys = ImmutableList.builder();
            for (String key : pluginKeys.split(","))
            {
                keys.add(key.trim());
            }
            return keys.build();
        }

        return defaultValue;
    }
}