package com.atlassian.upm.license.storage.lib;

import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.upm.osgi.Version;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.atlassian.upm.license.storage.lib.PluginLicenseStoragePluginInstaller.PLUGIN_LICENSE_STORAGE_PLUGIN_KEY;
import static com.atlassian.upm.license.storage.lib.PluginLicenseStoragePluginInstaller.UPM_PLUGIN_KEY;
import static com.atlassian.upm.license.storage.lib.VersionChecker.PluginState.INSUFFICIENT_VERSION_AND_DISABLED;
import static com.atlassian.upm.license.storage.lib.VersionChecker.PluginState.INSUFFICIENT_VERSION_AND_ENABLED;
import static com.atlassian.upm.license.storage.lib.VersionChecker.PluginState.NOT_INSTALLED;
import static com.atlassian.upm.license.storage.lib.VersionChecker.PluginState.SUFFICIENT_VERSION_AND_DISABLED;
import static com.atlassian.upm.license.storage.lib.VersionChecker.PluginState.SUFFICIENT_VERSION_AND_ENABLED;
import static com.atlassian.upm.osgi.impl.Versions.fromPlugin;
import static com.atlassian.upm.osgi.impl.Versions.fromString;

/**
 * Ideally this would be an injectable-component, but in order to lessen the
 * footprint required for third party developers to use this API, let's make
 * this a static utility class.
 */
class VersionChecker
{
    private static final Logger log = LoggerFactory.getLogger(VersionChecker.class);

    private VersionChecker() {}

    /**
     * Returns true if UPM 2.0.1 or later is installed, false otherwise.
     * Note that UPM 2.0 will return false because of UPM-1715.
     *
     * @param pluginAccessor plugin accessor
     * @return true if UPM 2.0.1 or later is installed, false otherwise.
     */
    static boolean isUpm201OrLaterInstalled(PluginAccessor pluginAccessor)
    {
        return isUpmVersionEqualToOrAfter("2.0.1", pluginAccessor);
    }

    /**
     * Returns true if UPM 2.8 or later is installed, false otherwise.
     *
     * @param pluginAccessor plugin accessor
     * @return true if UPM 2.8 or later is installed, false otherwise.
     */
    static boolean isUpm28OrLaterInstalled(PluginAccessor pluginAccessor)
    {
        return isUpmVersionEqualToOrAfter("2.8", pluginAccessor);
    }

    private static boolean isUpmVersionEqualToOrAfter(String version, PluginAccessor pluginAccessor)
    {
        Plugin plugin = pluginAccessor.getPlugin(UPM_PLUGIN_KEY);
        if (plugin == null)
        {
            //UPM is not installed
            return false;
        }

        try
        {
            //don't cache the Version result because of a possible dependency on missing {@code Option} class
            Version upmVersion = fromPlugin(plugin, false);
            Version requiredVersion = fromString(version, false);
            return requiredVersion.compareTo(upmVersion) <= 0;
        }
        catch (Exception e)
        {
            log.debug("Error while detecting UPM version", e);
            return false;
        }
    }

    /**
     * Returns the plugin state of the Plugin License Storage plugin.
     *
     * @param pluginAccessor plugin accessor
     * @param incomingVersion the incoming Plugin License Storage plugin version
     * @return the plugin state of the Plugin License Storage plugin.
     */
    static PluginState getPluginLicenseStoragePluginState(PluginAccessor pluginAccessor, String incomingVersion)
    {
        final Plugin plugin = pluginAccessor.getPlugin(PLUGIN_LICENSE_STORAGE_PLUGIN_KEY);
        if (plugin == null)
        {
            return NOT_INSTALLED;
        }

        try
        {
            Version installedPluginVersion = fromPlugin(plugin, false);
            Version newPluginVersion = fromString(incomingVersion, false);
            boolean sufficientVersion = newPluginVersion.compareTo(installedPluginVersion) <= 0;
            boolean enabled = pluginAccessor.isPluginEnabled(PLUGIN_LICENSE_STORAGE_PLUGIN_KEY);

            return sufficientVersion ?
                    (enabled ? SUFFICIENT_VERSION_AND_ENABLED : SUFFICIENT_VERSION_AND_DISABLED) :
                    (enabled ? INSUFFICIENT_VERSION_AND_ENABLED : INSUFFICIENT_VERSION_AND_DISABLED);
        }
        catch (Exception e)
        {
            log.warn("Error while detecting Plugin License Storage plugin version", e);
            return NOT_INSTALLED;
        }
    }

    static enum PluginState
    {
        SUFFICIENT_VERSION_AND_ENABLED(true, true),
        SUFFICIENT_VERSION_AND_DISABLED(true, false),
        INSUFFICIENT_VERSION_AND_ENABLED(false, true),
        INSUFFICIENT_VERSION_AND_DISABLED(false, false),
        NOT_INSTALLED(false, true); //it will be enabled upon installation

        private final boolean enabled;
        private final boolean sufficientVersionInstalled;

        private PluginState(boolean sufficientVersionInstalled, boolean enabled)
        {
            this.sufficientVersionInstalled = sufficientVersionInstalled;
            this.enabled = enabled;
        }

        boolean isEnabled()
        {
            return enabled;
        }

        boolean isSufficientVersionInstalled()
        {
            return sufficientVersionInstalled;
        }
    }
}
