package com.atlassian.upm.license.storage.lib;

import java.net.URI;

import com.atlassian.upm.api.license.entity.PluginLicense;
import com.atlassian.upm.api.util.Option;

/**
 * Exposes some (but not all) of UPM 2.0's licensing API for third-party
 * plugins to use in the absence of a licensing-aware UPM (version 2.0.1+).
 * 
 */
public interface ThirdPartyPluginLicenseStorageManager
{
    /**
     * Returns the {@link PluginLicense} for the current plugin, if one exists.
     *
     * @return the {@link PluginLicense} for the current plugin, if one exists.
     * @throws PluginLicenseStoragePluginUnresolvedException if the Plugin License Storage plugin is required but unavailable 
     */
    Option<PluginLicense> getLicense() throws PluginLicenseStoragePluginUnresolvedException;

    /**
     * Returns the raw license string for the current plugin, if one exists.
     *
     * @return the raw license string for the current plugin, if one exists.
     * @throws PluginLicenseStoragePluginUnresolvedException if the Plugin License Storage plugin is required but unavailable 
     */
    Option<String> getRawLicense() throws PluginLicenseStoragePluginUnresolvedException;

    /**
     * Sets the raw license string for the current plugin. Returns the previous value if one existed.
     * This method should only be used in the absence of a licensing-aware UPM.
     *
     * As of version 2.2.2, only administrators will be able to invoke this method.
     *
     * @param rawLicense the new raw license string
     * @return the previous value if one existed.
     * @throws PluginLicenseStoragePluginUnresolvedException if the Plugin License Storage plugin is required but unavailable
     * @throws PluginLicenseStoragePluginPermissionDeniedException if a non-administrator invokes this method (since v2.2.2)
     */
    Option<String> setRawLicense(String rawLicense) throws PluginLicenseStoragePluginUnresolvedException;

    /**
     * Removes the raw license string for the current plugin. Returns the previous value if one existed.
     * This method should only be used in the absence of a licensing-aware UPM.
     *
     * As of version 2.2.2, only administrators will be able to invoke this method.
     *
     * @return the previously set license if one existed.
     * @throws PluginLicenseStoragePluginUnresolvedException if the Plugin License Storage plugin is required but unavailable
     * @throws PluginLicenseStoragePluginPermissionDeniedException if a non-administrator invokes this method (since v2.2.2)
     */
    Option<String> removeRawLicense() throws PluginLicenseStoragePluginUnresolvedException;

    /**
     * Validates the given raw license string. Returns a {@link PluginLicense} if the raw license string is
     * valid, or {@code none()} for an invalid license.
     * This method should only be used in the absence of a licensing-aware UPM.
     *
     * @param rawLicense the raw license string to validate
     * @return a {@link PluginLicense} if the raw license string is valid
     * @throws PluginLicenseStoragePluginUnresolvedException if the Plugin License Storage plugin is required but unavailable 
     */
    Option<PluginLicense> validateLicense(String rawLicense) throws PluginLicenseStoragePluginUnresolvedException;

    /**
     * Returns true if UPM is installed and licensing aware (version 2.0.1+), false if not.
     *
     * @return true if UPM is installed and licensing aware, false if not.
     * @throws PluginLicenseStoragePluginUnresolvedException if the Plugin License Storage plugin is required but unavailable 
     */
    boolean isUpmLicensingAware();

    /**
     * Returns the current plugin's plugin key. This is provided as a convenience since plugins may wish to
     * disable themselves (using the plugin framework's PluginController) if there is no valid license;
     * using this method avoids having to hard-code the plugin key.
     *
     * @return the current plugin's plugin key.
     * @throws PluginLicenseStoragePluginUnresolvedException if the Plugin License Storage plugin is required but unavailable 
     */
    String getPluginKey() throws PluginLicenseStoragePluginUnresolvedException;

    /**
     * Returns the URI to manage the current plugin within UPM. If a licensing-aware UPM is currently installed,
     * administrators may manage this plugin's license at this URI.
     *
     * This URI will not resolve when UPM is not installed.
     *
     * @return the URI to manage the current plugin within UPM.
     * @throws PluginLicenseStoragePluginUnresolvedException if the Plugin License Storage plugin is required but unavailable 
     */
    URI getPluginManagementUri() throws PluginLicenseStoragePluginUnresolvedException;

    /**
     * Returns true if the current application instance is an OnDemand environment, false otherwise.
     *
     * @return true if the current application instance is an OnDemand environment, false otherwise.
     * @since 2.3
     */
    boolean isOnDemand();
}