package com.atlassian.upm.license.storage.lib;

import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.PluginArtifact;
import com.atlassian.plugin.PluginController;
import com.atlassian.sal.api.transaction.TransactionCallback;
import com.atlassian.sal.api.transaction.TransactionTemplate;
import com.atlassian.upm.UpmVersion;
import com.atlassian.upm.license.storage.lib.VersionChecker.PluginState;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;

import static com.atlassian.upm.license.storage.lib.PluginArtifactFactory.fromFile;
import static com.atlassian.upm.license.storage.lib.VersionChecker.getPluginLicenseStoragePluginState;
import static com.atlassian.upm.license.storage.lib.VersionChecker.isUpm201OrLaterInstalled;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Class to install and/or enable the Plugin License Storage plugin if deemed necessary.
 * Simply include this class as a Spring component to execute it.
 */
public class PluginLicenseStoragePluginInstaller implements InitializingBean
{
    static final String UPM_PLUGIN_KEY = "com.atlassian.upm.atlassian-universal-plugin-manager-plugin";
    static final String PLUGIN_LICENSE_STORAGE_PLUGIN_KEY = "com.atlassian.upm.plugin-license-storage-plugin";

    private static final Logger log = LoggerFactory.getLogger(PluginLicenseStoragePluginInstaller.class);

    private final PluginAccessor pluginAccessor;
    private final PluginController pluginController;
    private final TransactionTemplate txTemplate;

    public PluginLicenseStoragePluginInstaller(PluginAccessor pluginAccessor, PluginController pluginController, TransactionTemplate txTemplate)
    {
        this.pluginAccessor = checkNotNull(pluginAccessor, "pluginAccessor");
        this.pluginController = checkNotNull(pluginController, "pluginController");
        this.txTemplate = checkNotNull(txTemplate, "txTemplate");
    }

    /**
     * Install the storage plugin from the given {@link PluginArtifact}.
     *
     * @param pluginArtifact the plugin artifact
     * @return true if installation was successful, false if not
     */
    private boolean installStoragePlugin(final PluginArtifact pluginArtifact)
    {
        if (pluginArtifact == null)
        {
            log.error("Could not find the artifact needed to install the Plugin License Storage plugin. Did you use "
                      + "maven-dependency-plugin to include it within your plugin?");
            return false;
        }

        //install the plugin
        return txTemplate.execute(new TransactionCallback<Boolean>()
        {
            @SuppressWarnings({"deprecation"})
            public Boolean doInTransaction()
            {
                //use deprecated method for compatibility with older product versions
                String pluginKey = pluginController.installPlugin(pluginArtifact);

                if (pluginKey == null)
                {
                    log.error("Plugin failed to install: " + pluginArtifact.getName());
                    return false;
                }

                return true;
            }
        });
    }

    /**
     * Runs upon plugin startup/enablement. This will install the Plugin License Storage plugin if
     * a licensing-aware UPM is not installed or a sufficiently new version of the Plugin License Storage plugin.
     */
    @Override
    @SuppressWarnings({"deprecation"})
    public void afterPropertiesSet() throws Exception
    {
        try
        {
            //if no licensing-aware UPM is present
            if (!isUpm201OrLaterInstalled(pluginAccessor))
            {
                PluginState pluginState = getPluginLicenseStoragePluginState(pluginAccessor, UpmVersion.VERSION);

                //if the Plugin License Storage plugin needs to be updated
                if (!pluginState.isSufficientVersionInstalled())
                {
                    if (!installStoragePlugin(fromFile("plugin-license-storage-plugin.jar")))
                    {
                        log.error("Error while installing the Plugin License Storage plugin. Is the artifact on your plugin's classpath?");
                    }
                }

                //if the Plugin License Storage plugin needs to be enabled
                if (!pluginState.isEnabled())
                {
                    try
                    {
                        //use deprecated method for compatibility with older product versions
                        pluginController.enablePlugin(PLUGIN_LICENSE_STORAGE_PLUGIN_KEY);
                    }
                    catch (Exception e)
                    {
                        log.warn("Failed to enabled license storage plugin", e);
                    }
                }
            }
        }
        catch (Throwable th)
        {
            // In case any situation such as a NoClassDefFoundError or ServiceProxyDestroyedException occurs,
            // we do not want to prevent the plugin from starting up.
            //
            // Prior to version 2.3, these errors would occur when one Marketplace plugin would update the Storage plugin
            // while another plugin was bound to an older version. These errors should not be occurring anymore, but it is
            // still a good idea to have this safeguard in place just in case.
            log.error("Failed to install or enable the Plugin License Storage plugin version " + UpmVersion.VERSION, th);
        }
    }
}