package com.atlassian.upm.license.storage.lib;

import java.net.URI;

import com.atlassian.plugin.PluginAccessor;
import com.atlassian.sal.api.ApplicationProperties;
import com.atlassian.sal.api.pluginsettings.PluginSettings;
import com.atlassian.sal.api.pluginsettings.PluginSettingsFactory;
import com.atlassian.sal.api.transaction.TransactionCallback;
import com.atlassian.sal.api.transaction.TransactionTemplate;
import com.atlassian.upm.Strings;
import com.atlassian.upm.api.license.entity.Contact;
import com.atlassian.upm.api.license.entity.PluginLicense;
import com.atlassian.upm.api.util.Option;
import com.atlassian.upm.api.util.Pair;
import com.atlassian.upm.license.PluginLicenses;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableList;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import static com.atlassian.upm.api.util.Option.none;
import static com.atlassian.upm.api.util.Option.some;
import static com.atlassian.upm.api.util.Options.catOptions;
import static com.atlassian.upm.api.util.Pair.pair;
import static com.atlassian.upm.license.storage.lib.UrlEncoder.encode;
import static com.atlassian.upm.license.storage.lib.VersionChecker.isUpm28OrLaterInstalled;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Iterables.transform;
import static org.apache.commons.lang.StringUtils.isBlank;

public class AtlassianMarketplaceUriFactoryImpl implements AtlassianMarketplaceUriFactory, ApplicationContextAware
{
    private static final String CALLBACK_URL_KEY = "plugin.license.storage.callback.url";

    private final ThirdPartyPluginLicenseStorageManager licenseManager;
    private final PluginSettingsFactory pluginSettingsFactory;
    private final ApplicationProperties applicationProperties;
    private final TransactionTemplate txTemplate;
    private final PluginAccessor pluginAccessor;
    private ManagerAccessor managerAccessor;

    public AtlassianMarketplaceUriFactoryImpl(ThirdPartyPluginLicenseStorageManager licenseManager,
                                              PluginSettingsFactory pluginSettingsFactory,
                                              ApplicationProperties applicationProperties,
                                              PluginAccessor pluginAccessor,
                                              TransactionTemplate txTemplate)
    {
        this(licenseManager, pluginSettingsFactory, applicationProperties, pluginAccessor, txTemplate, null);
    }

    /**
     * Package-scoped constructor to allow mocking our {@link ManagerAccessor}.
     */
    AtlassianMarketplaceUriFactoryImpl(ThirdPartyPluginLicenseStorageManager licenseManager,
                                              PluginSettingsFactory pluginSettingsFactory,
                                              ApplicationProperties applicationProperties,
                                              PluginAccessor pluginAccessor,
                                              TransactionTemplate txTemplate,
                                              ManagerAccessor managerAccessor)
    {
        this.licenseManager = checkNotNull(licenseManager, "licenseManager");
        this.pluginSettingsFactory = checkNotNull(pluginSettingsFactory, "pluginSettingsFactory");
        this.applicationProperties = checkNotNull(applicationProperties, "applicationProperties");
        this.txTemplate = checkNotNull(txTemplate, "txTemplate");
        this.pluginAccessor = checkNotNull(pluginAccessor, "pluginAccessor");
        this.managerAccessor = managerAccessor;
    }

    @Override
    public URI getBuyPluginUri(URI callback) throws PluginLicenseStoragePluginUnresolvedException
    {
        return URI.create(getMyAtlassianUri("new").toASCIIString() +
                          buildRequestParamString(getParentSen(), getUsers(), getReferrer(),
                                                  getCallback(callback), getLicenseFieldName()));
    }

    @Override
    public URI getTryPluginUri(URI callback) throws PluginLicenseStoragePluginUnresolvedException
    {
        return URI.create(getMyAtlassianUri("try").toASCIIString() +
                          buildRequestParamString(getParentSen(), getOrganizationName(), getReferrer(),
                                                  getCallback(callback), getLicenseFieldName()));
    }

    @Override
    public URI getRenewPluginUri(URI callback) throws PluginLicenseStoragePluginUnresolvedException
    {
        return URI.create(getMyAtlassianUri("renew").toASCIIString() +
                          buildRequestParamString(getParentSen(), getAddonSen(), getOwner(), getUsers(),
                                                  getReferrer(), getCallback(callback), getLicenseFieldName()));
    }

    @Override
    public URI getUpgradePluginUri(URI callback) throws PluginLicenseStoragePluginUnresolvedException
    {
        return URI.create(getMyAtlassianUri("upgrade").toASCIIString() +
                          buildRequestParamString(getParentSen(), getAddonSen(), getOwner(), getUsers(),
                                                  getReferrer(), getCallback(callback), getLicenseFieldName()));
    }

    @Override
    public URI getReviewPluginUri(int stars, Option<String> review)
    {
        validateReviewOrWatchAccess();
        return getUpmRestUri("/available/%s/review", some(pair("stars", String.valueOf(stars))), getReview(review));
    }

    @Override
    public URI getWatchPluginUri()
    {
        validateReviewOrWatchAccess();
        return getUpmRestUri("/available/%s/watch");
    }

    @Override
    public boolean isPluginBuyable() throws PluginLicenseStoragePluginUnresolvedException
    {
        return !licenseManager.isUpmLicensingAware() && PluginLicenses.isPluginBuyable(licenseManager.getLicense());
    }

    @Override
    public boolean isPluginTryable() throws PluginLicenseStoragePluginUnresolvedException
    {
        return !licenseManager.isUpmLicensingAware() && PluginLicenses.isPluginTryable(licenseManager.getLicense());
    }

    @Override
    public boolean isPluginRenewable() throws PluginLicenseStoragePluginUnresolvedException
    {
        return !licenseManager.isUpmLicensingAware() && PluginLicenses.isPluginRenewable(licenseManager.getLicense());
    }

    @Override
    public boolean isPluginUpgradable() throws PluginLicenseStoragePluginUnresolvedException
    {
        return !licenseManager.isUpmLicensingAware() && PluginLicenses.isPluginUpgradable(licenseManager.getLicense());
    }

    @Override
    public boolean canReviewPlugin()
    {
        return canReviewOrWatch();
    }

    @Override
    public boolean canWatchPlugin()
    {
        return canReviewOrWatch();
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException
    {
        this.managerAccessor = new ManagerAccessor(applicationContext);
    }

    private String getPluginKey() throws PluginLicenseStoragePluginUnresolvedException
    {
        return licenseManager.getPluginKey();
    }

    private URI getMyAtlassianUri(String type) throws PluginLicenseStoragePluginUnresolvedException
    {
        // We need to access the host license to properly/completely build these URIs.
        // UPM's licensing API does not expose the host license - only the Plugin License Storage plugin does.
        // However, if UPM is licensing-aware, then there shouldn't be any visible legacy licensing UI anyways,
        // making these accessors unnecessary.
        if (licenseManager.isUpmLicensingAware())
        {
            throw new UnsupportedOperationException("Cannot build My Atlassian URI when UPM is licensing-aware: "
                                                    + "required data is unavailable. Please use UPM's licensing"
                                                    + "UI instead.");
        }

        //We don't have access to Sys here - we have to copy and paste this.
        String macBaseUrlSysProp = System.getProperty("mac.baseurl");
        String macBaseUrl = macBaseUrlSysProp != null ? macBaseUrlSysProp : "https://my.atlassian.com";

        return URI.create(macBaseUrl + "/addon/" + type + "/" + encode(getPluginKey()));
    }

    /**
     * Converts an array of possible request parameters into a usable request parameter segment of a URI.
     */
    private String buildRequestParamString(Option<Pair<String, String>>... params)
    {
        Iterable<Pair<String, String>> nonNoneParams = catOptions(ImmutableList.of(params));
        Iterable<String> paramStrings = transform(nonNoneParams, new Function<Pair<String, String>, String>()
        {
            @Override
            public String apply(Pair<String, String> param)
            {
                return param.first() + "=" + param.second();
            }
        });

        String joined = Joiner.on("&").join(paramStrings);
        if (StringUtils.isNotBlank(joined))
        {
            joined = "?" + joined;
        }

        return joined;
    }

    private URI getUpmRestUri(String path, Option<Pair<String, String>>... params)
    {
        String pluginKey = encode(managerAccessor.getLicenseManager().getPluginKey());
        return URI.create(applicationProperties.getBaseUrl() + "/rest/plugins/1.0" + String.format(path, pluginKey) + buildRequestParamString(params));
    }

    private Option<Pair<String, String>> getParentSen() throws PluginLicenseStoragePluginUnresolvedException
    {
        for (String hostSen : managerAccessor.getStorageManager().getHostLicenseSupportEntitlementNumber())
        {
            return some(pair("parent_sen", hostSen));
        }

        return none();
    }

    private Option<Pair<String, String>> getReview(Option<String> review)
    {
        for (String r : review)
        {
            if (!isBlank(r))
            {
                return some(pair("review", encode(r)));
            }
        }
        return none();
    }

    private Option<Pair<String, String>> getAddonSen() throws PluginLicenseStoragePluginUnresolvedException
    {
        for (PluginLicense license : licenseManager.getLicense())
        {
            for (String licenseSen : license.getSupportEntitlementNumber())
            {
                return some(pair("addon_sen", licenseSen));
            }
        }

        return none();
    }

    private Option<Pair<String, String>> getOrganizationName() throws PluginLicenseStoragePluginUnresolvedException
    {
        for (String orgName : managerAccessor.getStorageManager().getHostLicenseOrganizationName())
        {
            return some(pair("organisation_name", encode(orgName)));
        }

        return none();
    }

    private Option<Pair<String, String>> getOwner() throws PluginLicenseStoragePluginUnresolvedException
    {
        for (PluginLicense license : licenseManager.getLicense())
        {
            Iterable<String> emails = transform(license.getContacts(), new Function<Contact, String>()
            {
                @Override
                public String apply(Contact contact)
                {
                    return contact.getEmail();
                }
            });

            for (String email : Strings.getFirstNonEmpty(emails))
            {
                return some(pair("owner", encode(email)));
            }
        }

        return none();
    }

    private Option<Pair<String, String>> getReferrer()
    {
        return some(pair("referrer", "thirdpartyplugin"));
    }

    private Option<Pair<String, String>> getLicenseFieldName()
    {
        return some(pair("licensefieldname", "license"));
    }

    private Option<Pair<String, String>> getCallback(final URI callback) throws PluginLicenseStoragePluginUnresolvedException
    {
        if (!callback.isAbsolute())
        {
            throw new IllegalArgumentException("Callback URI must be absolute: " + callback);
        }
        
        txTemplate.execute(new TransactionCallback<Object>()
        {
            @Override
            public Object doInTransaction()
            {
                return getPluginSettings().put(CALLBACK_URL_KEY, callback.toASCIIString());
            }
        });
        
        String licenseReceipt = applicationProperties.getBaseUrl() + "/plugins/servlet/pluginlicensereceipt/" + encode(getPluginKey());
        return some(pair("callback", licenseReceipt));
    }

    private Option<Pair<String, String>> getUsers() throws PluginLicenseStoragePluginUnresolvedException
    {
        for (Integer maxEdition : managerAccessor.getStorageManager().getHostEdition())
        {
            return some(pair("users", maxEdition.toString()));
        }

        //the MAC API expects "-1" to represent an unlimited user count
        return some(pair("users", "-1"));
    }

    private PluginSettings getPluginSettings()
    {
        return pluginSettingsFactory.createGlobalSettings();
    }

    private boolean canReviewOrWatch()
    {
        return isUpm28OrLaterInstalled(pluginAccessor) &&
               managerAccessor.getUpmHostLicenseInformation().hostSen().isDefined();
    }

    private void validateReviewOrWatchAccess()
    {
        if (!isUpm28OrLaterInstalled(pluginAccessor))
        {
            throw new IllegalStateException("UPM 2.8+ is required");
        }
        else if (!managerAccessor.getUpmHostLicenseInformation().hostSen().isDefined())
        {
            throw new IllegalStateException("Application should have a Support Entitlement Number (SEN)");
        }
    }
}
