package com.atlassian.upm.license.storage.lib;

import java.net.URI;

import com.atlassian.upm.api.util.Option;

/**
 * Factory to produce URIs to get plugin licenses, when appropriate, from My Atlassian (MAC).
 * All HTTP requests to the returned URIs should be of method type POST.
 *
 * For example, your plugin can check {@link #isPluginBuyable()} and if so, display
 * a form button which POSTs to the return value of {@link #getBuyPluginUri(java.net.URI)}.
 */
public interface AtlassianMarketplaceUriFactory
{
    /**
     * Returns the URI to POST to for buying the current plugin.
     *
     * @param callback the URI to return to following the POST to My Atlassian. Must be absolute.
     * @return the URI to POST to for buying the current plugin.
     * @throws PluginLicenseStoragePluginUnresolvedException thrown if the Plugin License Storage plugin is needed but unavailable
     */
    URI getBuyPluginUri(URI callback) throws PluginLicenseStoragePluginUnresolvedException;

    /**
     * Returns the URI to POST to for trying the current plugin.
     *
     * @param callback the URI to return to following the POST to My Atlassian. Must be absolute.
     * @return the URI to POST to for trying the current plugin.
     * @throws PluginLicenseStoragePluginUnresolvedException thrown if the Plugin License Storage plugin is needed but unavailable
     */
    URI getTryPluginUri(URI callback) throws PluginLicenseStoragePluginUnresolvedException;

    /**
     * Returns the URI to POST to for renewing the current plugin.
     *
     * @param callback the URI to return to following the POST to My Atlassian. Must be absolute.
     * @return the URI to POST to for renewing the current plugin.
     * @throws PluginLicenseStoragePluginUnresolvedException thrown if the Plugin License Storage plugin is needed but unavailable
     */
    URI getRenewPluginUri(URI callback) throws PluginLicenseStoragePluginUnresolvedException;

    /**
     * Returns the URI to POST to for upgrading the current plugin.
     *
     * @param callback the URI to return to following the POST to My Atlassian. Must be absolute.
     * @return the URI to POST to for upgrading the current plugin.
     * @throws PluginLicenseStoragePluginUnresolvedException thrown if the Plugin License Storage plugin is needed but unavailable
     */
    URI getUpgradePluginUri(URI callback) throws PluginLicenseStoragePluginUnresolvedException;

    /**
     * Returns the URI to POST to for reviewing the current plugin. Should only be used if UPM 2.8+ is installed.
     *
     * @param stars The review rating, should be from 1 to 4
     * @param review Optional. The user's review text
     * @return the URI to POST to for reviewing the current plugin.
     * @throws IllegalStateException thrown if the method is accessed and {@link #canReviewPlugin()} is false
     * @since 2.8
     */
    URI getReviewPluginUri(int stars, Option<String> review);

    /**
     * Returns the URI to POST to for watching the current plugin. Should only be used if UPM 2.8+ is installed.
     *
     * @return the URI to POST to for watching the current plugin.
     * @throws IllegalStateException thrown if the method is accessed and {@link #canWatchPlugin()} is false
     * @since 2.8
     */
    URI getWatchPluginUri();

    /**
     * Returns {@code true} if the current plugin is eligible to be bought, {@code false} otherwise
     *
     * @return {@code true} if the current plugin is eligible to be bought, {@code false} otherwise
     * @throws PluginLicenseStoragePluginUnresolvedException thrown if the Plugin License Storage plugin is needed but unavailable
     */
    boolean isPluginBuyable() throws PluginLicenseStoragePluginUnresolvedException;

    /**
     * Returns {@code true} if the current plugin is eligible to be tried (evaluated), {@code false} otherwise
     *
     * @return {@code true} if the current plugin is eligible to be tried (evaluated), {@code false} otherwise
     * @throws PluginLicenseStoragePluginUnresolvedException thrown if the Plugin License Storage plugin is needed but unavailable
     */
    boolean isPluginTryable() throws PluginLicenseStoragePluginUnresolvedException;

    /**
     * Returns {@code true} if the current plugin is eligible to be renewed, {@code false} otherwise
     *
     * @return {@code true} if the current plugin is eligible to be renewed, {@code false} otherwise
     * @throws PluginLicenseStoragePluginUnresolvedException thrown if the Plugin License Storage plugin is needed but unavailable
     */
    boolean isPluginRenewable() throws PluginLicenseStoragePluginUnresolvedException;

    /**
     * Returns {@code true} if the current plugin is eligible to be upgraded, {@code false} otherwise
     *
     * @return {@code true} if the current plugin is eligible to be upgraded, {@code false} otherwise
     * @throws PluginLicenseStoragePluginUnresolvedException thrown if the Plugin License Storage plugin is needed but unavailable
     */
    boolean isPluginUpgradable() throws PluginLicenseStoragePluginUnresolvedException;

    /**
     * Returns {@code true} if the current plugin can be reviewed, {@code false} otherwise.
     * The plugin can be reviewed if the application is running UPM v2.8+, and has a valid
     * license and Support Entitlement Number (SEN)
     *
     * @return {@code true} if the current plugin can be reviewed, {@code false} otherwise
     * @since 2.8
     */
    boolean canReviewPlugin();

    /**
     * Returns {@code true} if the current plugin can be watched, {@code false} otherwise
     * The plugin can be watched if the application is running UPM v2.8+, and has a valid
     * license and Support Entitlement Number (SEN)
     *
     * @return {@code true} if the current plugin can be watched, {@code false} otherwise
     * @since 2.8
     */
    boolean canWatchPlugin();
}
