package com.atlassian.marketplace.client.model;

import java.util.Collection;
import java.util.Date;

import com.atlassian.upm.api.util.Option;

import com.google.common.collect.ImmutableList;

import org.codehaus.jackson.annotate.JsonCreator;
import org.codehaus.jackson.annotate.JsonIgnore;
import org.codehaus.jackson.annotate.JsonProperty;

import static com.atlassian.marketplace.client.util.ModelUtil.requireList;
import static com.atlassian.marketplace.client.util.ModelUtil.requireProperty;
import static com.atlassian.upm.api.util.Option.option;

/**
 * Pricing information for a Paid-via-Atlassian plugin.
 */
public final class Pricing
{
    @JsonProperty private final Links links;
    @JsonProperty private final Collection<PricingItem> items;
    @JsonProperty private final boolean expertDiscountOptOut;
    @JsonProperty private final String parent;
    @JsonProperty private final Date lastModified;
    @JsonProperty private final Collection<PricingItem> legacyItems;
    @JsonProperty private final String nonLegacyStartVersion;
    @JsonProperty private final String legacyMaxVersion;
    
    @JsonCreator
    Pricing(@JsonProperty("links") Links links,
            @JsonProperty("items") Collection<PricingItem> items,
            @JsonProperty("expertDiscountOptOut") Boolean expertDiscountOptOut,
            @JsonProperty("parent") String parent,
            @JsonProperty("lastModified") Date lastModified,
            @JsonProperty("legacyItems") Collection<PricingItem> legacyItems,
            @JsonProperty("nonLegacyStartVersion") String nonLegacyStartVersion,
            @JsonProperty("legacyMaxVersion") String legacyMaxVersion)
    {
        this.links = requireProperty(links, "links");
        this.items = requireList(items, "items");
        this.expertDiscountOptOut = requireProperty(expertDiscountOptOut, "expertDiscountOptOut");
        this.parent = parent;  // optional
        this.lastModified = lastModified;  // optional
        this.legacyItems = (legacyItems == null) ? ImmutableList.<PricingItem>of() : ImmutableList.copyOf(legacyItems);
        this.nonLegacyStartVersion = nonLegacyStartVersion;  // optional
        this.legacyMaxVersion = legacyMaxVersion;  // optional
    }

    public Links getLinks()
    {
        return links;
    }

    /**
     * A list of available pricing tiers.
     */
    @JsonIgnore
    public Iterable<PricingItem> getItems()
    {
        return ImmutableList.copyOf(items);
    }

    /**
     * True if the vendor does <i>not</i> provide a discount for Atlassian Experts.
     */
    public boolean isExpertDiscountOptOut()
    {
        return expertDiscountOptOut;
    }

    /**
     * Name of the base pricing that provides the defaults for these prices, if any (e.g. "jira").
     */
    @JsonIgnore
    public Option<String> getParent()
    {
        return option(parent);
    }

    /**
     * The date on which these prices were last updated.
     */
    @JsonIgnore
    public Option<Date> getLastModified()
    {
        return option(lastModified);
    }

    /**
     * Pricing tiers applicable to an old version of the application.
     */
    @JsonIgnore
    public Iterable<PricingItem> getLegacyItems()
    {
        return ImmutableList.copyOf(legacyItems);
    }

    /**
     * The version string of the lowest application version that uses the {@link #getLegacyItems legacy pricing}.
     */
    @JsonIgnore
    public Option<String> getNonLegacyStartVersion()
    {
        return option(nonLegacyStartVersion);
    }

    /**
     * The version string of the highest application version that uses the {@link #getLegacyItems legacy pricing}.
     */
    @JsonIgnore
    public Option<String> getLegacyMaxVersion()
    {
        return option(legacyMaxVersion);
    }
}
