package com.atlassian.marketplace.client.model;

import java.net.URI;
import java.util.Collection;
import java.util.Date;

import com.atlassian.upm.api.util.Option;

import com.google.common.collect.ImmutableList;

import org.codehaus.jackson.annotate.JsonCreator;
import org.codehaus.jackson.annotate.JsonIgnore;
import org.codehaus.jackson.annotate.JsonProperty;

import static com.atlassian.marketplace.client.util.ModelUtil.requireLink;
import static com.atlassian.marketplace.client.util.ModelUtil.requireList;
import static com.atlassian.marketplace.client.util.ModelUtil.requireProperty;
import static com.atlassian.upm.api.util.Option.option;

/**
 * Information about a plugin listed on the Atlassian Marketplace.
 *
 * Note that the full set of properties of this object will only be filled in if you are
 * specifically querying a single plugin.  Queries that return a list of plugins will only
 * fill in a subset of properties.
 */
public final class Plugin
{
    @JsonProperty private final Links links;
    @JsonProperty private final Date creationDate;
    @JsonProperty private final String name;
    @JsonProperty private final String pluginKey;
    @JsonProperty private final String description;
    @JsonProperty private final String summary;
    @JsonProperty private final boolean deployable;
    @JsonProperty private final long downloadCount;
    @JsonProperty private final Modification lastModified;
    @JsonProperty private final Vendor vendor;
    @JsonProperty private final PluginMedia media;
    @JsonProperty private final ReviewSummary reviewSummary;
    @JsonProperty private final Reviews reviews;
    @JsonProperty private final Collection<PluginCategory> categories;
    @JsonProperty private final PluginVersion version;
    @JsonProperty private final PluginVersions versions;
    @JsonProperty private final boolean isOldVersion;
    @JsonProperty private final Collection<ApplicationSummary> compatibleApplications;
    @JsonProperty private final Pricing pricing;
    
    private final URI webUri;

    // fields omitted:  googleAnalyticsId
    
    @JsonCreator
    Plugin(@JsonProperty("links") Links links,
           @JsonProperty("creationDate") Date creationDate,
           @JsonProperty("name") String name,
           @JsonProperty("pluginKey") String pluginKey,
           @JsonProperty("description") String description,
           @JsonProperty("summary") String summary,
           @JsonProperty("deployable") Boolean deployable,
           @JsonProperty("downloadCount") Long downloadCount,
           @JsonProperty("lastModified") Modification lastModified,
           @JsonProperty("vendor") Vendor vendor,
           @JsonProperty("media") PluginMedia media,
           @JsonProperty("reviewSummary") ReviewSummary reviewSummary,
           @JsonProperty("reviews") Reviews reviews,
           @JsonProperty("categories") Collection<PluginCategory> categories,
           @JsonProperty("version") PluginVersion version,
           @JsonProperty("versions") PluginVersions versions,
           @JsonProperty("isOldVersion") Boolean isOldVersion,
           @JsonProperty("compatibleApplications") Collection<ApplicationSummary> compatibleApplications,
           @JsonProperty("pricing") Pricing pricing)
    {
        this.links = requireProperty(links, "links");
        this.creationDate = requireProperty(creationDate, "creationDate");
        this.name = requireProperty(name, "name");
        this.pluginKey = requireProperty(pluginKey, "pluginKey");
        this.description = description;  // optional
        this.summary = summary;  // optional
        this.deployable = requireProperty(deployable, "deployable");
        this.downloadCount = requireProperty(downloadCount, "downloadCount");
        this.lastModified = lastModified;  // optional
        this.vendor = vendor;  // optional
        this.media = requireProperty(media, "media");
        this.reviewSummary = requireProperty(reviewSummary, "reviewSummary");
        this.reviews = requireProperty(reviews, "reviews");
        this.categories = requireList(categories, "categories");
        this.version = requireProperty(version, "version");
        this.versions = requireProperty(versions, "versions");
        this.isOldVersion = requireProperty(isOldVersion, "isOldVersion");
        this.compatibleApplications = requireList(compatibleApplications, "compatibleApplications");
        this.pricing = pricing;  // optional
        
        this.webUri = requireLink(links, "alternate");
    }

    public Links getLinks()
    {
        return links;
    }

    /**
     * The date on which the first version of the plugin was published.
     */
    public Date getCreationDate()
    {
        return creationDate;
    }

    /**
     * The plugin name.
     */
    public String getName()
    {
        return name;
    }

    /**
     * The unique plugin key.
     */
    public String getPluginKey()
    {
        return pluginKey;
    }

    /**
     * The full text description of the plugin.
     */
    @JsonIgnore
    public Option<String> getDescription()
    {
        return option(description);
    }

    /**
     * A shorter description of the plugin.
     */
    @JsonIgnore
    public Option<String> getSummary()
    {
        return option(summary);
    }

    /**
     * True if the plugin can be installed through the Plugin Manager.
     */
    public boolean isDeployable()
    {
        return deployable;
    }

    /**
     * The number of times this plugin has been downloaded.
     */
    public long getDownloadCount()
    {
        return downloadCount;
    }

    /**
     * Information about the last change to this plugin, if any.
     */
    @JsonIgnore
    public Option<Modification> getLastModified()
    {
        return option(lastModified);
    }

    /**
     * The vendor of the plugin.
     */
    @JsonIgnore
    public Option<Vendor> getVendor()
    {
        return option(vendor);
    }

    /**
     * Images associated with the plugin.
     */
    public PluginMedia getMedia()
    {
        return media;
    }

    /**
     * A summary of reviews of the plugin.
     */
    public ReviewSummary getReviewSummary()
    {
        return reviewSummary;
    }

    /**
     * The full list of reviews of the plugin.
     */
    public Reviews getReviews()
    {
        return reviews;
    }

    /**
     * Categories to which this plugin belongs.
     */
    @JsonIgnore
    public Iterable<PluginCategory> getCategories()
    {
        return ImmutableList.copyOf(categories);
    }

    /**
     * The specific version being queried, if any.
     */
    public PluginVersion getVersion()
    {
        return version;
    }

    /**
     * The total number of published versions of this plugin.  This will be the total count even if
     * {@link #getVersions()} is returning a subset.
     */
    @JsonIgnore
    public int getVersionCount()
    {
        return versions.getCount();
    }
    
    /**
     * The full list of published versions of this plugin - or a subset of that list, if you restricted
     * the query with {@link com.atlassian.marketplace.client.api.PluginDetailQuery.Builder#limitVersions(Option)}
     * or with application version compatibility parameters.
     */
    @JsonIgnore
    public Iterable<PluginVersion> getVersions()
    {
        return versions.getVersions();
    }

    /**
     * True if the version in {@link #getVersion()} is <i>not</i> the latest published version.
     */
    public boolean isOldVersion()
    {
        return isOldVersion;
    }

    /**
     * The Atlassian application(s) that this plugin is compatible with.
     */
    @JsonIgnore
    public Iterable<ApplicationSummary> getCompatibleApplications()
    {
        return ImmutableList.copyOf(compatibleApplications);
    }

    /**
     * Pricing information, if this is a Paid-via-Atlassian plugin.
     */
    @JsonIgnore
    public Option<Pricing> getPricing()
    {
        return option(pricing);
    }
    
    /**
     * Shortcut for getting the "alternate" link, if any, from the links map.  This is the
     * URI of the plugin details on the Marketplace website.
     */
    @JsonIgnore
    public URI getWebUri()
    {
        return webUri;
    }

    /**
     * Container for list of versions, used internally.
     */
    public static final class PluginVersions
    {
        @JsonProperty private final Collection<PluginVersion> versions;
        @JsonProperty private final int count;
        
        @JsonCreator
        PluginVersions(@JsonProperty("versions") Collection<PluginVersion> versions,
                       @JsonProperty("count") Integer count)
        {
            this.versions = requireList(versions, "versions");
            this.count = requireProperty(count, "count");
        }

        /**
         * The list of plugin versions.
         */
        @JsonIgnore
        public Iterable<PluginVersion> getVersions()
        {
            return ImmutableList.copyOf(versions);
        }
        
        /**
         * The number of versions.
         */
        public int getCount()
        {
            return count;
        }
    }
}
